/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.osis.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an existing OpenSearch Ingestion pipeline.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Pipeline implements SdkPojo, Serializable, ToCopyableBuilder<Pipeline.Builder, Pipeline> {
    private static final SdkField<String> PIPELINE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PipelineName").getter(getter(Pipeline::pipelineName)).setter(setter(Builder::pipelineName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PipelineName").build()).build();

    private static final SdkField<String> PIPELINE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PipelineArn").getter(getter(Pipeline::pipelineArn)).setter(setter(Builder::pipelineArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PipelineArn").build()).build();

    private static final SdkField<Integer> MIN_UNITS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinUnits").getter(getter(Pipeline::minUnits)).setter(setter(Builder::minUnits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinUnits").build()).build();

    private static final SdkField<Integer> MAX_UNITS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxUnits").getter(getter(Pipeline::maxUnits)).setter(setter(Builder::maxUnits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxUnits").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Pipeline::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<PipelineStatusReason> STATUS_REASON_FIELD = SdkField
            .<PipelineStatusReason> builder(MarshallingType.SDK_POJO).memberName("StatusReason")
            .getter(getter(Pipeline::statusReason)).setter(setter(Builder::statusReason))
            .constructor(PipelineStatusReason::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final SdkField<String> PIPELINE_CONFIGURATION_BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PipelineConfigurationBody").getter(getter(Pipeline::pipelineConfigurationBody))
            .setter(setter(Builder::pipelineConfigurationBody))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PipelineConfigurationBody").build())
            .build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(Pipeline::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedAt").getter(getter(Pipeline::lastUpdatedAt)).setter(setter(Builder::lastUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedAt").build()).build();

    private static final SdkField<List<String>> INGEST_ENDPOINT_URLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IngestEndpointUrls")
            .getter(getter(Pipeline::ingestEndpointUrls))
            .setter(setter(Builder::ingestEndpointUrls))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IngestEndpointUrls").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LogPublishingOptions> LOG_PUBLISHING_OPTIONS_FIELD = SdkField
            .<LogPublishingOptions> builder(MarshallingType.SDK_POJO).memberName("LogPublishingOptions")
            .getter(getter(Pipeline::logPublishingOptions)).setter(setter(Builder::logPublishingOptions))
            .constructor(LogPublishingOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogPublishingOptions").build())
            .build();

    private static final SdkField<List<VpcEndpoint>> VPC_ENDPOINTS_FIELD = SdkField
            .<List<VpcEndpoint>> builder(MarshallingType.LIST)
            .memberName("VpcEndpoints")
            .getter(getter(Pipeline::vpcEndpoints))
            .setter(setter(Builder::vpcEndpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcEndpoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<VpcEndpoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(VpcEndpoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<BufferOptions> BUFFER_OPTIONS_FIELD = SdkField
            .<BufferOptions> builder(MarshallingType.SDK_POJO).memberName("BufferOptions")
            .getter(getter(Pipeline::bufferOptions)).setter(setter(Builder::bufferOptions)).constructor(BufferOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BufferOptions").build()).build();

    private static final SdkField<EncryptionAtRestOptions> ENCRYPTION_AT_REST_OPTIONS_FIELD = SdkField
            .<EncryptionAtRestOptions> builder(MarshallingType.SDK_POJO).memberName("EncryptionAtRestOptions")
            .getter(getter(Pipeline::encryptionAtRestOptions)).setter(setter(Builder::encryptionAtRestOptions))
            .constructor(EncryptionAtRestOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionAtRestOptions").build())
            .build();

    private static final SdkField<List<ServiceVpcEndpoint>> SERVICE_VPC_ENDPOINTS_FIELD = SdkField
            .<List<ServiceVpcEndpoint>> builder(MarshallingType.LIST)
            .memberName("ServiceVpcEndpoints")
            .getter(getter(Pipeline::serviceVpcEndpoints))
            .setter(setter(Builder::serviceVpcEndpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceVpcEndpoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ServiceVpcEndpoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(ServiceVpcEndpoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(Pipeline::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PIPELINE_NAME_FIELD,
            PIPELINE_ARN_FIELD, MIN_UNITS_FIELD, MAX_UNITS_FIELD, STATUS_FIELD, STATUS_REASON_FIELD,
            PIPELINE_CONFIGURATION_BODY_FIELD, CREATED_AT_FIELD, LAST_UPDATED_AT_FIELD, INGEST_ENDPOINT_URLS_FIELD,
            LOG_PUBLISHING_OPTIONS_FIELD, VPC_ENDPOINTS_FIELD, BUFFER_OPTIONS_FIELD, ENCRYPTION_AT_REST_OPTIONS_FIELD,
            SERVICE_VPC_ENDPOINTS_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String pipelineName;

    private final String pipelineArn;

    private final Integer minUnits;

    private final Integer maxUnits;

    private final String status;

    private final PipelineStatusReason statusReason;

    private final String pipelineConfigurationBody;

    private final Instant createdAt;

    private final Instant lastUpdatedAt;

    private final List<String> ingestEndpointUrls;

    private final LogPublishingOptions logPublishingOptions;

    private final List<VpcEndpoint> vpcEndpoints;

    private final BufferOptions bufferOptions;

    private final EncryptionAtRestOptions encryptionAtRestOptions;

    private final List<ServiceVpcEndpoint> serviceVpcEndpoints;

    private final List<Tag> tags;

    private Pipeline(BuilderImpl builder) {
        this.pipelineName = builder.pipelineName;
        this.pipelineArn = builder.pipelineArn;
        this.minUnits = builder.minUnits;
        this.maxUnits = builder.maxUnits;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.pipelineConfigurationBody = builder.pipelineConfigurationBody;
        this.createdAt = builder.createdAt;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.ingestEndpointUrls = builder.ingestEndpointUrls;
        this.logPublishingOptions = builder.logPublishingOptions;
        this.vpcEndpoints = builder.vpcEndpoints;
        this.bufferOptions = builder.bufferOptions;
        this.encryptionAtRestOptions = builder.encryptionAtRestOptions;
        this.serviceVpcEndpoints = builder.serviceVpcEndpoints;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the pipeline.
     * </p>
     * 
     * @return The name of the pipeline.
     */
    public final String pipelineName() {
        return pipelineName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the pipeline.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the pipeline.
     */
    public final String pipelineArn() {
        return pipelineArn;
    }

    /**
     * <p>
     * The minimum pipeline capacity, in Ingestion Compute Units (ICUs).
     * </p>
     * 
     * @return The minimum pipeline capacity, in Ingestion Compute Units (ICUs).
     */
    public final Integer minUnits() {
        return minUnits;
    }

    /**
     * <p>
     * The maximum pipeline capacity, in Ingestion Compute Units (ICUs).
     * </p>
     * 
     * @return The maximum pipeline capacity, in Ingestion Compute Units (ICUs).
     */
    public final Integer maxUnits() {
        return maxUnits;
    }

    /**
     * <p>
     * The current status of the pipeline.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PipelineStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the pipeline.
     * @see PipelineStatus
     */
    public final PipelineStatus status() {
        return PipelineStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the pipeline.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PipelineStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the pipeline.
     * @see PipelineStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The reason for the current status of the pipeline.
     * </p>
     * 
     * @return The reason for the current status of the pipeline.
     */
    public final PipelineStatusReason statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * The Data Prepper pipeline configuration in YAML format.
     * </p>
     * 
     * @return The Data Prepper pipeline configuration in YAML format.
     */
    public final String pipelineConfigurationBody() {
        return pipelineConfigurationBody;
    }

    /**
     * <p>
     * The date and time when the pipeline was created.
     * </p>
     * 
     * @return The date and time when the pipeline was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The date and time when the pipeline was last updated.
     * </p>
     * 
     * @return The date and time when the pipeline was last updated.
     */
    public final Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the IngestEndpointUrls property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIngestEndpointUrls() {
        return ingestEndpointUrls != null && !(ingestEndpointUrls instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ingestion endpoints for the pipeline, which you can send data to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIngestEndpointUrls} method.
     * </p>
     * 
     * @return The ingestion endpoints for the pipeline, which you can send data to.
     */
    public final List<String> ingestEndpointUrls() {
        return ingestEndpointUrls;
    }

    /**
     * <p>
     * Key-value pairs that represent log publishing settings.
     * </p>
     * 
     * @return Key-value pairs that represent log publishing settings.
     */
    public final LogPublishingOptions logPublishingOptions() {
        return logPublishingOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the VpcEndpoints property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVpcEndpoints() {
        return vpcEndpoints != null && !(vpcEndpoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The VPC interface endpoints that have access to the pipeline.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVpcEndpoints} method.
     * </p>
     * 
     * @return The VPC interface endpoints that have access to the pipeline.
     */
    public final List<VpcEndpoint> vpcEndpoints() {
        return vpcEndpoints;
    }

    /**
     * Returns the value of the BufferOptions property for this object.
     * 
     * @return The value of the BufferOptions property for this object.
     */
    public final BufferOptions bufferOptions() {
        return bufferOptions;
    }

    /**
     * Returns the value of the EncryptionAtRestOptions property for this object.
     * 
     * @return The value of the EncryptionAtRestOptions property for this object.
     */
    public final EncryptionAtRestOptions encryptionAtRestOptions() {
        return encryptionAtRestOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the ServiceVpcEndpoints property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasServiceVpcEndpoints() {
        return serviceVpcEndpoints != null && !(serviceVpcEndpoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of VPC endpoints that OpenSearch Ingestion has created to other AWS services.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasServiceVpcEndpoints} method.
     * </p>
     * 
     * @return A list of VPC endpoints that OpenSearch Ingestion has created to other AWS services.
     */
    public final List<ServiceVpcEndpoint> serviceVpcEndpoints() {
        return serviceVpcEndpoints;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags associated with the given pipeline.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags associated with the given pipeline.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(pipelineName());
        hashCode = 31 * hashCode + Objects.hashCode(pipelineArn());
        hashCode = 31 * hashCode + Objects.hashCode(minUnits());
        hashCode = 31 * hashCode + Objects.hashCode(maxUnits());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(pipelineConfigurationBody());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasIngestEndpointUrls() ? ingestEndpointUrls() : null);
        hashCode = 31 * hashCode + Objects.hashCode(logPublishingOptions());
        hashCode = 31 * hashCode + Objects.hashCode(hasVpcEndpoints() ? vpcEndpoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(bufferOptions());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionAtRestOptions());
        hashCode = 31 * hashCode + Objects.hashCode(hasServiceVpcEndpoints() ? serviceVpcEndpoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Pipeline)) {
            return false;
        }
        Pipeline other = (Pipeline) obj;
        return Objects.equals(pipelineName(), other.pipelineName()) && Objects.equals(pipelineArn(), other.pipelineArn())
                && Objects.equals(minUnits(), other.minUnits()) && Objects.equals(maxUnits(), other.maxUnits())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason())
                && Objects.equals(pipelineConfigurationBody(), other.pipelineConfigurationBody())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt())
                && hasIngestEndpointUrls() == other.hasIngestEndpointUrls()
                && Objects.equals(ingestEndpointUrls(), other.ingestEndpointUrls())
                && Objects.equals(logPublishingOptions(), other.logPublishingOptions())
                && hasVpcEndpoints() == other.hasVpcEndpoints() && Objects.equals(vpcEndpoints(), other.vpcEndpoints())
                && Objects.equals(bufferOptions(), other.bufferOptions())
                && Objects.equals(encryptionAtRestOptions(), other.encryptionAtRestOptions())
                && hasServiceVpcEndpoints() == other.hasServiceVpcEndpoints()
                && Objects.equals(serviceVpcEndpoints(), other.serviceVpcEndpoints()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Pipeline").add("PipelineName", pipelineName()).add("PipelineArn", pipelineArn())
                .add("MinUnits", minUnits()).add("MaxUnits", maxUnits()).add("Status", statusAsString())
                .add("StatusReason", statusReason()).add("PipelineConfigurationBody", pipelineConfigurationBody())
                .add("CreatedAt", createdAt()).add("LastUpdatedAt", lastUpdatedAt())
                .add("IngestEndpointUrls", hasIngestEndpointUrls() ? ingestEndpointUrls() : null)
                .add("LogPublishingOptions", logPublishingOptions())
                .add("VpcEndpoints", hasVpcEndpoints() ? vpcEndpoints() : null).add("BufferOptions", bufferOptions())
                .add("EncryptionAtRestOptions", encryptionAtRestOptions())
                .add("ServiceVpcEndpoints", hasServiceVpcEndpoints() ? serviceVpcEndpoints() : null)
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PipelineName":
            return Optional.ofNullable(clazz.cast(pipelineName()));
        case "PipelineArn":
            return Optional.ofNullable(clazz.cast(pipelineArn()));
        case "MinUnits":
            return Optional.ofNullable(clazz.cast(minUnits()));
        case "MaxUnits":
            return Optional.ofNullable(clazz.cast(maxUnits()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "PipelineConfigurationBody":
            return Optional.ofNullable(clazz.cast(pipelineConfigurationBody()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "LastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        case "IngestEndpointUrls":
            return Optional.ofNullable(clazz.cast(ingestEndpointUrls()));
        case "LogPublishingOptions":
            return Optional.ofNullable(clazz.cast(logPublishingOptions()));
        case "VpcEndpoints":
            return Optional.ofNullable(clazz.cast(vpcEndpoints()));
        case "BufferOptions":
            return Optional.ofNullable(clazz.cast(bufferOptions()));
        case "EncryptionAtRestOptions":
            return Optional.ofNullable(clazz.cast(encryptionAtRestOptions()));
        case "ServiceVpcEndpoints":
            return Optional.ofNullable(clazz.cast(serviceVpcEndpoints()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Pipeline, T> g) {
        return obj -> g.apply((Pipeline) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Pipeline> {
        /**
         * <p>
         * The name of the pipeline.
         * </p>
         * 
         * @param pipelineName
         *        The name of the pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pipelineName(String pipelineName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the pipeline.
         * </p>
         * 
         * @param pipelineArn
         *        The Amazon Resource Name (ARN) of the pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pipelineArn(String pipelineArn);

        /**
         * <p>
         * The minimum pipeline capacity, in Ingestion Compute Units (ICUs).
         * </p>
         * 
         * @param minUnits
         *        The minimum pipeline capacity, in Ingestion Compute Units (ICUs).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minUnits(Integer minUnits);

        /**
         * <p>
         * The maximum pipeline capacity, in Ingestion Compute Units (ICUs).
         * </p>
         * 
         * @param maxUnits
         *        The maximum pipeline capacity, in Ingestion Compute Units (ICUs).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxUnits(Integer maxUnits);

        /**
         * <p>
         * The current status of the pipeline.
         * </p>
         * 
         * @param status
         *        The current status of the pipeline.
         * @see PipelineStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PipelineStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the pipeline.
         * </p>
         * 
         * @param status
         *        The current status of the pipeline.
         * @see PipelineStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PipelineStatus
         */
        Builder status(PipelineStatus status);

        /**
         * <p>
         * The reason for the current status of the pipeline.
         * </p>
         * 
         * @param statusReason
         *        The reason for the current status of the pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(PipelineStatusReason statusReason);

        /**
         * <p>
         * The reason for the current status of the pipeline.
         * </p>
         * This is a convenience method that creates an instance of the {@link PipelineStatusReason.Builder} avoiding
         * the need to create one manually via {@link PipelineStatusReason#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PipelineStatusReason.Builder#build()} is called immediately and
         * its result is passed to {@link #statusReason(PipelineStatusReason)}.
         * 
         * @param statusReason
         *        a consumer that will call methods on {@link PipelineStatusReason.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statusReason(PipelineStatusReason)
         */
        default Builder statusReason(Consumer<PipelineStatusReason.Builder> statusReason) {
            return statusReason(PipelineStatusReason.builder().applyMutation(statusReason).build());
        }

        /**
         * <p>
         * The Data Prepper pipeline configuration in YAML format.
         * </p>
         * 
         * @param pipelineConfigurationBody
         *        The Data Prepper pipeline configuration in YAML format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pipelineConfigurationBody(String pipelineConfigurationBody);

        /**
         * <p>
         * The date and time when the pipeline was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time when the pipeline was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The date and time when the pipeline was last updated.
         * </p>
         * 
         * @param lastUpdatedAt
         *        The date and time when the pipeline was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * The ingestion endpoints for the pipeline, which you can send data to.
         * </p>
         * 
         * @param ingestEndpointUrls
         *        The ingestion endpoints for the pipeline, which you can send data to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ingestEndpointUrls(Collection<String> ingestEndpointUrls);

        /**
         * <p>
         * The ingestion endpoints for the pipeline, which you can send data to.
         * </p>
         * 
         * @param ingestEndpointUrls
         *        The ingestion endpoints for the pipeline, which you can send data to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ingestEndpointUrls(String... ingestEndpointUrls);

        /**
         * <p>
         * Key-value pairs that represent log publishing settings.
         * </p>
         * 
         * @param logPublishingOptions
         *        Key-value pairs that represent log publishing settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPublishingOptions(LogPublishingOptions logPublishingOptions);

        /**
         * <p>
         * Key-value pairs that represent log publishing settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link LogPublishingOptions.Builder} avoiding
         * the need to create one manually via {@link LogPublishingOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LogPublishingOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #logPublishingOptions(LogPublishingOptions)}.
         * 
         * @param logPublishingOptions
         *        a consumer that will call methods on {@link LogPublishingOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logPublishingOptions(LogPublishingOptions)
         */
        default Builder logPublishingOptions(Consumer<LogPublishingOptions.Builder> logPublishingOptions) {
            return logPublishingOptions(LogPublishingOptions.builder().applyMutation(logPublishingOptions).build());
        }

        /**
         * <p>
         * The VPC interface endpoints that have access to the pipeline.
         * </p>
         * 
         * @param vpcEndpoints
         *        The VPC interface endpoints that have access to the pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcEndpoints(Collection<VpcEndpoint> vpcEndpoints);

        /**
         * <p>
         * The VPC interface endpoints that have access to the pipeline.
         * </p>
         * 
         * @param vpcEndpoints
         *        The VPC interface endpoints that have access to the pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcEndpoints(VpcEndpoint... vpcEndpoints);

        /**
         * <p>
         * The VPC interface endpoints that have access to the pipeline.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.osis.model.VpcEndpoint.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.osis.model.VpcEndpoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.osis.model.VpcEndpoint.Builder#build()} is called immediately and its
         * result is passed to {@link #vpcEndpoints(List<VpcEndpoint>)}.
         * 
         * @param vpcEndpoints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.osis.model.VpcEndpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcEndpoints(java.util.Collection<VpcEndpoint>)
         */
        Builder vpcEndpoints(Consumer<VpcEndpoint.Builder>... vpcEndpoints);

        /**
         * Sets the value of the BufferOptions property for this object.
         *
         * @param bufferOptions
         *        The new value for the BufferOptions property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bufferOptions(BufferOptions bufferOptions);

        /**
         * Sets the value of the BufferOptions property for this object.
         *
         * This is a convenience method that creates an instance of the {@link BufferOptions.Builder} avoiding the need
         * to create one manually via {@link BufferOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link BufferOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #bufferOptions(BufferOptions)}.
         * 
         * @param bufferOptions
         *        a consumer that will call methods on {@link BufferOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bufferOptions(BufferOptions)
         */
        default Builder bufferOptions(Consumer<BufferOptions.Builder> bufferOptions) {
            return bufferOptions(BufferOptions.builder().applyMutation(bufferOptions).build());
        }

        /**
         * Sets the value of the EncryptionAtRestOptions property for this object.
         *
         * @param encryptionAtRestOptions
         *        The new value for the EncryptionAtRestOptions property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionAtRestOptions(EncryptionAtRestOptions encryptionAtRestOptions);

        /**
         * Sets the value of the EncryptionAtRestOptions property for this object.
         *
         * This is a convenience method that creates an instance of the {@link EncryptionAtRestOptions.Builder} avoiding
         * the need to create one manually via {@link EncryptionAtRestOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionAtRestOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionAtRestOptions(EncryptionAtRestOptions)}.
         * 
         * @param encryptionAtRestOptions
         *        a consumer that will call methods on {@link EncryptionAtRestOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionAtRestOptions(EncryptionAtRestOptions)
         */
        default Builder encryptionAtRestOptions(Consumer<EncryptionAtRestOptions.Builder> encryptionAtRestOptions) {
            return encryptionAtRestOptions(EncryptionAtRestOptions.builder().applyMutation(encryptionAtRestOptions).build());
        }

        /**
         * <p>
         * A list of VPC endpoints that OpenSearch Ingestion has created to other AWS services.
         * </p>
         * 
         * @param serviceVpcEndpoints
         *        A list of VPC endpoints that OpenSearch Ingestion has created to other AWS services.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceVpcEndpoints(Collection<ServiceVpcEndpoint> serviceVpcEndpoints);

        /**
         * <p>
         * A list of VPC endpoints that OpenSearch Ingestion has created to other AWS services.
         * </p>
         * 
         * @param serviceVpcEndpoints
         *        A list of VPC endpoints that OpenSearch Ingestion has created to other AWS services.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceVpcEndpoints(ServiceVpcEndpoint... serviceVpcEndpoints);

        /**
         * <p>
         * A list of VPC endpoints that OpenSearch Ingestion has created to other AWS services.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.osis.model.ServiceVpcEndpoint.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.osis.model.ServiceVpcEndpoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.osis.model.ServiceVpcEndpoint.Builder#build()} is called immediately
         * and its result is passed to {@link #serviceVpcEndpoints(List<ServiceVpcEndpoint>)}.
         * 
         * @param serviceVpcEndpoints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.osis.model.ServiceVpcEndpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serviceVpcEndpoints(java.util.Collection<ServiceVpcEndpoint>)
         */
        Builder serviceVpcEndpoints(Consumer<ServiceVpcEndpoint.Builder>... serviceVpcEndpoints);

        /**
         * <p>
         * A list of tags associated with the given pipeline.
         * </p>
         * 
         * @param tags
         *        A list of tags associated with the given pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags associated with the given pipeline.
         * </p>
         * 
         * @param tags
         *        A list of tags associated with the given pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags associated with the given pipeline.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.osis.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.osis.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.osis.model.Tag.Builder#build()}
         * is called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.osis.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String pipelineName;

        private String pipelineArn;

        private Integer minUnits;

        private Integer maxUnits;

        private String status;

        private PipelineStatusReason statusReason;

        private String pipelineConfigurationBody;

        private Instant createdAt;

        private Instant lastUpdatedAt;

        private List<String> ingestEndpointUrls = DefaultSdkAutoConstructList.getInstance();

        private LogPublishingOptions logPublishingOptions;

        private List<VpcEndpoint> vpcEndpoints = DefaultSdkAutoConstructList.getInstance();

        private BufferOptions bufferOptions;

        private EncryptionAtRestOptions encryptionAtRestOptions;

        private List<ServiceVpcEndpoint> serviceVpcEndpoints = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Pipeline model) {
            pipelineName(model.pipelineName);
            pipelineArn(model.pipelineArn);
            minUnits(model.minUnits);
            maxUnits(model.maxUnits);
            status(model.status);
            statusReason(model.statusReason);
            pipelineConfigurationBody(model.pipelineConfigurationBody);
            createdAt(model.createdAt);
            lastUpdatedAt(model.lastUpdatedAt);
            ingestEndpointUrls(model.ingestEndpointUrls);
            logPublishingOptions(model.logPublishingOptions);
            vpcEndpoints(model.vpcEndpoints);
            bufferOptions(model.bufferOptions);
            encryptionAtRestOptions(model.encryptionAtRestOptions);
            serviceVpcEndpoints(model.serviceVpcEndpoints);
            tags(model.tags);
        }

        public final String getPipelineName() {
            return pipelineName;
        }

        public final void setPipelineName(String pipelineName) {
            this.pipelineName = pipelineName;
        }

        @Override
        public final Builder pipelineName(String pipelineName) {
            this.pipelineName = pipelineName;
            return this;
        }

        public final String getPipelineArn() {
            return pipelineArn;
        }

        public final void setPipelineArn(String pipelineArn) {
            this.pipelineArn = pipelineArn;
        }

        @Override
        public final Builder pipelineArn(String pipelineArn) {
            this.pipelineArn = pipelineArn;
            return this;
        }

        public final Integer getMinUnits() {
            return minUnits;
        }

        public final void setMinUnits(Integer minUnits) {
            this.minUnits = minUnits;
        }

        @Override
        public final Builder minUnits(Integer minUnits) {
            this.minUnits = minUnits;
            return this;
        }

        public final Integer getMaxUnits() {
            return maxUnits;
        }

        public final void setMaxUnits(Integer maxUnits) {
            this.maxUnits = maxUnits;
        }

        @Override
        public final Builder maxUnits(Integer maxUnits) {
            this.maxUnits = maxUnits;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(PipelineStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final PipelineStatusReason.Builder getStatusReason() {
            return statusReason != null ? statusReason.toBuilder() : null;
        }

        public final void setStatusReason(PipelineStatusReason.BuilderImpl statusReason) {
            this.statusReason = statusReason != null ? statusReason.build() : null;
        }

        @Override
        public final Builder statusReason(PipelineStatusReason statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final String getPipelineConfigurationBody() {
            return pipelineConfigurationBody;
        }

        public final void setPipelineConfigurationBody(String pipelineConfigurationBody) {
            this.pipelineConfigurationBody = pipelineConfigurationBody;
        }

        @Override
        public final Builder pipelineConfigurationBody(String pipelineConfigurationBody) {
            this.pipelineConfigurationBody = pipelineConfigurationBody;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final Collection<String> getIngestEndpointUrls() {
            if (ingestEndpointUrls instanceof SdkAutoConstructList) {
                return null;
            }
            return ingestEndpointUrls;
        }

        public final void setIngestEndpointUrls(Collection<String> ingestEndpointUrls) {
            this.ingestEndpointUrls = IngestEndpointUrlsListCopier.copy(ingestEndpointUrls);
        }

        @Override
        public final Builder ingestEndpointUrls(Collection<String> ingestEndpointUrls) {
            this.ingestEndpointUrls = IngestEndpointUrlsListCopier.copy(ingestEndpointUrls);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ingestEndpointUrls(String... ingestEndpointUrls) {
            ingestEndpointUrls(Arrays.asList(ingestEndpointUrls));
            return this;
        }

        public final LogPublishingOptions.Builder getLogPublishingOptions() {
            return logPublishingOptions != null ? logPublishingOptions.toBuilder() : null;
        }

        public final void setLogPublishingOptions(LogPublishingOptions.BuilderImpl logPublishingOptions) {
            this.logPublishingOptions = logPublishingOptions != null ? logPublishingOptions.build() : null;
        }

        @Override
        public final Builder logPublishingOptions(LogPublishingOptions logPublishingOptions) {
            this.logPublishingOptions = logPublishingOptions;
            return this;
        }

        public final List<VpcEndpoint.Builder> getVpcEndpoints() {
            List<VpcEndpoint.Builder> result = VpcEndpointsListCopier.copyToBuilder(this.vpcEndpoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVpcEndpoints(Collection<VpcEndpoint.BuilderImpl> vpcEndpoints) {
            this.vpcEndpoints = VpcEndpointsListCopier.copyFromBuilder(vpcEndpoints);
        }

        @Override
        public final Builder vpcEndpoints(Collection<VpcEndpoint> vpcEndpoints) {
            this.vpcEndpoints = VpcEndpointsListCopier.copy(vpcEndpoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcEndpoints(VpcEndpoint... vpcEndpoints) {
            vpcEndpoints(Arrays.asList(vpcEndpoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcEndpoints(Consumer<VpcEndpoint.Builder>... vpcEndpoints) {
            vpcEndpoints(Stream.of(vpcEndpoints).map(c -> VpcEndpoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final BufferOptions.Builder getBufferOptions() {
            return bufferOptions != null ? bufferOptions.toBuilder() : null;
        }

        public final void setBufferOptions(BufferOptions.BuilderImpl bufferOptions) {
            this.bufferOptions = bufferOptions != null ? bufferOptions.build() : null;
        }

        @Override
        public final Builder bufferOptions(BufferOptions bufferOptions) {
            this.bufferOptions = bufferOptions;
            return this;
        }

        public final EncryptionAtRestOptions.Builder getEncryptionAtRestOptions() {
            return encryptionAtRestOptions != null ? encryptionAtRestOptions.toBuilder() : null;
        }

        public final void setEncryptionAtRestOptions(EncryptionAtRestOptions.BuilderImpl encryptionAtRestOptions) {
            this.encryptionAtRestOptions = encryptionAtRestOptions != null ? encryptionAtRestOptions.build() : null;
        }

        @Override
        public final Builder encryptionAtRestOptions(EncryptionAtRestOptions encryptionAtRestOptions) {
            this.encryptionAtRestOptions = encryptionAtRestOptions;
            return this;
        }

        public final List<ServiceVpcEndpoint.Builder> getServiceVpcEndpoints() {
            List<ServiceVpcEndpoint.Builder> result = ServiceVpcEndpointsListCopier.copyToBuilder(this.serviceVpcEndpoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setServiceVpcEndpoints(Collection<ServiceVpcEndpoint.BuilderImpl> serviceVpcEndpoints) {
            this.serviceVpcEndpoints = ServiceVpcEndpointsListCopier.copyFromBuilder(serviceVpcEndpoints);
        }

        @Override
        public final Builder serviceVpcEndpoints(Collection<ServiceVpcEndpoint> serviceVpcEndpoints) {
            this.serviceVpcEndpoints = ServiceVpcEndpointsListCopier.copy(serviceVpcEndpoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serviceVpcEndpoints(ServiceVpcEndpoint... serviceVpcEndpoints) {
            serviceVpcEndpoints(Arrays.asList(serviceVpcEndpoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serviceVpcEndpoints(Consumer<ServiceVpcEndpoint.Builder>... serviceVpcEndpoints) {
            serviceVpcEndpoints(Stream.of(serviceVpcEndpoints).map(c -> ServiceVpcEndpoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Pipeline build() {
            return new Pipeline(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
