/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.outposts.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an order.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Order implements SdkPojo, Serializable, ToCopyableBuilder<Order.Builder, Order> {
    private static final SdkField<String> OUTPOST_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutpostId").getter(getter(Order::outpostId)).setter(setter(Builder::outpostId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutpostId").build()).build();

    private static final SdkField<String> ORDER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OrderId").getter(getter(Order::orderId)).setter(setter(Builder::orderId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrderId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Order::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<LineItem>> LINE_ITEMS_FIELD = SdkField
            .<List<LineItem>> builder(MarshallingType.LIST)
            .memberName("LineItems")
            .getter(getter(Order::lineItems))
            .setter(setter(Builder::lineItems))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LineItems").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LineItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(LineItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PAYMENT_OPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PaymentOption").getter(getter(Order::paymentOptionAsString)).setter(setter(Builder::paymentOption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PaymentOption").build()).build();

    private static final SdkField<Instant> ORDER_SUBMISSION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("OrderSubmissionDate").getter(getter(Order::orderSubmissionDate))
            .setter(setter(Builder::orderSubmissionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrderSubmissionDate").build())
            .build();

    private static final SdkField<Instant> ORDER_FULFILLED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("OrderFulfilledDate").getter(getter(Order::orderFulfilledDate))
            .setter(setter(Builder::orderFulfilledDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrderFulfilledDate").build())
            .build();

    private static final SdkField<String> PAYMENT_TERM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PaymentTerm").getter(getter(Order::paymentTermAsString)).setter(setter(Builder::paymentTerm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PaymentTerm").build()).build();

    private static final SdkField<String> ORDER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OrderType").getter(getter(Order::orderTypeAsString)).setter(setter(Builder::orderType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrderType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OUTPOST_ID_FIELD,
            ORDER_ID_FIELD, STATUS_FIELD, LINE_ITEMS_FIELD, PAYMENT_OPTION_FIELD, ORDER_SUBMISSION_DATE_FIELD,
            ORDER_FULFILLED_DATE_FIELD, PAYMENT_TERM_FIELD, ORDER_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String outpostId;

    private final String orderId;

    private final String status;

    private final List<LineItem> lineItems;

    private final String paymentOption;

    private final Instant orderSubmissionDate;

    private final Instant orderFulfilledDate;

    private final String paymentTerm;

    private final String orderType;

    private Order(BuilderImpl builder) {
        this.outpostId = builder.outpostId;
        this.orderId = builder.orderId;
        this.status = builder.status;
        this.lineItems = builder.lineItems;
        this.paymentOption = builder.paymentOption;
        this.orderSubmissionDate = builder.orderSubmissionDate;
        this.orderFulfilledDate = builder.orderFulfilledDate;
        this.paymentTerm = builder.paymentTerm;
        this.orderType = builder.orderType;
    }

    /**
     * <p>
     * The ID of the Outpost in the order.
     * </p>
     * 
     * @return The ID of the Outpost in the order.
     */
    public final String outpostId() {
        return outpostId;
    }

    /**
     * <p>
     * The ID of the order.
     * </p>
     * 
     * @return The ID of the order.
     */
    public final String orderId() {
        return orderId;
    }

    /**
     * <p>
     * The status of the order.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PREPARING</code> - Order is received and being prepared.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code> - Order is either being built, shipped, or installed. To get more details, see the line
     * item status.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COMPLETED</code> - Order is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCELLED</code> - Order is cancelled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ERROR</code> - Customer should contact support.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The following status are deprecated: <code>RECEIVED</code>, <code>PENDING</code>, <code>PROCESSING</code>,
     * <code>INSTALLING</code>, and <code>FULFILLED</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link OrderStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the order.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PREPARING</code> - Order is received and being prepared.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code> - Order is either being built, shipped, or installed. To get more details, see
     *         the line item status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COMPLETED</code> - Order is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCELLED</code> - Order is cancelled.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ERROR</code> - Customer should contact support.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The following status are deprecated: <code>RECEIVED</code>, <code>PENDING</code>, <code>PROCESSING</code>, <code>INSTALLING</code>, and <code>FULFILLED</code>.
     *         </p>
     * @see OrderStatus
     */
    public final OrderStatus status() {
        return OrderStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the order.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PREPARING</code> - Order is received and being prepared.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code> - Order is either being built, shipped, or installed. To get more details, see the line
     * item status.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COMPLETED</code> - Order is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCELLED</code> - Order is cancelled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ERROR</code> - Customer should contact support.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The following status are deprecated: <code>RECEIVED</code>, <code>PENDING</code>, <code>PROCESSING</code>,
     * <code>INSTALLING</code>, and <code>FULFILLED</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link OrderStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the order.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PREPARING</code> - Order is received and being prepared.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code> - Order is either being built, shipped, or installed. To get more details, see
     *         the line item status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COMPLETED</code> - Order is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCELLED</code> - Order is cancelled.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ERROR</code> - Customer should contact support.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The following status are deprecated: <code>RECEIVED</code>, <code>PENDING</code>, <code>PROCESSING</code>, <code>INSTALLING</code>, and <code>FULFILLED</code>.
     *         </p>
     * @see OrderStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the LineItems property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasLineItems() {
        return lineItems != null && !(lineItems instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The line items for the order
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLineItems} method.
     * </p>
     * 
     * @return The line items for the order
     */
    public final List<LineItem> lineItems() {
        return lineItems;
    }

    /**
     * <p>
     * The payment option for the order.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #paymentOption}
     * will return {@link PaymentOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #paymentOptionAsString}.
     * </p>
     * 
     * @return The payment option for the order.
     * @see PaymentOption
     */
    public final PaymentOption paymentOption() {
        return PaymentOption.fromValue(paymentOption);
    }

    /**
     * <p>
     * The payment option for the order.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #paymentOption}
     * will return {@link PaymentOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #paymentOptionAsString}.
     * </p>
     * 
     * @return The payment option for the order.
     * @see PaymentOption
     */
    public final String paymentOptionAsString() {
        return paymentOption;
    }

    /**
     * <p>
     * The submission date for the order.
     * </p>
     * 
     * @return The submission date for the order.
     */
    public final Instant orderSubmissionDate() {
        return orderSubmissionDate;
    }

    /**
     * <p>
     * The fulfillment date of the order.
     * </p>
     * 
     * @return The fulfillment date of the order.
     */
    public final Instant orderFulfilledDate() {
        return orderFulfilledDate;
    }

    /**
     * <p>
     * The payment term.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #paymentTerm} will
     * return {@link PaymentTerm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #paymentTermAsString}.
     * </p>
     * 
     * @return The payment term.
     * @see PaymentTerm
     */
    public final PaymentTerm paymentTerm() {
        return PaymentTerm.fromValue(paymentTerm);
    }

    /**
     * <p>
     * The payment term.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #paymentTerm} will
     * return {@link PaymentTerm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #paymentTermAsString}.
     * </p>
     * 
     * @return The payment term.
     * @see PaymentTerm
     */
    public final String paymentTermAsString() {
        return paymentTerm;
    }

    /**
     * <p>
     * The type of order.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #orderType} will
     * return {@link OrderType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #orderTypeAsString}.
     * </p>
     * 
     * @return The type of order.
     * @see OrderType
     */
    public final OrderType orderType() {
        return OrderType.fromValue(orderType);
    }

    /**
     * <p>
     * The type of order.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #orderType} will
     * return {@link OrderType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #orderTypeAsString}.
     * </p>
     * 
     * @return The type of order.
     * @see OrderType
     */
    public final String orderTypeAsString() {
        return orderType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(outpostId());
        hashCode = 31 * hashCode + Objects.hashCode(orderId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasLineItems() ? lineItems() : null);
        hashCode = 31 * hashCode + Objects.hashCode(paymentOptionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(orderSubmissionDate());
        hashCode = 31 * hashCode + Objects.hashCode(orderFulfilledDate());
        hashCode = 31 * hashCode + Objects.hashCode(paymentTermAsString());
        hashCode = 31 * hashCode + Objects.hashCode(orderTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Order)) {
            return false;
        }
        Order other = (Order) obj;
        return Objects.equals(outpostId(), other.outpostId()) && Objects.equals(orderId(), other.orderId())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasLineItems() == other.hasLineItems()
                && Objects.equals(lineItems(), other.lineItems())
                && Objects.equals(paymentOptionAsString(), other.paymentOptionAsString())
                && Objects.equals(orderSubmissionDate(), other.orderSubmissionDate())
                && Objects.equals(orderFulfilledDate(), other.orderFulfilledDate())
                && Objects.equals(paymentTermAsString(), other.paymentTermAsString())
                && Objects.equals(orderTypeAsString(), other.orderTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Order").add("OutpostId", outpostId()).add("OrderId", orderId()).add("Status", statusAsString())
                .add("LineItems", hasLineItems() ? lineItems() : null).add("PaymentOption", paymentOptionAsString())
                .add("OrderSubmissionDate", orderSubmissionDate()).add("OrderFulfilledDate", orderFulfilledDate())
                .add("PaymentTerm", paymentTermAsString()).add("OrderType", orderTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OutpostId":
            return Optional.ofNullable(clazz.cast(outpostId()));
        case "OrderId":
            return Optional.ofNullable(clazz.cast(orderId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "LineItems":
            return Optional.ofNullable(clazz.cast(lineItems()));
        case "PaymentOption":
            return Optional.ofNullable(clazz.cast(paymentOptionAsString()));
        case "OrderSubmissionDate":
            return Optional.ofNullable(clazz.cast(orderSubmissionDate()));
        case "OrderFulfilledDate":
            return Optional.ofNullable(clazz.cast(orderFulfilledDate()));
        case "PaymentTerm":
            return Optional.ofNullable(clazz.cast(paymentTermAsString()));
        case "OrderType":
            return Optional.ofNullable(clazz.cast(orderTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Order, T> g) {
        return obj -> g.apply((Order) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Order> {
        /**
         * <p>
         * The ID of the Outpost in the order.
         * </p>
         * 
         * @param outpostId
         *        The ID of the Outpost in the order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outpostId(String outpostId);

        /**
         * <p>
         * The ID of the order.
         * </p>
         * 
         * @param orderId
         *        The ID of the order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder orderId(String orderId);

        /**
         * <p>
         * The status of the order.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PREPARING</code> - Order is received and being prepared.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code> - Order is either being built, shipped, or installed. To get more details, see the
         * line item status.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COMPLETED</code> - Order is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCELLED</code> - Order is cancelled.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ERROR</code> - Customer should contact support.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The following status are deprecated: <code>RECEIVED</code>, <code>PENDING</code>, <code>PROCESSING</code>,
         * <code>INSTALLING</code>, and <code>FULFILLED</code>.
         * </p>
         * </note>
         * 
         * @param status
         *        The status of the order.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PREPARING</code> - Order is received and being prepared.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code> - Order is either being built, shipped, or installed. To get more details,
         *        see the line item status.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COMPLETED</code> - Order is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCELLED</code> - Order is cancelled.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ERROR</code> - Customer should contact support.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The following status are deprecated: <code>RECEIVED</code>, <code>PENDING</code>,
         *        <code>PROCESSING</code>, <code>INSTALLING</code>, and <code>FULFILLED</code>.
         *        </p>
         * @see OrderStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OrderStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the order.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PREPARING</code> - Order is received and being prepared.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code> - Order is either being built, shipped, or installed. To get more details, see the
         * line item status.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COMPLETED</code> - Order is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCELLED</code> - Order is cancelled.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ERROR</code> - Customer should contact support.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The following status are deprecated: <code>RECEIVED</code>, <code>PENDING</code>, <code>PROCESSING</code>,
         * <code>INSTALLING</code>, and <code>FULFILLED</code>.
         * </p>
         * </note>
         * 
         * @param status
         *        The status of the order.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PREPARING</code> - Order is received and being prepared.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code> - Order is either being built, shipped, or installed. To get more details,
         *        see the line item status.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COMPLETED</code> - Order is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCELLED</code> - Order is cancelled.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ERROR</code> - Customer should contact support.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The following status are deprecated: <code>RECEIVED</code>, <code>PENDING</code>,
         *        <code>PROCESSING</code>, <code>INSTALLING</code>, and <code>FULFILLED</code>.
         *        </p>
         * @see OrderStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OrderStatus
         */
        Builder status(OrderStatus status);

        /**
         * <p>
         * The line items for the order
         * </p>
         * 
         * @param lineItems
         *        The line items for the order
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lineItems(Collection<LineItem> lineItems);

        /**
         * <p>
         * The line items for the order
         * </p>
         * 
         * @param lineItems
         *        The line items for the order
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lineItems(LineItem... lineItems);

        /**
         * <p>
         * The line items for the order
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.outposts.model.LineItem.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.outposts.model.LineItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.outposts.model.LineItem.Builder#build()} is called immediately and its
         * result is passed to {@link #lineItems(List<LineItem>)}.
         * 
         * @param lineItems
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.outposts.model.LineItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lineItems(java.util.Collection<LineItem>)
         */
        Builder lineItems(Consumer<LineItem.Builder>... lineItems);

        /**
         * <p>
         * The payment option for the order.
         * </p>
         * 
         * @param paymentOption
         *        The payment option for the order.
         * @see PaymentOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PaymentOption
         */
        Builder paymentOption(String paymentOption);

        /**
         * <p>
         * The payment option for the order.
         * </p>
         * 
         * @param paymentOption
         *        The payment option for the order.
         * @see PaymentOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PaymentOption
         */
        Builder paymentOption(PaymentOption paymentOption);

        /**
         * <p>
         * The submission date for the order.
         * </p>
         * 
         * @param orderSubmissionDate
         *        The submission date for the order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder orderSubmissionDate(Instant orderSubmissionDate);

        /**
         * <p>
         * The fulfillment date of the order.
         * </p>
         * 
         * @param orderFulfilledDate
         *        The fulfillment date of the order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder orderFulfilledDate(Instant orderFulfilledDate);

        /**
         * <p>
         * The payment term.
         * </p>
         * 
         * @param paymentTerm
         *        The payment term.
         * @see PaymentTerm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PaymentTerm
         */
        Builder paymentTerm(String paymentTerm);

        /**
         * <p>
         * The payment term.
         * </p>
         * 
         * @param paymentTerm
         *        The payment term.
         * @see PaymentTerm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PaymentTerm
         */
        Builder paymentTerm(PaymentTerm paymentTerm);

        /**
         * <p>
         * The type of order.
         * </p>
         * 
         * @param orderType
         *        The type of order.
         * @see OrderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OrderType
         */
        Builder orderType(String orderType);

        /**
         * <p>
         * The type of order.
         * </p>
         * 
         * @param orderType
         *        The type of order.
         * @see OrderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OrderType
         */
        Builder orderType(OrderType orderType);
    }

    static final class BuilderImpl implements Builder {
        private String outpostId;

        private String orderId;

        private String status;

        private List<LineItem> lineItems = DefaultSdkAutoConstructList.getInstance();

        private String paymentOption;

        private Instant orderSubmissionDate;

        private Instant orderFulfilledDate;

        private String paymentTerm;

        private String orderType;

        private BuilderImpl() {
        }

        private BuilderImpl(Order model) {
            outpostId(model.outpostId);
            orderId(model.orderId);
            status(model.status);
            lineItems(model.lineItems);
            paymentOption(model.paymentOption);
            orderSubmissionDate(model.orderSubmissionDate);
            orderFulfilledDate(model.orderFulfilledDate);
            paymentTerm(model.paymentTerm);
            orderType(model.orderType);
        }

        public final String getOutpostId() {
            return outpostId;
        }

        public final void setOutpostId(String outpostId) {
            this.outpostId = outpostId;
        }

        @Override
        public final Builder outpostId(String outpostId) {
            this.outpostId = outpostId;
            return this;
        }

        public final String getOrderId() {
            return orderId;
        }

        public final void setOrderId(String orderId) {
            this.orderId = orderId;
        }

        @Override
        public final Builder orderId(String orderId) {
            this.orderId = orderId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(OrderStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final List<LineItem.Builder> getLineItems() {
            List<LineItem.Builder> result = LineItemListDefinitionCopier.copyToBuilder(this.lineItems);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLineItems(Collection<LineItem.BuilderImpl> lineItems) {
            this.lineItems = LineItemListDefinitionCopier.copyFromBuilder(lineItems);
        }

        @Override
        public final Builder lineItems(Collection<LineItem> lineItems) {
            this.lineItems = LineItemListDefinitionCopier.copy(lineItems);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lineItems(LineItem... lineItems) {
            lineItems(Arrays.asList(lineItems));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lineItems(Consumer<LineItem.Builder>... lineItems) {
            lineItems(Stream.of(lineItems).map(c -> LineItem.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getPaymentOption() {
            return paymentOption;
        }

        public final void setPaymentOption(String paymentOption) {
            this.paymentOption = paymentOption;
        }

        @Override
        public final Builder paymentOption(String paymentOption) {
            this.paymentOption = paymentOption;
            return this;
        }

        @Override
        public final Builder paymentOption(PaymentOption paymentOption) {
            this.paymentOption(paymentOption == null ? null : paymentOption.toString());
            return this;
        }

        public final Instant getOrderSubmissionDate() {
            return orderSubmissionDate;
        }

        public final void setOrderSubmissionDate(Instant orderSubmissionDate) {
            this.orderSubmissionDate = orderSubmissionDate;
        }

        @Override
        public final Builder orderSubmissionDate(Instant orderSubmissionDate) {
            this.orderSubmissionDate = orderSubmissionDate;
            return this;
        }

        public final Instant getOrderFulfilledDate() {
            return orderFulfilledDate;
        }

        public final void setOrderFulfilledDate(Instant orderFulfilledDate) {
            this.orderFulfilledDate = orderFulfilledDate;
        }

        @Override
        public final Builder orderFulfilledDate(Instant orderFulfilledDate) {
            this.orderFulfilledDate = orderFulfilledDate;
            return this;
        }

        public final String getPaymentTerm() {
            return paymentTerm;
        }

        public final void setPaymentTerm(String paymentTerm) {
            this.paymentTerm = paymentTerm;
        }

        @Override
        public final Builder paymentTerm(String paymentTerm) {
            this.paymentTerm = paymentTerm;
            return this;
        }

        @Override
        public final Builder paymentTerm(PaymentTerm paymentTerm) {
            this.paymentTerm(paymentTerm == null ? null : paymentTerm.toString());
            return this;
        }

        public final String getOrderType() {
            return orderType;
        }

        public final void setOrderType(String orderType) {
            this.orderType = orderType;
        }

        @Override
        public final Builder orderType(String orderType) {
            this.orderType = orderType;
            return this;
        }

        @Override
        public final Builder orderType(OrderType orderType) {
            this.orderType(orderType == null ? null : orderType.toString());
            return this;
        }

        @Override
        public Order build() {
            return new Order(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
