/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.panorama;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.panorama.internal.PanoramaServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.panorama.model.AccessDeniedException;
import software.amazon.awssdk.services.panorama.model.ConflictException;
import software.amazon.awssdk.services.panorama.model.CreateApplicationInstanceRequest;
import software.amazon.awssdk.services.panorama.model.CreateApplicationInstanceResponse;
import software.amazon.awssdk.services.panorama.model.CreateJobForDevicesRequest;
import software.amazon.awssdk.services.panorama.model.CreateJobForDevicesResponse;
import software.amazon.awssdk.services.panorama.model.CreateNodeFromTemplateJobRequest;
import software.amazon.awssdk.services.panorama.model.CreateNodeFromTemplateJobResponse;
import software.amazon.awssdk.services.panorama.model.CreatePackageImportJobRequest;
import software.amazon.awssdk.services.panorama.model.CreatePackageImportJobResponse;
import software.amazon.awssdk.services.panorama.model.CreatePackageRequest;
import software.amazon.awssdk.services.panorama.model.CreatePackageResponse;
import software.amazon.awssdk.services.panorama.model.DeleteDeviceRequest;
import software.amazon.awssdk.services.panorama.model.DeleteDeviceResponse;
import software.amazon.awssdk.services.panorama.model.DeletePackageRequest;
import software.amazon.awssdk.services.panorama.model.DeletePackageResponse;
import software.amazon.awssdk.services.panorama.model.DeregisterPackageVersionRequest;
import software.amazon.awssdk.services.panorama.model.DeregisterPackageVersionResponse;
import software.amazon.awssdk.services.panorama.model.DescribeApplicationInstanceDetailsRequest;
import software.amazon.awssdk.services.panorama.model.DescribeApplicationInstanceDetailsResponse;
import software.amazon.awssdk.services.panorama.model.DescribeApplicationInstanceRequest;
import software.amazon.awssdk.services.panorama.model.DescribeApplicationInstanceResponse;
import software.amazon.awssdk.services.panorama.model.DescribeDeviceJobRequest;
import software.amazon.awssdk.services.panorama.model.DescribeDeviceJobResponse;
import software.amazon.awssdk.services.panorama.model.DescribeDeviceRequest;
import software.amazon.awssdk.services.panorama.model.DescribeDeviceResponse;
import software.amazon.awssdk.services.panorama.model.DescribeNodeFromTemplateJobRequest;
import software.amazon.awssdk.services.panorama.model.DescribeNodeFromTemplateJobResponse;
import software.amazon.awssdk.services.panorama.model.DescribeNodeRequest;
import software.amazon.awssdk.services.panorama.model.DescribeNodeResponse;
import software.amazon.awssdk.services.panorama.model.DescribePackageImportJobRequest;
import software.amazon.awssdk.services.panorama.model.DescribePackageImportJobResponse;
import software.amazon.awssdk.services.panorama.model.DescribePackageRequest;
import software.amazon.awssdk.services.panorama.model.DescribePackageResponse;
import software.amazon.awssdk.services.panorama.model.DescribePackageVersionRequest;
import software.amazon.awssdk.services.panorama.model.DescribePackageVersionResponse;
import software.amazon.awssdk.services.panorama.model.InternalServerException;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstanceDependenciesRequest;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstanceDependenciesResponse;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstanceNodeInstancesRequest;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstanceNodeInstancesResponse;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstancesRequest;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstancesResponse;
import software.amazon.awssdk.services.panorama.model.ListDevicesJobsRequest;
import software.amazon.awssdk.services.panorama.model.ListDevicesJobsResponse;
import software.amazon.awssdk.services.panorama.model.ListDevicesRequest;
import software.amazon.awssdk.services.panorama.model.ListDevicesResponse;
import software.amazon.awssdk.services.panorama.model.ListNodeFromTemplateJobsRequest;
import software.amazon.awssdk.services.panorama.model.ListNodeFromTemplateJobsResponse;
import software.amazon.awssdk.services.panorama.model.ListNodesRequest;
import software.amazon.awssdk.services.panorama.model.ListNodesResponse;
import software.amazon.awssdk.services.panorama.model.ListPackageImportJobsRequest;
import software.amazon.awssdk.services.panorama.model.ListPackageImportJobsResponse;
import software.amazon.awssdk.services.panorama.model.ListPackagesRequest;
import software.amazon.awssdk.services.panorama.model.ListPackagesResponse;
import software.amazon.awssdk.services.panorama.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.panorama.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.panorama.model.PanoramaException;
import software.amazon.awssdk.services.panorama.model.ProvisionDeviceRequest;
import software.amazon.awssdk.services.panorama.model.ProvisionDeviceResponse;
import software.amazon.awssdk.services.panorama.model.RegisterPackageVersionRequest;
import software.amazon.awssdk.services.panorama.model.RegisterPackageVersionResponse;
import software.amazon.awssdk.services.panorama.model.RemoveApplicationInstanceRequest;
import software.amazon.awssdk.services.panorama.model.RemoveApplicationInstanceResponse;
import software.amazon.awssdk.services.panorama.model.ResourceNotFoundException;
import software.amazon.awssdk.services.panorama.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.panorama.model.SignalApplicationInstanceNodeInstancesRequest;
import software.amazon.awssdk.services.panorama.model.SignalApplicationInstanceNodeInstancesResponse;
import software.amazon.awssdk.services.panorama.model.TagResourceRequest;
import software.amazon.awssdk.services.panorama.model.TagResourceResponse;
import software.amazon.awssdk.services.panorama.model.UntagResourceRequest;
import software.amazon.awssdk.services.panorama.model.UntagResourceResponse;
import software.amazon.awssdk.services.panorama.model.UpdateDeviceMetadataRequest;
import software.amazon.awssdk.services.panorama.model.UpdateDeviceMetadataResponse;
import software.amazon.awssdk.services.panorama.model.ValidationException;
import software.amazon.awssdk.services.panorama.transform.CreateApplicationInstanceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.CreateJobForDevicesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.CreateNodeFromTemplateJobRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.CreatePackageImportJobRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.CreatePackageRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DeleteDeviceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DeletePackageRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DeregisterPackageVersionRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeApplicationInstanceDetailsRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeApplicationInstanceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeDeviceJobRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeDeviceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeNodeFromTemplateJobRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeNodeRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribePackageImportJobRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribePackageRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribePackageVersionRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListApplicationInstanceDependenciesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListApplicationInstanceNodeInstancesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListApplicationInstancesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListDevicesJobsRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListDevicesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListNodeFromTemplateJobsRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListNodesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListPackageImportJobsRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListPackagesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ProvisionDeviceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.RegisterPackageVersionRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.RemoveApplicationInstanceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.SignalApplicationInstanceNodeInstancesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.UpdateDeviceMetadataRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link PanoramaClient}.
 *
 * @see PanoramaClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPanoramaClient implements PanoramaClient {
    private static final Logger log = Logger.loggerFor(DefaultPanoramaClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPanoramaClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an application instance and deploys it to a device.
     * </p>
     *
     * @param createApplicationInstanceRequest
     * @return Result of the CreateApplicationInstance operation returned by the service.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ServiceQuotaExceededException
     *         The request would cause a limit to be exceeded.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.CreateApplicationInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/CreateApplicationInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateApplicationInstanceResponse createApplicationInstance(
            CreateApplicationInstanceRequest createApplicationInstanceRequest) throws ValidationException, AccessDeniedException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateApplicationInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateApplicationInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createApplicationInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplicationInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationInstanceRequest, CreateApplicationInstanceResponse>()
                            .withOperationName("CreateApplicationInstance").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createApplicationInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateApplicationInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a job to run on a device. A job can update a device's software or reboot it.
     * </p>
     *
     * @param createJobForDevicesRequest
     * @return Result of the CreateJobForDevices operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.CreateJobForDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/CreateJobForDevices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateJobForDevicesResponse createJobForDevices(CreateJobForDevicesRequest createJobForDevicesRequest)
            throws ConflictException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateJobForDevicesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateJobForDevicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createJobForDevicesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobForDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJobForDevices");

            return clientHandler.execute(new ClientExecutionParams<CreateJobForDevicesRequest, CreateJobForDevicesResponse>()
                    .withOperationName("CreateJobForDevices").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createJobForDevicesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateJobForDevicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a camera stream node.
     * </p>
     *
     * @param createNodeFromTemplateJobRequest
     * @return Result of the CreateNodeFromTemplateJob operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.CreateNodeFromTemplateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/CreateNodeFromTemplateJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateNodeFromTemplateJobResponse createNodeFromTemplateJob(
            CreateNodeFromTemplateJobRequest createNodeFromTemplateJobRequest) throws ConflictException, ValidationException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateNodeFromTemplateJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateNodeFromTemplateJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createNodeFromTemplateJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNodeFromTemplateJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNodeFromTemplateJob");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateNodeFromTemplateJobRequest, CreateNodeFromTemplateJobResponse>()
                            .withOperationName("CreateNodeFromTemplateJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createNodeFromTemplateJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateNodeFromTemplateJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a package and storage location in an Amazon S3 access point.
     * </p>
     *
     * @param createPackageRequest
     * @return Result of the CreatePackage operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.CreatePackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/CreatePackage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreatePackageResponse createPackage(CreatePackageRequest createPackageRequest) throws ConflictException,
            ValidationException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePackageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreatePackageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPackageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePackage");

            return clientHandler.execute(new ClientExecutionParams<CreatePackageRequest, CreatePackageResponse>()
                    .withOperationName("CreatePackage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createPackageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreatePackageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Imports a node package.
     * </p>
     *
     * @param createPackageImportJobRequest
     * @return Result of the CreatePackageImportJob operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.CreatePackageImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/CreatePackageImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePackageImportJobResponse createPackageImportJob(CreatePackageImportJobRequest createPackageImportJobRequest)
            throws ConflictException, ValidationException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePackageImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePackageImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPackageImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPackageImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePackageImportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePackageImportJobRequest, CreatePackageImportJobResponse>()
                            .withOperationName("CreatePackageImportJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createPackageImportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreatePackageImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a device.
     * </p>
     *
     * @param deleteDeviceRequest
     * @return Result of the DeleteDevice operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DeleteDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DeleteDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteDeviceResponse deleteDevice(DeleteDeviceRequest deleteDeviceRequest) throws ConflictException,
            ValidationException, AccessDeniedException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDeviceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDevice");

            return clientHandler.execute(new ClientExecutionParams<DeleteDeviceRequest, DeleteDeviceResponse>()
                    .withOperationName("DeleteDevice").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteDeviceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDeviceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a package.
     * </p>
     * <note>
     * <p>
     * To delete a package, you need permission to call <code>s3:DeleteObject</code> in addition to permissions for the
     * AWS Panorama API.
     * </p>
     * </note>
     *
     * @param deletePackageRequest
     * @return Result of the DeletePackage operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DeletePackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DeletePackage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePackageResponse deletePackage(DeletePackageRequest deletePackageRequest) throws ConflictException,
            ValidationException, AccessDeniedException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePackageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePackageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePackageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePackage");

            return clientHandler.execute(new ClientExecutionParams<DeletePackageRequest, DeletePackageResponse>()
                    .withOperationName("DeletePackage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deletePackageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePackageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deregisters a package version.
     * </p>
     *
     * @param deregisterPackageVersionRequest
     * @return Result of the DeregisterPackageVersion operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DeregisterPackageVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DeregisterPackageVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeregisterPackageVersionResponse deregisterPackageVersion(
            DeregisterPackageVersionRequest deregisterPackageVersionRequest) throws ConflictException, ValidationException,
            AccessDeniedException, ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterPackageVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterPackageVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterPackageVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterPackageVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterPackageVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<DeregisterPackageVersionRequest, DeregisterPackageVersionResponse>()
                            .withOperationName("DeregisterPackageVersion").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deregisterPackageVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeregisterPackageVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about an application instance on a device.
     * </p>
     *
     * @param describeApplicationInstanceRequest
     * @return Result of the DescribeApplicationInstance operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DescribeApplicationInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeApplicationInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeApplicationInstanceResponse describeApplicationInstance(
            DescribeApplicationInstanceRequest describeApplicationInstanceRequest) throws ConflictException, ValidationException,
            AccessDeniedException, ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeApplicationInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeApplicationInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeApplicationInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeApplicationInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeApplicationInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeApplicationInstanceRequest, DescribeApplicationInstanceResponse>()
                            .withOperationName("DescribeApplicationInstance").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeApplicationInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeApplicationInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about an application instance's configuration manifest.
     * </p>
     *
     * @param describeApplicationInstanceDetailsRequest
     * @return Result of the DescribeApplicationInstanceDetails operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DescribeApplicationInstanceDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeApplicationInstanceDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeApplicationInstanceDetailsResponse describeApplicationInstanceDetails(
            DescribeApplicationInstanceDetailsRequest describeApplicationInstanceDetailsRequest) throws ConflictException,
            ValidationException, AccessDeniedException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeApplicationInstanceDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeApplicationInstanceDetailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeApplicationInstanceDetailsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeApplicationInstanceDetailsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeApplicationInstanceDetails");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeApplicationInstanceDetailsRequest, DescribeApplicationInstanceDetailsResponse>()
                            .withOperationName("DescribeApplicationInstanceDetails").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeApplicationInstanceDetailsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeApplicationInstanceDetailsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a device.
     * </p>
     *
     * @param describeDeviceRequest
     * @return Result of the DescribeDevice operation returned by the service.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DescribeDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeDeviceResponse describeDevice(DescribeDeviceRequest describeDeviceRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDeviceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDevice");

            return clientHandler.execute(new ClientExecutionParams<DescribeDeviceRequest, DescribeDeviceResponse>()
                    .withOperationName("DescribeDevice").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeDeviceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDeviceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a device job.
     * </p>
     *
     * @param describeDeviceJobRequest
     * @return Result of the DescribeDeviceJob operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DescribeDeviceJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeDeviceJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeDeviceJobResponse describeDeviceJob(DescribeDeviceJobRequest describeDeviceJobRequest)
            throws ConflictException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDeviceJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDeviceJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDeviceJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeviceJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDeviceJob");

            return clientHandler.execute(new ClientExecutionParams<DescribeDeviceJobRequest, DescribeDeviceJobResponse>()
                    .withOperationName("DescribeDeviceJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeDeviceJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDeviceJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a node.
     * </p>
     *
     * @param describeNodeRequest
     * @return Result of the DescribeNode operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DescribeNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeNodeResponse describeNode(DescribeNodeRequest describeNodeRequest) throws ConflictException,
            ValidationException, AccessDeniedException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeNodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeNodeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNode");

            return clientHandler.execute(new ClientExecutionParams<DescribeNodeRequest, DescribeNodeResponse>()
                    .withOperationName("DescribeNode").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeNodeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeNodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a job to create a camera stream node.
     * </p>
     *
     * @param describeNodeFromTemplateJobRequest
     * @return Result of the DescribeNodeFromTemplateJob operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DescribeNodeFromTemplateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeNodeFromTemplateJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeNodeFromTemplateJobResponse describeNodeFromTemplateJob(
            DescribeNodeFromTemplateJobRequest describeNodeFromTemplateJobRequest) throws ConflictException, ValidationException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeNodeFromTemplateJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeNodeFromTemplateJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeNodeFromTemplateJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeNodeFromTemplateJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNodeFromTemplateJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeNodeFromTemplateJobRequest, DescribeNodeFromTemplateJobResponse>()
                            .withOperationName("DescribeNodeFromTemplateJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeNodeFromTemplateJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeNodeFromTemplateJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a package.
     * </p>
     *
     * @param describePackageRequest
     * @return Result of the DescribePackage operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DescribePackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribePackage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribePackageResponse describePackage(DescribePackageRequest describePackageRequest) throws ConflictException,
            ValidationException, AccessDeniedException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePackageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribePackageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePackageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackage");

            return clientHandler.execute(new ClientExecutionParams<DescribePackageRequest, DescribePackageResponse>()
                    .withOperationName("DescribePackage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describePackageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribePackageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a package import job.
     * </p>
     *
     * @param describePackageImportJobRequest
     * @return Result of the DescribePackageImportJob operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DescribePackageImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribePackageImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribePackageImportJobResponse describePackageImportJob(
            DescribePackageImportJobRequest describePackageImportJobRequest) throws ConflictException, ValidationException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePackageImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribePackageImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePackageImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePackageImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackageImportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribePackageImportJobRequest, DescribePackageImportJobResponse>()
                            .withOperationName("DescribePackageImportJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describePackageImportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribePackageImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a package version.
     * </p>
     *
     * @param describePackageVersionRequest
     * @return Result of the DescribePackageVersion operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.DescribePackageVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribePackageVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribePackageVersionResponse describePackageVersion(DescribePackageVersionRequest describePackageVersionRequest)
            throws ConflictException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePackageVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribePackageVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePackageVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePackageVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackageVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribePackageVersionRequest, DescribePackageVersionResponse>()
                            .withOperationName("DescribePackageVersion").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describePackageVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribePackageVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of application instance dependencies.
     * </p>
     *
     * @param listApplicationInstanceDependenciesRequest
     * @return Result of the ListApplicationInstanceDependencies operation returned by the service.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.ListApplicationInstanceDependencies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListApplicationInstanceDependencies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationInstanceDependenciesResponse listApplicationInstanceDependencies(
            ListApplicationInstanceDependenciesRequest listApplicationInstanceDependenciesRequest) throws AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationInstanceDependenciesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListApplicationInstanceDependenciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApplicationInstanceDependenciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listApplicationInstanceDependenciesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationInstanceDependencies");

            return clientHandler
                    .execute(new ClientExecutionParams<ListApplicationInstanceDependenciesRequest, ListApplicationInstanceDependenciesResponse>()
                            .withOperationName("ListApplicationInstanceDependencies").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listApplicationInstanceDependenciesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListApplicationInstanceDependenciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of application node instances.
     * </p>
     *
     * @param listApplicationInstanceNodeInstancesRequest
     * @return Result of the ListApplicationInstanceNodeInstances operation returned by the service.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.ListApplicationInstanceNodeInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListApplicationInstanceNodeInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationInstanceNodeInstancesResponse listApplicationInstanceNodeInstances(
            ListApplicationInstanceNodeInstancesRequest listApplicationInstanceNodeInstancesRequest)
            throws AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationInstanceNodeInstancesResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListApplicationInstanceNodeInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApplicationInstanceNodeInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listApplicationInstanceNodeInstancesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationInstanceNodeInstances");

            return clientHandler
                    .execute(new ClientExecutionParams<ListApplicationInstanceNodeInstancesRequest, ListApplicationInstanceNodeInstancesResponse>()
                            .withOperationName("ListApplicationInstanceNodeInstances").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listApplicationInstanceNodeInstancesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListApplicationInstanceNodeInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of application instances.
     * </p>
     *
     * @param listApplicationInstancesRequest
     * @return Result of the ListApplicationInstances operation returned by the service.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.ListApplicationInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListApplicationInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationInstancesResponse listApplicationInstances(
            ListApplicationInstancesRequest listApplicationInstancesRequest) throws AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListApplicationInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApplicationInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationInstances");

            return clientHandler
                    .execute(new ClientExecutionParams<ListApplicationInstancesRequest, ListApplicationInstancesResponse>()
                            .withOperationName("ListApplicationInstances").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listApplicationInstancesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListApplicationInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of devices.
     * </p>
     *
     * @param listDevicesRequest
     * @return Result of the ListDevices operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.ListDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListDevices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDevicesResponse listDevices(ListDevicesRequest listDevicesRequest) throws ConflictException, ValidationException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDevicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDevicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDevicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDevices");

            return clientHandler.execute(new ClientExecutionParams<ListDevicesRequest, ListDevicesResponse>()
                    .withOperationName("ListDevices").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listDevicesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDevicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of jobs.
     * </p>
     *
     * @param listDevicesJobsRequest
     * @return Result of the ListDevicesJobs operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.ListDevicesJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListDevicesJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDevicesJobsResponse listDevicesJobs(ListDevicesJobsRequest listDevicesJobsRequest) throws ConflictException,
            ValidationException, AccessDeniedException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDevicesJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDevicesJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDevicesJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDevicesJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDevicesJobs");

            return clientHandler.execute(new ClientExecutionParams<ListDevicesJobsRequest, ListDevicesJobsResponse>()
                    .withOperationName("ListDevicesJobs").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDevicesJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDevicesJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of camera stream node jobs.
     * </p>
     *
     * @param listNodeFromTemplateJobsRequest
     * @return Result of the ListNodeFromTemplateJobs operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.ListNodeFromTemplateJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListNodeFromTemplateJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNodeFromTemplateJobsResponse listNodeFromTemplateJobs(
            ListNodeFromTemplateJobsRequest listNodeFromTemplateJobsRequest) throws ConflictException, ValidationException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNodeFromTemplateJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListNodeFromTemplateJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNodeFromTemplateJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNodeFromTemplateJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNodeFromTemplateJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListNodeFromTemplateJobsRequest, ListNodeFromTemplateJobsResponse>()
                            .withOperationName("ListNodeFromTemplateJobs").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listNodeFromTemplateJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListNodeFromTemplateJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of nodes.
     * </p>
     *
     * @param listNodesRequest
     * @return Result of the ListNodes operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.ListNodes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListNodes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListNodesResponse listNodes(ListNodesRequest listNodesRequest) throws ConflictException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNodesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListNodesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNodesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNodesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNodes");

            return clientHandler.execute(new ClientExecutionParams<ListNodesRequest, ListNodesResponse>()
                    .withOperationName("ListNodes").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listNodesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListNodesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of package import jobs.
     * </p>
     *
     * @param listPackageImportJobsRequest
     * @return Result of the ListPackageImportJobs operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.ListPackageImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListPackageImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPackageImportJobsResponse listPackageImportJobs(ListPackageImportJobsRequest listPackageImportJobsRequest)
            throws ConflictException, ValidationException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPackageImportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPackageImportJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPackageImportJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackageImportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackageImportJobs");

            return clientHandler.execute(new ClientExecutionParams<ListPackageImportJobsRequest, ListPackageImportJobsResponse>()
                    .withOperationName("ListPackageImportJobs").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPackageImportJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPackageImportJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of packages.
     * </p>
     *
     * @param listPackagesRequest
     * @return Result of the ListPackages operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.ListPackages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListPackages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPackagesResponse listPackages(ListPackagesRequest listPackagesRequest) throws ConflictException,
            ValidationException, AccessDeniedException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPackagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPackagesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPackagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackages");

            return clientHandler.execute(new ClientExecutionParams<ListPackagesRequest, ListPackagesResponse>()
                    .withOperationName("ListPackages").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPackagesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPackagesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a device and returns a configuration archive. The configuration archive is a ZIP file that contains a
     * provisioning certificate that is valid for 5 minutes. Name the configuration archive
     * <code>certificates-omni_<i>device-name</i>.zip</code> and transfer it to the device within 5 minutes. Use the
     * included USB storage device and connect it to the USB 3.0 port next to the HDMI output.
     * </p>
     *
     * @param provisionDeviceRequest
     * @return Result of the ProvisionDevice operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ServiceQuotaExceededException
     *         The request would cause a limit to be exceeded.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.ProvisionDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ProvisionDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ProvisionDeviceResponse provisionDevice(ProvisionDeviceRequest provisionDeviceRequest) throws ConflictException,
            ValidationException, AccessDeniedException, ServiceQuotaExceededException, InternalServerException,
            AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ProvisionDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ProvisionDeviceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(provisionDeviceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, provisionDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ProvisionDevice");

            return clientHandler.execute(new ClientExecutionParams<ProvisionDeviceRequest, ProvisionDeviceResponse>()
                    .withOperationName("ProvisionDevice").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(provisionDeviceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ProvisionDeviceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Registers a package version.
     * </p>
     *
     * @param registerPackageVersionRequest
     * @return Result of the RegisterPackageVersion operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.RegisterPackageVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/RegisterPackageVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RegisterPackageVersionResponse registerPackageVersion(RegisterPackageVersionRequest registerPackageVersionRequest)
            throws ConflictException, ValidationException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterPackageVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RegisterPackageVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerPackageVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerPackageVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterPackageVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<RegisterPackageVersionRequest, RegisterPackageVersionResponse>()
                            .withOperationName("RegisterPackageVersion").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(registerPackageVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RegisterPackageVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes an application instance.
     * </p>
     *
     * @param removeApplicationInstanceRequest
     * @return Result of the RemoveApplicationInstance operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.RemoveApplicationInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/RemoveApplicationInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveApplicationInstanceResponse removeApplicationInstance(
            RemoveApplicationInstanceRequest removeApplicationInstanceRequest) throws ConflictException, ValidationException,
            AccessDeniedException, ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveApplicationInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveApplicationInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeApplicationInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeApplicationInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveApplicationInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveApplicationInstanceRequest, RemoveApplicationInstanceResponse>()
                            .withOperationName("RemoveApplicationInstance").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(removeApplicationInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveApplicationInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Signal camera nodes to stop or resume.
     * </p>
     *
     * @param signalApplicationInstanceNodeInstancesRequest
     * @return Result of the SignalApplicationInstanceNodeInstances operation returned by the service.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ServiceQuotaExceededException
     *         The request would cause a limit to be exceeded.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.SignalApplicationInstanceNodeInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/SignalApplicationInstanceNodeInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SignalApplicationInstanceNodeInstancesResponse signalApplicationInstanceNodeInstances(
            SignalApplicationInstanceNodeInstancesRequest signalApplicationInstanceNodeInstancesRequest)
            throws ValidationException, AccessDeniedException, ServiceQuotaExceededException, InternalServerException,
            AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SignalApplicationInstanceNodeInstancesResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, SignalApplicationInstanceNodeInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(signalApplicationInstanceNodeInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                signalApplicationInstanceNodeInstancesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SignalApplicationInstanceNodeInstances");

            return clientHandler
                    .execute(new ClientExecutionParams<SignalApplicationInstanceNodeInstancesRequest, SignalApplicationInstanceNodeInstancesResponse>()
                            .withOperationName("SignalApplicationInstanceNodeInstances").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(signalApplicationInstanceNodeInstancesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SignalApplicationInstanceNodeInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a device's metadata.
     * </p>
     *
     * @param updateDeviceMetadataRequest
     * @return Result of the UpdateDeviceMetadata operation returned by the service.
     * @throws ConflictException
     *         The target resource is in use.
     * @throws ValidationException
     *         The request contains an invalid parameter value.
     * @throws AccessDeniedException
     *         The requestor does not have permission to access the target action or resource.
     * @throws ResourceNotFoundException
     *         The target resource was not found.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PanoramaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PanoramaClient.UpdateDeviceMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/UpdateDeviceMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateDeviceMetadataResponse updateDeviceMetadata(UpdateDeviceMetadataRequest updateDeviceMetadataRequest)
            throws ConflictException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, PanoramaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDeviceMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDeviceMetadataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDeviceMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDeviceMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDeviceMetadata");

            return clientHandler.execute(new ClientExecutionParams<UpdateDeviceMetadataRequest, UpdateDeviceMetadataResponse>()
                    .withOperationName("UpdateDeviceMetadata").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateDeviceMetadataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDeviceMetadataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        PanoramaServiceClientConfigurationBuilder serviceConfigBuilder = new PanoramaServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PanoramaException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final PanoramaServiceClientConfiguration serviceClientConfiguration() {
        return new PanoramaServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
