/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Parameter information for generating a WrappedKeyBlock for key exchange.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WrappedKey implements SdkPojo, Serializable, ToCopyableBuilder<WrappedKey.Builder, WrappedKey> {
    private static final SdkField<String> KEY_CHECK_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyCheckValue").getter(getter(WrappedKey::keyCheckValue)).setter(setter(Builder::keyCheckValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyCheckValue").build()).build();

    private static final SdkField<String> KEY_CHECK_VALUE_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyCheckValueAlgorithm").getter(getter(WrappedKey::keyCheckValueAlgorithmAsString))
            .setter(setter(Builder::keyCheckValueAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyCheckValueAlgorithm").build())
            .build();

    private static final SdkField<String> KEY_MATERIAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyMaterial").getter(getter(WrappedKey::keyMaterial)).setter(setter(Builder::keyMaterial))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyMaterial").build()).build();

    private static final SdkField<String> WRAPPED_KEY_MATERIAL_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WrappedKeyMaterialFormat").getter(getter(WrappedKey::wrappedKeyMaterialFormatAsString))
            .setter(setter(Builder::wrappedKeyMaterialFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WrappedKeyMaterialFormat").build())
            .build();

    private static final SdkField<String> WRAPPING_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WrappingKeyArn").getter(getter(WrappedKey::wrappingKeyArn)).setter(setter(Builder::wrappingKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WrappingKeyArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_CHECK_VALUE_FIELD,
            KEY_CHECK_VALUE_ALGORITHM_FIELD, KEY_MATERIAL_FIELD, WRAPPED_KEY_MATERIAL_FORMAT_FIELD, WRAPPING_KEY_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String keyCheckValue;

    private final String keyCheckValueAlgorithm;

    private final String keyMaterial;

    private final String wrappedKeyMaterialFormat;

    private final String wrappingKeyArn;

    private WrappedKey(BuilderImpl builder) {
        this.keyCheckValue = builder.keyCheckValue;
        this.keyCheckValueAlgorithm = builder.keyCheckValueAlgorithm;
        this.keyMaterial = builder.keyMaterial;
        this.wrappedKeyMaterialFormat = builder.wrappedKeyMaterialFormat;
        this.wrappingKeyArn = builder.wrappingKeyArn;
    }

    /**
     * <p>
     * The key check value (KCV) is used to check if all parties holding a given key have the same key or to detect that
     * a key has changed.
     * </p>
     * 
     * @return The key check value (KCV) is used to check if all parties holding a given key have the same key or to
     *         detect that a key has changed.
     */
    public final String keyCheckValue() {
        return keyCheckValue;
    }

    /**
     * <p>
     * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It is
     * used to validate the key integrity.
     * </p>
     * <p>
     * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked and
     * retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a CMAC
     * algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the encrypted
     * result.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #keyCheckValueAlgorithm} will return {@link KeyCheckValueAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #keyCheckValueAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV).
     *         It is used to validate the key integrity.</p>
     *         <p>
     *         For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
     *         checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
     *         computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
     *         order bytes of the encrypted result.
     * @see KeyCheckValueAlgorithm
     */
    public final KeyCheckValueAlgorithm keyCheckValueAlgorithm() {
        return KeyCheckValueAlgorithm.fromValue(keyCheckValueAlgorithm);
    }

    /**
     * <p>
     * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It is
     * used to validate the key integrity.
     * </p>
     * <p>
     * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked and
     * retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a CMAC
     * algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the encrypted
     * result.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #keyCheckValueAlgorithm} will return {@link KeyCheckValueAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #keyCheckValueAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV).
     *         It is used to validate the key integrity.</p>
     *         <p>
     *         For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
     *         checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
     *         computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
     *         order bytes of the encrypted result.
     * @see KeyCheckValueAlgorithm
     */
    public final String keyCheckValueAlgorithmAsString() {
        return keyCheckValueAlgorithm;
    }

    /**
     * <p>
     * Parameter information for generating a wrapped key using TR-31 or TR-34 skey exchange method.
     * </p>
     * 
     * @return Parameter information for generating a wrapped key using TR-31 or TR-34 skey exchange method.
     */
    public final String keyMaterial() {
        return keyMaterial;
    }

    /**
     * <p>
     * The key block format of a wrapped key.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #wrappedKeyMaterialFormat} will return {@link WrappedKeyMaterialFormat#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #wrappedKeyMaterialFormatAsString}.
     * </p>
     * 
     * @return The key block format of a wrapped key.
     * @see WrappedKeyMaterialFormat
     */
    public final WrappedKeyMaterialFormat wrappedKeyMaterialFormat() {
        return WrappedKeyMaterialFormat.fromValue(wrappedKeyMaterialFormat);
    }

    /**
     * <p>
     * The key block format of a wrapped key.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #wrappedKeyMaterialFormat} will return {@link WrappedKeyMaterialFormat#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #wrappedKeyMaterialFormatAsString}.
     * </p>
     * 
     * @return The key block format of a wrapped key.
     * @see WrappedKeyMaterialFormat
     */
    public final String wrappedKeyMaterialFormatAsString() {
        return wrappedKeyMaterialFormat;
    }

    /**
     * <p>
     * The <code>KeyARN</code> of the wrapped key.
     * </p>
     * 
     * @return The <code>KeyARN</code> of the wrapped key.
     */
    public final String wrappingKeyArn() {
        return wrappingKeyArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(keyCheckValue());
        hashCode = 31 * hashCode + Objects.hashCode(keyCheckValueAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyMaterial());
        hashCode = 31 * hashCode + Objects.hashCode(wrappedKeyMaterialFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(wrappingKeyArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WrappedKey)) {
            return false;
        }
        WrappedKey other = (WrappedKey) obj;
        return Objects.equals(keyCheckValue(), other.keyCheckValue())
                && Objects.equals(keyCheckValueAlgorithmAsString(), other.keyCheckValueAlgorithmAsString())
                && Objects.equals(keyMaterial(), other.keyMaterial())
                && Objects.equals(wrappedKeyMaterialFormatAsString(), other.wrappedKeyMaterialFormatAsString())
                && Objects.equals(wrappingKeyArn(), other.wrappingKeyArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WrappedKey").add("KeyCheckValue", keyCheckValue())
                .add("KeyCheckValueAlgorithm", keyCheckValueAlgorithmAsString())
                .add("KeyMaterial", keyMaterial() == null ? null : "*** Sensitive Data Redacted ***")
                .add("WrappedKeyMaterialFormat", wrappedKeyMaterialFormatAsString()).add("WrappingKeyArn", wrappingKeyArn())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KeyCheckValue":
            return Optional.ofNullable(clazz.cast(keyCheckValue()));
        case "KeyCheckValueAlgorithm":
            return Optional.ofNullable(clazz.cast(keyCheckValueAlgorithmAsString()));
        case "KeyMaterial":
            return Optional.ofNullable(clazz.cast(keyMaterial()));
        case "WrappedKeyMaterialFormat":
            return Optional.ofNullable(clazz.cast(wrappedKeyMaterialFormatAsString()));
        case "WrappingKeyArn":
            return Optional.ofNullable(clazz.cast(wrappingKeyArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WrappedKey, T> g) {
        return obj -> g.apply((WrappedKey) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WrappedKey> {
        /**
         * <p>
         * The key check value (KCV) is used to check if all parties holding a given key have the same key or to detect
         * that a key has changed.
         * </p>
         * 
         * @param keyCheckValue
         *        The key check value (KCV) is used to check if all parties holding a given key have the same key or to
         *        detect that a key has changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyCheckValue(String keyCheckValue);

        /**
         * <p>
         * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It
         * is used to validate the key integrity.
         * </p>
         * <p>
         * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked
         * and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a
         * CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the
         * encrypted result.
         * </p>
         * 
         * @param keyCheckValueAlgorithm
         *        The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value
         *        (KCV). It is used to validate the key integrity.</p>
         *        <p>
         *        For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
         *        checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
         *        computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
         *        order bytes of the encrypted result.
         * @see KeyCheckValueAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyCheckValueAlgorithm
         */
        Builder keyCheckValueAlgorithm(String keyCheckValueAlgorithm);

        /**
         * <p>
         * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It
         * is used to validate the key integrity.
         * </p>
         * <p>
         * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked
         * and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a
         * CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the
         * encrypted result.
         * </p>
         * 
         * @param keyCheckValueAlgorithm
         *        The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value
         *        (KCV). It is used to validate the key integrity.</p>
         *        <p>
         *        For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
         *        checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
         *        computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
         *        order bytes of the encrypted result.
         * @see KeyCheckValueAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyCheckValueAlgorithm
         */
        Builder keyCheckValueAlgorithm(KeyCheckValueAlgorithm keyCheckValueAlgorithm);

        /**
         * <p>
         * Parameter information for generating a wrapped key using TR-31 or TR-34 skey exchange method.
         * </p>
         * 
         * @param keyMaterial
         *        Parameter information for generating a wrapped key using TR-31 or TR-34 skey exchange method.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyMaterial(String keyMaterial);

        /**
         * <p>
         * The key block format of a wrapped key.
         * </p>
         * 
         * @param wrappedKeyMaterialFormat
         *        The key block format of a wrapped key.
         * @see WrappedKeyMaterialFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WrappedKeyMaterialFormat
         */
        Builder wrappedKeyMaterialFormat(String wrappedKeyMaterialFormat);

        /**
         * <p>
         * The key block format of a wrapped key.
         * </p>
         * 
         * @param wrappedKeyMaterialFormat
         *        The key block format of a wrapped key.
         * @see WrappedKeyMaterialFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WrappedKeyMaterialFormat
         */
        Builder wrappedKeyMaterialFormat(WrappedKeyMaterialFormat wrappedKeyMaterialFormat);

        /**
         * <p>
         * The <code>KeyARN</code> of the wrapped key.
         * </p>
         * 
         * @param wrappingKeyArn
         *        The <code>KeyARN</code> of the wrapped key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder wrappingKeyArn(String wrappingKeyArn);
    }

    static final class BuilderImpl implements Builder {
        private String keyCheckValue;

        private String keyCheckValueAlgorithm;

        private String keyMaterial;

        private String wrappedKeyMaterialFormat;

        private String wrappingKeyArn;

        private BuilderImpl() {
        }

        private BuilderImpl(WrappedKey model) {
            keyCheckValue(model.keyCheckValue);
            keyCheckValueAlgorithm(model.keyCheckValueAlgorithm);
            keyMaterial(model.keyMaterial);
            wrappedKeyMaterialFormat(model.wrappedKeyMaterialFormat);
            wrappingKeyArn(model.wrappingKeyArn);
        }

        public final String getKeyCheckValue() {
            return keyCheckValue;
        }

        public final void setKeyCheckValue(String keyCheckValue) {
            this.keyCheckValue = keyCheckValue;
        }

        @Override
        public final Builder keyCheckValue(String keyCheckValue) {
            this.keyCheckValue = keyCheckValue;
            return this;
        }

        public final String getKeyCheckValueAlgorithm() {
            return keyCheckValueAlgorithm;
        }

        public final void setKeyCheckValueAlgorithm(String keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm = keyCheckValueAlgorithm;
        }

        @Override
        public final Builder keyCheckValueAlgorithm(String keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm = keyCheckValueAlgorithm;
            return this;
        }

        @Override
        public final Builder keyCheckValueAlgorithm(KeyCheckValueAlgorithm keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm(keyCheckValueAlgorithm == null ? null : keyCheckValueAlgorithm.toString());
            return this;
        }

        public final String getKeyMaterial() {
            return keyMaterial;
        }

        public final void setKeyMaterial(String keyMaterial) {
            this.keyMaterial = keyMaterial;
        }

        @Override
        public final Builder keyMaterial(String keyMaterial) {
            this.keyMaterial = keyMaterial;
            return this;
        }

        public final String getWrappedKeyMaterialFormat() {
            return wrappedKeyMaterialFormat;
        }

        public final void setWrappedKeyMaterialFormat(String wrappedKeyMaterialFormat) {
            this.wrappedKeyMaterialFormat = wrappedKeyMaterialFormat;
        }

        @Override
        public final Builder wrappedKeyMaterialFormat(String wrappedKeyMaterialFormat) {
            this.wrappedKeyMaterialFormat = wrappedKeyMaterialFormat;
            return this;
        }

        @Override
        public final Builder wrappedKeyMaterialFormat(WrappedKeyMaterialFormat wrappedKeyMaterialFormat) {
            this.wrappedKeyMaterialFormat(wrappedKeyMaterialFormat == null ? null : wrappedKeyMaterialFormat.toString());
            return this;
        }

        public final String getWrappingKeyArn() {
            return wrappingKeyArn;
        }

        public final void setWrappingKeyArn(String wrappingKeyArn) {
            this.wrappingKeyArn = wrappingKeyArn;
        }

        @Override
        public final Builder wrappingKeyArn(String wrappingKeyArn) {
            this.wrappingKeyArn = wrappingKeyArn;
            return this;
        }

        @Override
        public WrappedKey build() {
            return new WrappedKey(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
