/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pipes.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The logging configuration settings for the pipe.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PipeLogConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<PipeLogConfiguration.Builder, PipeLogConfiguration> {
    private static final SdkField<CloudwatchLogsLogDestination> CLOUDWATCH_LOGS_LOG_DESTINATION_FIELD = SdkField
            .<CloudwatchLogsLogDestination> builder(MarshallingType.SDK_POJO)
            .memberName("CloudwatchLogsLogDestination")
            .getter(getter(PipeLogConfiguration::cloudwatchLogsLogDestination))
            .setter(setter(Builder::cloudwatchLogsLogDestination))
            .constructor(CloudwatchLogsLogDestination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudwatchLogsLogDestination")
                    .build()).build();

    private static final SdkField<FirehoseLogDestination> FIREHOSE_LOG_DESTINATION_FIELD = SdkField
            .<FirehoseLogDestination> builder(MarshallingType.SDK_POJO).memberName("FirehoseLogDestination")
            .getter(getter(PipeLogConfiguration::firehoseLogDestination)).setter(setter(Builder::firehoseLogDestination))
            .constructor(FirehoseLogDestination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirehoseLogDestination").build())
            .build();

    private static final SdkField<List<String>> INCLUDE_EXECUTION_DATA_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IncludeExecutionData")
            .getter(getter(PipeLogConfiguration::includeExecutionDataAsStrings))
            .setter(setter(Builder::includeExecutionDataWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeExecutionData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Level")
            .getter(getter(PipeLogConfiguration::levelAsString)).setter(setter(Builder::level))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Level").build()).build();

    private static final SdkField<S3LogDestination> S3_LOG_DESTINATION_FIELD = SdkField
            .<S3LogDestination> builder(MarshallingType.SDK_POJO).memberName("S3LogDestination")
            .getter(getter(PipeLogConfiguration::s3LogDestination)).setter(setter(Builder::s3LogDestination))
            .constructor(S3LogDestination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3LogDestination").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CLOUDWATCH_LOGS_LOG_DESTINATION_FIELD, FIREHOSE_LOG_DESTINATION_FIELD, INCLUDE_EXECUTION_DATA_FIELD, LEVEL_FIELD,
            S3_LOG_DESTINATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final CloudwatchLogsLogDestination cloudwatchLogsLogDestination;

    private final FirehoseLogDestination firehoseLogDestination;

    private final List<String> includeExecutionData;

    private final String level;

    private final S3LogDestination s3LogDestination;

    private PipeLogConfiguration(BuilderImpl builder) {
        this.cloudwatchLogsLogDestination = builder.cloudwatchLogsLogDestination;
        this.firehoseLogDestination = builder.firehoseLogDestination;
        this.includeExecutionData = builder.includeExecutionData;
        this.level = builder.level;
        this.s3LogDestination = builder.s3LogDestination;
    }

    /**
     * <p>
     * The Amazon CloudWatch Logs logging configuration settings for the pipe.
     * </p>
     * 
     * @return The Amazon CloudWatch Logs logging configuration settings for the pipe.
     */
    public final CloudwatchLogsLogDestination cloudwatchLogsLogDestination() {
        return cloudwatchLogsLogDestination;
    }

    /**
     * <p>
     * The Amazon Kinesis Data Firehose logging configuration settings for the pipe.
     * </p>
     * 
     * @return The Amazon Kinesis Data Firehose logging configuration settings for the pipe.
     */
    public final FirehoseLogDestination firehoseLogDestination() {
        return firehoseLogDestination;
    }

    /**
     * <p>
     * Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
     * <code>awsResponse</code> fields) is included in the log messages for this pipe.
     * </p>
     * <p>
     * This applies to all log destinations for the pipe.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
     * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludeExecutionData} method.
     * </p>
     * 
     * @return Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
     *         <code>awsResponse</code> fields) is included in the log messages for this pipe.</p>
     *         <p>
     *         This applies to all log destinations for the pipe.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
     *         >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
     */
    public final List<IncludeExecutionDataOption> includeExecutionData() {
        return IncludeExecutionDataCopier.copyStringToEnum(includeExecutionData);
    }

    /**
     * For responses, this returns true if the service returned a value for the IncludeExecutionData property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIncludeExecutionData() {
        return includeExecutionData != null && !(includeExecutionData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
     * <code>awsResponse</code> fields) is included in the log messages for this pipe.
     * </p>
     * <p>
     * This applies to all log destinations for the pipe.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
     * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludeExecutionData} method.
     * </p>
     * 
     * @return Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
     *         <code>awsResponse</code> fields) is included in the log messages for this pipe.</p>
     *         <p>
     *         This applies to all log destinations for the pipe.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
     *         >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
     */
    public final List<String> includeExecutionDataAsStrings() {
        return includeExecutionData;
    }

    /**
     * <p>
     * The level of logging detail to include. This applies to all log destinations for the pipe.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #level} will return
     * {@link LogLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #levelAsString}.
     * </p>
     * 
     * @return The level of logging detail to include. This applies to all log destinations for the pipe.
     * @see LogLevel
     */
    public final LogLevel level() {
        return LogLevel.fromValue(level);
    }

    /**
     * <p>
     * The level of logging detail to include. This applies to all log destinations for the pipe.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #level} will return
     * {@link LogLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #levelAsString}.
     * </p>
     * 
     * @return The level of logging detail to include. This applies to all log destinations for the pipe.
     * @see LogLevel
     */
    public final String levelAsString() {
        return level;
    }

    /**
     * <p>
     * The Amazon S3 logging configuration settings for the pipe.
     * </p>
     * 
     * @return The Amazon S3 logging configuration settings for the pipe.
     */
    public final S3LogDestination s3LogDestination() {
        return s3LogDestination;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cloudwatchLogsLogDestination());
        hashCode = 31 * hashCode + Objects.hashCode(firehoseLogDestination());
        hashCode = 31 * hashCode + Objects.hashCode(hasIncludeExecutionData() ? includeExecutionDataAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(levelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(s3LogDestination());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PipeLogConfiguration)) {
            return false;
        }
        PipeLogConfiguration other = (PipeLogConfiguration) obj;
        return Objects.equals(cloudwatchLogsLogDestination(), other.cloudwatchLogsLogDestination())
                && Objects.equals(firehoseLogDestination(), other.firehoseLogDestination())
                && hasIncludeExecutionData() == other.hasIncludeExecutionData()
                && Objects.equals(includeExecutionDataAsStrings(), other.includeExecutionDataAsStrings())
                && Objects.equals(levelAsString(), other.levelAsString())
                && Objects.equals(s3LogDestination(), other.s3LogDestination());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PipeLogConfiguration").add("CloudwatchLogsLogDestination", cloudwatchLogsLogDestination())
                .add("FirehoseLogDestination", firehoseLogDestination())
                .add("IncludeExecutionData", hasIncludeExecutionData() ? includeExecutionDataAsStrings() : null)
                .add("Level", levelAsString()).add("S3LogDestination", s3LogDestination()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CloudwatchLogsLogDestination":
            return Optional.ofNullable(clazz.cast(cloudwatchLogsLogDestination()));
        case "FirehoseLogDestination":
            return Optional.ofNullable(clazz.cast(firehoseLogDestination()));
        case "IncludeExecutionData":
            return Optional.ofNullable(clazz.cast(includeExecutionDataAsStrings()));
        case "Level":
            return Optional.ofNullable(clazz.cast(levelAsString()));
        case "S3LogDestination":
            return Optional.ofNullable(clazz.cast(s3LogDestination()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PipeLogConfiguration, T> g) {
        return obj -> g.apply((PipeLogConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PipeLogConfiguration> {
        /**
         * <p>
         * The Amazon CloudWatch Logs logging configuration settings for the pipe.
         * </p>
         * 
         * @param cloudwatchLogsLogDestination
         *        The Amazon CloudWatch Logs logging configuration settings for the pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudwatchLogsLogDestination(CloudwatchLogsLogDestination cloudwatchLogsLogDestination);

        /**
         * <p>
         * The Amazon CloudWatch Logs logging configuration settings for the pipe.
         * </p>
         * This is a convenience method that creates an instance of the {@link CloudwatchLogsLogDestination.Builder}
         * avoiding the need to create one manually via {@link CloudwatchLogsLogDestination#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CloudwatchLogsLogDestination.Builder#build()} is called
         * immediately and its result is passed to {@link #cloudwatchLogsLogDestination(CloudwatchLogsLogDestination)}.
         * 
         * @param cloudwatchLogsLogDestination
         *        a consumer that will call methods on {@link CloudwatchLogsLogDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudwatchLogsLogDestination(CloudwatchLogsLogDestination)
         */
        default Builder cloudwatchLogsLogDestination(Consumer<CloudwatchLogsLogDestination.Builder> cloudwatchLogsLogDestination) {
            return cloudwatchLogsLogDestination(CloudwatchLogsLogDestination.builder()
                    .applyMutation(cloudwatchLogsLogDestination).build());
        }

        /**
         * <p>
         * The Amazon Kinesis Data Firehose logging configuration settings for the pipe.
         * </p>
         * 
         * @param firehoseLogDestination
         *        The Amazon Kinesis Data Firehose logging configuration settings for the pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firehoseLogDestination(FirehoseLogDestination firehoseLogDestination);

        /**
         * <p>
         * The Amazon Kinesis Data Firehose logging configuration settings for the pipe.
         * </p>
         * This is a convenience method that creates an instance of the {@link FirehoseLogDestination.Builder} avoiding
         * the need to create one manually via {@link FirehoseLogDestination#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FirehoseLogDestination.Builder#build()} is called immediately and
         * its result is passed to {@link #firehoseLogDestination(FirehoseLogDestination)}.
         * 
         * @param firehoseLogDestination
         *        a consumer that will call methods on {@link FirehoseLogDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #firehoseLogDestination(FirehoseLogDestination)
         */
        default Builder firehoseLogDestination(Consumer<FirehoseLogDestination.Builder> firehoseLogDestination) {
            return firehoseLogDestination(FirehoseLogDestination.builder().applyMutation(firehoseLogDestination).build());
        }

        /**
         * <p>
         * Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
         * <code>awsResponse</code> fields) is included in the log messages for this pipe.
         * </p>
         * <p>
         * This applies to all log destinations for the pipe.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * 
         * @param includeExecutionData
         *        Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
         *        <code>awsResponse</code> fields) is included in the log messages for this pipe.</p>
         *        <p>
         *        This applies to all log destinations for the pipe.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         *        >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeExecutionDataWithStrings(Collection<String> includeExecutionData);

        /**
         * <p>
         * Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
         * <code>awsResponse</code> fields) is included in the log messages for this pipe.
         * </p>
         * <p>
         * This applies to all log destinations for the pipe.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * 
         * @param includeExecutionData
         *        Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
         *        <code>awsResponse</code> fields) is included in the log messages for this pipe.</p>
         *        <p>
         *        This applies to all log destinations for the pipe.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         *        >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeExecutionDataWithStrings(String... includeExecutionData);

        /**
         * <p>
         * Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
         * <code>awsResponse</code> fields) is included in the log messages for this pipe.
         * </p>
         * <p>
         * This applies to all log destinations for the pipe.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * 
         * @param includeExecutionData
         *        Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
         *        <code>awsResponse</code> fields) is included in the log messages for this pipe.</p>
         *        <p>
         *        This applies to all log destinations for the pipe.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         *        >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeExecutionData(Collection<IncludeExecutionDataOption> includeExecutionData);

        /**
         * <p>
         * Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
         * <code>awsResponse</code> fields) is included in the log messages for this pipe.
         * </p>
         * <p>
         * This applies to all log destinations for the pipe.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * 
         * @param includeExecutionData
         *        Whether the execution data (specifically, the <code>payload</code>, <code>awsRequest</code>, and
         *        <code>awsResponse</code> fields) is included in the log messages for this pipe.</p>
         *        <p>
         *        This applies to all log destinations for the pipe.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         *        >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeExecutionData(IncludeExecutionDataOption... includeExecutionData);

        /**
         * <p>
         * The level of logging detail to include. This applies to all log destinations for the pipe.
         * </p>
         * 
         * @param level
         *        The level of logging detail to include. This applies to all log destinations for the pipe.
         * @see LogLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogLevel
         */
        Builder level(String level);

        /**
         * <p>
         * The level of logging detail to include. This applies to all log destinations for the pipe.
         * </p>
         * 
         * @param level
         *        The level of logging detail to include. This applies to all log destinations for the pipe.
         * @see LogLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogLevel
         */
        Builder level(LogLevel level);

        /**
         * <p>
         * The Amazon S3 logging configuration settings for the pipe.
         * </p>
         * 
         * @param s3LogDestination
         *        The Amazon S3 logging configuration settings for the pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3LogDestination(S3LogDestination s3LogDestination);

        /**
         * <p>
         * The Amazon S3 logging configuration settings for the pipe.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3LogDestination.Builder} avoiding the
         * need to create one manually via {@link S3LogDestination#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3LogDestination.Builder#build()} is called immediately and its
         * result is passed to {@link #s3LogDestination(S3LogDestination)}.
         * 
         * @param s3LogDestination
         *        a consumer that will call methods on {@link S3LogDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3LogDestination(S3LogDestination)
         */
        default Builder s3LogDestination(Consumer<S3LogDestination.Builder> s3LogDestination) {
            return s3LogDestination(S3LogDestination.builder().applyMutation(s3LogDestination).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private CloudwatchLogsLogDestination cloudwatchLogsLogDestination;

        private FirehoseLogDestination firehoseLogDestination;

        private List<String> includeExecutionData = DefaultSdkAutoConstructList.getInstance();

        private String level;

        private S3LogDestination s3LogDestination;

        private BuilderImpl() {
        }

        private BuilderImpl(PipeLogConfiguration model) {
            cloudwatchLogsLogDestination(model.cloudwatchLogsLogDestination);
            firehoseLogDestination(model.firehoseLogDestination);
            includeExecutionDataWithStrings(model.includeExecutionData);
            level(model.level);
            s3LogDestination(model.s3LogDestination);
        }

        public final CloudwatchLogsLogDestination.Builder getCloudwatchLogsLogDestination() {
            return cloudwatchLogsLogDestination != null ? cloudwatchLogsLogDestination.toBuilder() : null;
        }

        public final void setCloudwatchLogsLogDestination(CloudwatchLogsLogDestination.BuilderImpl cloudwatchLogsLogDestination) {
            this.cloudwatchLogsLogDestination = cloudwatchLogsLogDestination != null ? cloudwatchLogsLogDestination.build()
                    : null;
        }

        @Override
        public final Builder cloudwatchLogsLogDestination(CloudwatchLogsLogDestination cloudwatchLogsLogDestination) {
            this.cloudwatchLogsLogDestination = cloudwatchLogsLogDestination;
            return this;
        }

        public final FirehoseLogDestination.Builder getFirehoseLogDestination() {
            return firehoseLogDestination != null ? firehoseLogDestination.toBuilder() : null;
        }

        public final void setFirehoseLogDestination(FirehoseLogDestination.BuilderImpl firehoseLogDestination) {
            this.firehoseLogDestination = firehoseLogDestination != null ? firehoseLogDestination.build() : null;
        }

        @Override
        public final Builder firehoseLogDestination(FirehoseLogDestination firehoseLogDestination) {
            this.firehoseLogDestination = firehoseLogDestination;
            return this;
        }

        public final Collection<String> getIncludeExecutionData() {
            if (includeExecutionData instanceof SdkAutoConstructList) {
                return null;
            }
            return includeExecutionData;
        }

        public final void setIncludeExecutionData(Collection<String> includeExecutionData) {
            this.includeExecutionData = IncludeExecutionDataCopier.copy(includeExecutionData);
        }

        @Override
        public final Builder includeExecutionDataWithStrings(Collection<String> includeExecutionData) {
            this.includeExecutionData = IncludeExecutionDataCopier.copy(includeExecutionData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder includeExecutionDataWithStrings(String... includeExecutionData) {
            includeExecutionDataWithStrings(Arrays.asList(includeExecutionData));
            return this;
        }

        @Override
        public final Builder includeExecutionData(Collection<IncludeExecutionDataOption> includeExecutionData) {
            this.includeExecutionData = IncludeExecutionDataCopier.copyEnumToString(includeExecutionData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder includeExecutionData(IncludeExecutionDataOption... includeExecutionData) {
            includeExecutionData(Arrays.asList(includeExecutionData));
            return this;
        }

        public final String getLevel() {
            return level;
        }

        public final void setLevel(String level) {
            this.level = level;
        }

        @Override
        public final Builder level(String level) {
            this.level = level;
            return this;
        }

        @Override
        public final Builder level(LogLevel level) {
            this.level(level == null ? null : level.toString());
            return this;
        }

        public final S3LogDestination.Builder getS3LogDestination() {
            return s3LogDestination != null ? s3LogDestination.toBuilder() : null;
        }

        public final void setS3LogDestination(S3LogDestination.BuilderImpl s3LogDestination) {
            this.s3LogDestination = s3LogDestination != null ? s3LogDestination.build() : null;
        }

        @Override
        public final Builder s3LogDestination(S3LogDestination s3LogDestination) {
            this.s3LogDestination = s3LogDestination;
            return this;
        }

        @Override
        public PipeLogConfiguration build() {
            return new PipeLogConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
