/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourcegroupstaggingapi;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.resourcegroupstaggingapi.internal.ResourceGroupsTaggingApiServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.ConcurrentModificationException;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.ConstraintViolationException;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.DescribeReportCreationRequest;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.DescribeReportCreationResponse;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.GetComplianceSummaryRequest;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.GetComplianceSummaryResponse;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.GetResourcesRequest;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.GetResourcesResponse;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.GetTagKeysRequest;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.GetTagKeysResponse;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.GetTagValuesRequest;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.GetTagValuesResponse;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.InternalServiceException;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.InvalidParameterException;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.PaginationTokenExpiredException;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.ResourceGroupsTaggingApiException;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.StartReportCreationRequest;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.StartReportCreationResponse;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.TagResourcesRequest;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.TagResourcesResponse;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.ThrottledException;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.UntagResourcesRequest;
import software.amazon.awssdk.services.resourcegroupstaggingapi.model.UntagResourcesResponse;
import software.amazon.awssdk.services.resourcegroupstaggingapi.transform.DescribeReportCreationRequestMarshaller;
import software.amazon.awssdk.services.resourcegroupstaggingapi.transform.GetComplianceSummaryRequestMarshaller;
import software.amazon.awssdk.services.resourcegroupstaggingapi.transform.GetResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroupstaggingapi.transform.GetTagKeysRequestMarshaller;
import software.amazon.awssdk.services.resourcegroupstaggingapi.transform.GetTagValuesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroupstaggingapi.transform.StartReportCreationRequestMarshaller;
import software.amazon.awssdk.services.resourcegroupstaggingapi.transform.TagResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroupstaggingapi.transform.UntagResourcesRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ResourceGroupsTaggingApiClient}.
 *
 * @see ResourceGroupsTaggingApiClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultResourceGroupsTaggingApiClient implements ResourceGroupsTaggingApiClient {
    private static final Logger log = Logger.loggerFor(DefaultResourceGroupsTaggingApiClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultResourceGroupsTaggingApiClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Describes the status of the <code>StartReportCreation</code> operation.
     * </p>
     * <p>
     * You can call this operation only from the organization's management account and from the us-east-1 Region.
     * </p>
     *
     * @param describeReportCreationRequest
     * @return Result of the DescribeReportCreation operation returned by the service.
     * @throws ConstraintViolationException
     *         The request was denied because performing this operation violates a constraint. </p>
     *         <p>
     *         Some of the reasons in the following list might not apply to this specific operation.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You must meet the prerequisites for using tag policies. For information, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies-prereqs.html"
     *         >Prerequisites and Permissions for Using Tag Policies</a> in the <i>Organizations User Guide.</i>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You must enable the tag policies service principal (<code>tagpolicies.tag.amazonaws.com</code>) to
     *         integrate with Organizations For information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAWSServiceAccess.html"
     *         >EnableAWSServiceAccess</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You must have a tag policy attached to the organization root, an OU, or an account.
     *         </p>
     *         </li>
     * @throws InternalServiceException
     *         The request processing failed because of an unknown error, exception, or failure. You can retry the
     *         request.
     * @throws InvalidParameterException
     *         This error indicates one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A parameter is missing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A malformed string was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An out-of-range value was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The target ID is invalid, unsupported, or doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can't access the Amazon S3 bucket for report storage. For more information, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies-prereqs.html#bucket-policies-org-report"
     *         >Additional Requirements for Organization-wide Tag Compliance Reports</a> in the <i>Organizations User
     *         Guide.</i>
     *         </p>
     *         </li>
     * @throws ThrottledException
     *         The request was denied to limit the frequency of submitted requests.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsTaggingApiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsTaggingApiClient.DescribeReportCreation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resourcegroupstaggingapi-2017-01-26/DescribeReportCreation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeReportCreationResponse describeReportCreation(DescribeReportCreationRequest describeReportCreationRequest)
            throws ConstraintViolationException, InternalServiceException, InvalidParameterException, ThrottledException,
            AwsServiceException, SdkClientException, ResourceGroupsTaggingApiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeReportCreationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeReportCreationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeReportCreationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeReportCreationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups Tagging API");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReportCreation");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeReportCreationRequest, DescribeReportCreationResponse>()
                            .withOperationName("DescribeReportCreation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeReportCreationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeReportCreationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a table that shows counts of resources that are noncompliant with their tag policies.
     * </p>
     * <p>
     * For more information on tag policies, see <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies.html">Tag
     * Policies</a> in the <i>Organizations User Guide.</i>
     * </p>
     * <p>
     * You can call this operation only from the organization's management account and from the us-east-1 Region.
     * </p>
     * <p>
     * This operation supports pagination, where the response can be sent in multiple pages. You should check the
     * <code>PaginationToken</code> response parameter to determine if there are additional results available to return.
     * Repeat the query, passing the <code>PaginationToken</code> response parameter value as an input to the next
     * request until you recieve a <code>null</code> value. A null value for <code>PaginationToken</code> indicates that
     * there are no more results waiting to be returned.
     * </p>
     *
     * @param getComplianceSummaryRequest
     * @return Result of the GetComplianceSummary operation returned by the service.
     * @throws ConstraintViolationException
     *         The request was denied because performing this operation violates a constraint. </p>
     *         <p>
     *         Some of the reasons in the following list might not apply to this specific operation.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You must meet the prerequisites for using tag policies. For information, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies-prereqs.html"
     *         >Prerequisites and Permissions for Using Tag Policies</a> in the <i>Organizations User Guide.</i>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You must enable the tag policies service principal (<code>tagpolicies.tag.amazonaws.com</code>) to
     *         integrate with Organizations For information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAWSServiceAccess.html"
     *         >EnableAWSServiceAccess</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You must have a tag policy attached to the organization root, an OU, or an account.
     *         </p>
     *         </li>
     * @throws InternalServiceException
     *         The request processing failed because of an unknown error, exception, or failure. You can retry the
     *         request.
     * @throws InvalidParameterException
     *         This error indicates one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A parameter is missing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A malformed string was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An out-of-range value was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The target ID is invalid, unsupported, or doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can't access the Amazon S3 bucket for report storage. For more information, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies-prereqs.html#bucket-policies-org-report"
     *         >Additional Requirements for Organization-wide Tag Compliance Reports</a> in the <i>Organizations User
     *         Guide.</i>
     *         </p>
     *         </li>
     * @throws ThrottledException
     *         The request was denied to limit the frequency of submitted requests.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsTaggingApiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsTaggingApiClient.GetComplianceSummary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resourcegroupstaggingapi-2017-01-26/GetComplianceSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetComplianceSummaryResponse getComplianceSummary(GetComplianceSummaryRequest getComplianceSummaryRequest)
            throws ConstraintViolationException, InternalServiceException, InvalidParameterException, ThrottledException,
            AwsServiceException, SdkClientException, ResourceGroupsTaggingApiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetComplianceSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetComplianceSummaryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getComplianceSummaryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComplianceSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups Tagging API");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComplianceSummary");

            return clientHandler.execute(new ClientExecutionParams<GetComplianceSummaryRequest, GetComplianceSummaryResponse>()
                    .withOperationName("GetComplianceSummary").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getComplianceSummaryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetComplianceSummaryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns all the tagged or previously tagged resources that are located in the specified Amazon Web Services
     * Region for the account.
     * </p>
     * <p>
     * Depending on what information you want returned, you can also specify the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Filters</i> that specify what tags and resource types you want returned. The response includes all tags that
     * are associated with the requested resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * Information about compliance with the account's effective tag policy. For more information on tag policies, see
     * <a href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies.html">Tag
     * Policies</a> in the <i>Organizations User Guide.</i>
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation supports pagination, where the response can be sent in multiple pages. You should check the
     * <code>PaginationToken</code> response parameter to determine if there are additional results available to return.
     * Repeat the query, passing the <code>PaginationToken</code> response parameter value as an input to the next
     * request until you recieve a <code>null</code> value. A null value for <code>PaginationToken</code> indicates that
     * there are no more results waiting to be returned.
     * </p>
     *
     * @param getResourcesRequest
     * @return Result of the GetResources operation returned by the service.
     * @throws InvalidParameterException
     *         This error indicates one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A parameter is missing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A malformed string was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An out-of-range value was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The target ID is invalid, unsupported, or doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can't access the Amazon S3 bucket for report storage. For more information, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies-prereqs.html#bucket-policies-org-report"
     *         >Additional Requirements for Organization-wide Tag Compliance Reports</a> in the <i>Organizations User
     *         Guide.</i>
     *         </p>
     *         </li>
     * @throws ThrottledException
     *         The request was denied to limit the frequency of submitted requests.
     * @throws InternalServiceException
     *         The request processing failed because of an unknown error, exception, or failure. You can retry the
     *         request.
     * @throws PaginationTokenExpiredException
     *         A <code>PaginationToken</code> is valid for a maximum of 15 minutes. Your request was denied because the
     *         specified <code>PaginationToken</code> has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsTaggingApiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsTaggingApiClient.GetResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resourcegroupstaggingapi-2017-01-26/GetResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourcesResponse getResources(GetResourcesRequest getResourcesRequest) throws InvalidParameterException,
            ThrottledException, InternalServiceException, PaginationTokenExpiredException, AwsServiceException,
            SdkClientException, ResourceGroupsTaggingApiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups Tagging API");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResources");

            return clientHandler.execute(new ClientExecutionParams<GetResourcesRequest, GetResourcesResponse>()
                    .withOperationName("GetResources").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns all tag keys currently in use in the specified Amazon Web Services Region for the calling account.
     * </p>
     * <p>
     * This operation supports pagination, where the response can be sent in multiple pages. You should check the
     * <code>PaginationToken</code> response parameter to determine if there are additional results available to return.
     * Repeat the query, passing the <code>PaginationToken</code> response parameter value as an input to the next
     * request until you recieve a <code>null</code> value. A null value for <code>PaginationToken</code> indicates that
     * there are no more results waiting to be returned.
     * </p>
     *
     * @param getTagKeysRequest
     * @return Result of the GetTagKeys operation returned by the service.
     * @throws InvalidParameterException
     *         This error indicates one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A parameter is missing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A malformed string was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An out-of-range value was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The target ID is invalid, unsupported, or doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can't access the Amazon S3 bucket for report storage. For more information, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies-prereqs.html#bucket-policies-org-report"
     *         >Additional Requirements for Organization-wide Tag Compliance Reports</a> in the <i>Organizations User
     *         Guide.</i>
     *         </p>
     *         </li>
     * @throws ThrottledException
     *         The request was denied to limit the frequency of submitted requests.
     * @throws InternalServiceException
     *         The request processing failed because of an unknown error, exception, or failure. You can retry the
     *         request.
     * @throws PaginationTokenExpiredException
     *         A <code>PaginationToken</code> is valid for a maximum of 15 minutes. Your request was denied because the
     *         specified <code>PaginationToken</code> has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsTaggingApiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsTaggingApiClient.GetTagKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resourcegroupstaggingapi-2017-01-26/GetTagKeys"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetTagKeysResponse getTagKeys(GetTagKeysRequest getTagKeysRequest) throws InvalidParameterException,
            ThrottledException, InternalServiceException, PaginationTokenExpiredException, AwsServiceException,
            SdkClientException, ResourceGroupsTaggingApiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTagKeysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTagKeysResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTagKeysRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTagKeysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups Tagging API");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTagKeys");

            return clientHandler.execute(new ClientExecutionParams<GetTagKeysRequest, GetTagKeysResponse>()
                    .withOperationName("GetTagKeys").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getTagKeysRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTagKeysRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns all tag values for the specified key that are used in the specified Amazon Web Services Region for the
     * calling account.
     * </p>
     * <p>
     * This operation supports pagination, where the response can be sent in multiple pages. You should check the
     * <code>PaginationToken</code> response parameter to determine if there are additional results available to return.
     * Repeat the query, passing the <code>PaginationToken</code> response parameter value as an input to the next
     * request until you recieve a <code>null</code> value. A null value for <code>PaginationToken</code> indicates that
     * there are no more results waiting to be returned.
     * </p>
     *
     * @param getTagValuesRequest
     * @return Result of the GetTagValues operation returned by the service.
     * @throws InvalidParameterException
     *         This error indicates one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A parameter is missing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A malformed string was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An out-of-range value was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The target ID is invalid, unsupported, or doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can't access the Amazon S3 bucket for report storage. For more information, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies-prereqs.html#bucket-policies-org-report"
     *         >Additional Requirements for Organization-wide Tag Compliance Reports</a> in the <i>Organizations User
     *         Guide.</i>
     *         </p>
     *         </li>
     * @throws ThrottledException
     *         The request was denied to limit the frequency of submitted requests.
     * @throws InternalServiceException
     *         The request processing failed because of an unknown error, exception, or failure. You can retry the
     *         request.
     * @throws PaginationTokenExpiredException
     *         A <code>PaginationToken</code> is valid for a maximum of 15 minutes. Your request was denied because the
     *         specified <code>PaginationToken</code> has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsTaggingApiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsTaggingApiClient.GetTagValues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resourcegroupstaggingapi-2017-01-26/GetTagValues"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetTagValuesResponse getTagValues(GetTagValuesRequest getTagValuesRequest) throws InvalidParameterException,
            ThrottledException, InternalServiceException, PaginationTokenExpiredException, AwsServiceException,
            SdkClientException, ResourceGroupsTaggingApiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTagValuesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTagValuesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTagValuesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTagValuesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups Tagging API");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTagValues");

            return clientHandler.execute(new ClientExecutionParams<GetTagValuesRequest, GetTagValuesResponse>()
                    .withOperationName("GetTagValues").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTagValuesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTagValuesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Generates a report that lists all tagged resources in the accounts across your organization and tells whether
     * each resource is compliant with the effective tag policy. Compliance data is refreshed daily. The report is
     * generated asynchronously.
     * </p>
     * <p>
     * The generated report is saved to the following location:
     * </p>
     * <p>
     * <code>s3://example-bucket/AwsTagPolicies/o-exampleorgid/YYYY-MM-ddTHH:mm:ssZ/report.csv</code>
     * </p>
     * <p>
     * You can call this operation only from the organization's management account and from the us-east-1 Region.
     * </p>
     *
     * @param startReportCreationRequest
     * @return Result of the StartReportCreation operation returned by the service.
     * @throws ConcurrentModificationException
     *         The target of the operation is currently being modified by a different request. Try again later.
     * @throws ConstraintViolationException
     *         The request was denied because performing this operation violates a constraint. </p>
     *         <p>
     *         Some of the reasons in the following list might not apply to this specific operation.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You must meet the prerequisites for using tag policies. For information, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies-prereqs.html"
     *         >Prerequisites and Permissions for Using Tag Policies</a> in the <i>Organizations User Guide.</i>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You must enable the tag policies service principal (<code>tagpolicies.tag.amazonaws.com</code>) to
     *         integrate with Organizations For information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAWSServiceAccess.html"
     *         >EnableAWSServiceAccess</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You must have a tag policy attached to the organization root, an OU, or an account.
     *         </p>
     *         </li>
     * @throws InternalServiceException
     *         The request processing failed because of an unknown error, exception, or failure. You can retry the
     *         request.
     * @throws InvalidParameterException
     *         This error indicates one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A parameter is missing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A malformed string was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An out-of-range value was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The target ID is invalid, unsupported, or doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can't access the Amazon S3 bucket for report storage. For more information, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies-prereqs.html#bucket-policies-org-report"
     *         >Additional Requirements for Organization-wide Tag Compliance Reports</a> in the <i>Organizations User
     *         Guide.</i>
     *         </p>
     *         </li>
     * @throws ThrottledException
     *         The request was denied to limit the frequency of submitted requests.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsTaggingApiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsTaggingApiClient.StartReportCreation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resourcegroupstaggingapi-2017-01-26/StartReportCreation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartReportCreationResponse startReportCreation(StartReportCreationRequest startReportCreationRequest)
            throws ConcurrentModificationException, ConstraintViolationException, InternalServiceException,
            InvalidParameterException, ThrottledException, AwsServiceException, SdkClientException,
            ResourceGroupsTaggingApiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartReportCreationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartReportCreationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startReportCreationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startReportCreationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups Tagging API");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartReportCreation");

            return clientHandler.execute(new ClientExecutionParams<StartReportCreationRequest, StartReportCreationResponse>()
                    .withOperationName("StartReportCreation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startReportCreationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartReportCreationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Applies one or more tags to the specified resources. Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Not all resources can have tags. For a list of services with resources that support tagging using this operation,
     * see <a
     * href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/supported-services.html">Services
     * that support the Resource Groups Tagging API</a>. If the resource doesn't yet support this operation, the
     * resource's service might support tagging using its own API operations. For more information, refer to the
     * documentation for that service.
     * </p>
     * </li>
     * <li>
     * <p>
     * Each resource can have up to 50 tags. For other limits, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html#tag-conventions">Tag Naming and Usage
     * Conventions</a> in the <i>Amazon Web Services General Reference.</i>
     * </p>
     * </li>
     * <li>
     * <p>
     * You can only tag resources that are located in the specified Amazon Web Services Region for the Amazon Web
     * Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * To add tags to a resource, you need the necessary permissions for the service that the resource belongs to as
     * well as permissions for adding tags. For more information, see the documentation for each service.
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * Do not store personally identifiable information (PII) or other confidential or sensitive information in tags. We
     * use tags to provide you with billing and administration services. Tags are not intended to be used for private or
     * sensitive data.
     * </p>
     * </important>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * In addition to the <code>tag:TagResources</code> permission required by this operation, you must also have the
     * tagging permission defined by the service that created the resource. For example, to tag an Amazon EC2 instance
     * using the <code>TagResources</code> operation, you must have both of the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tag:TagResource</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:CreateTags</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param tagResourcesRequest
     * @return Result of the TagResources operation returned by the service.
     * @throws InvalidParameterException
     *         This error indicates one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A parameter is missing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A malformed string was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An out-of-range value was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The target ID is invalid, unsupported, or doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can't access the Amazon S3 bucket for report storage. For more information, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies-prereqs.html#bucket-policies-org-report"
     *         >Additional Requirements for Organization-wide Tag Compliance Reports</a> in the <i>Organizations User
     *         Guide.</i>
     *         </p>
     *         </li>
     * @throws ThrottledException
     *         The request was denied to limit the frequency of submitted requests.
     * @throws InternalServiceException
     *         The request processing failed because of an unknown error, exception, or failure. You can retry the
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsTaggingApiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsTaggingApiClient.TagResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resourcegroupstaggingapi-2017-01-26/TagResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourcesResponse tagResources(TagResourcesRequest tagResourcesRequest) throws InvalidParameterException,
            ThrottledException, InternalServiceException, AwsServiceException, SdkClientException,
            ResourceGroupsTaggingApiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourcesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups Tagging API");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResources");

            return clientHandler.execute(new ClientExecutionParams<TagResourcesRequest, TagResourcesResponse>()
                    .withOperationName("TagResources").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(tagResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resources. When you specify a tag key, the action removes both that
     * key and its associated value. The operation succeeds even if you attempt to remove tags from a resource that were
     * already removed. Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To remove tags from a resource, you need the necessary permissions for the service that the resource belongs to
     * as well as permissions for removing tags. For more information, see the documentation for the service whose
     * resource you want to untag.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can only tag resources that are located in the specified Amazon Web Services Region for the calling Amazon
     * Web Services account.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * In addition to the <code>tag:UntagResources</code> permission required by this operation, you must also have the
     * remove tags permission defined by the service that created the resource. For example, to remove the tags from an
     * Amazon EC2 instance using the <code>UntagResources</code> operation, you must have both of the following
     * permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tag:UntagResource</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DeleteTags</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param untagResourcesRequest
     * @return Result of the UntagResources operation returned by the service.
     * @throws InvalidParameterException
     *         This error indicates one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A parameter is missing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A malformed string was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An out-of-range value was supplied for the request parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The target ID is invalid, unsupported, or doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can't access the Amazon S3 bucket for report storage. For more information, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_tag-policies-prereqs.html#bucket-policies-org-report"
     *         >Additional Requirements for Organization-wide Tag Compliance Reports</a> in the <i>Organizations User
     *         Guide.</i>
     *         </p>
     *         </li>
     * @throws ThrottledException
     *         The request was denied to limit the frequency of submitted requests.
     * @throws InternalServiceException
     *         The request processing failed because of an unknown error, exception, or failure. You can retry the
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsTaggingApiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsTaggingApiClient.UntagResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resourcegroupstaggingapi-2017-01-26/UntagResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourcesResponse untagResources(UntagResourcesRequest untagResourcesRequest) throws InvalidParameterException,
            ThrottledException, InternalServiceException, AwsServiceException, SdkClientException,
            ResourceGroupsTaggingApiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourcesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups Tagging API");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResources");

            return clientHandler.execute(new ClientExecutionParams<UntagResourcesRequest, UntagResourcesResponse>()
                    .withOperationName("UntagResources").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        ResourceGroupsTaggingApiServiceClientConfigurationBuilder serviceConfigBuilder = new ResourceGroupsTaggingApiServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ResourceGroupsTaggingApiException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConstraintViolationException")
                                .exceptionBuilderSupplier(ConstraintViolationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottledException")
                                .exceptionBuilderSupplier(ThrottledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PaginationTokenExpiredException")
                                .exceptionBuilderSupplier(PaginationTokenExpiredException::builder).httpStatusCode(400).build());
    }

    @Override
    public final ResourceGroupsTaggingApiServiceClientConfiguration serviceClientConfiguration() {
        return new ResourceGroupsTaggingApiServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
