/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourcegroupstaggingapi.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the errors that are returned for each failed resource. This information can include
 * <code>InternalServiceException</code> and <code>InvalidParameterException</code> errors. It can also include any
 * valid error code returned by the Amazon Web Services service that hosts the resource that the ARN key represents.
 * </p>
 * <p>
 * The following are common error codes that you might receive from other Amazon Web Services services:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>InternalServiceException</b> – This can mean that the Resource Groups Tagging API didn't receive a response from
 * another Amazon Web Services service. It can also mean that the resource type in the request is not supported by the
 * Resource Groups Tagging API. In these cases, it's safe to retry the request and then call <a
 * href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html">GetResources</a>
 * to verify the changes.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>AccessDeniedException</b> – This can mean that you need permission to call the tagging operations in the Amazon
 * Web Services service that contains the resource. For example, to use the Resource Groups Tagging API to tag a Amazon
 * CloudWatch alarm resource, you need permission to call both <a
 * href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_TagResources.html">
 * <code>TagResources</code> </a> <i>and</i> <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_TagResource.html">
 * <code>TagResource</code> </a> in the CloudWatch API.
 * </p>
 * </li>
 * </ul>
 * <p>
 * For more information on errors that are generated from other Amazon Web Services services, see the documentation for
 * that service.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FailureInfo implements SdkPojo, Serializable, ToCopyableBuilder<FailureInfo.Builder, FailureInfo> {
    private static final SdkField<Integer> STATUS_CODE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("StatusCode").getter(getter(FailureInfo::statusCode)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusCode").build()).build();

    private static final SdkField<String> ERROR_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorCode").getter(getter(FailureInfo::errorCodeAsString)).setter(setter(Builder::errorCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorCode").build()).build();

    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorMessage").getter(getter(FailureInfo::errorMessage)).setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorMessage").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_CODE_FIELD,
            ERROR_CODE_FIELD, ERROR_MESSAGE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer statusCode;

    private final String errorCode;

    private final String errorMessage;

    private FailureInfo(BuilderImpl builder) {
        this.statusCode = builder.statusCode;
        this.errorCode = builder.errorCode;
        this.errorMessage = builder.errorMessage;
    }

    /**
     * <p>
     * The HTTP status code of the common error.
     * </p>
     * 
     * @return The HTTP status code of the common error.
     */
    public final Integer statusCode() {
        return statusCode;
    }

    /**
     * <p>
     * The code of the common error. Valid values include <code>InternalServiceException</code>,
     * <code>InvalidParameterException</code>, and any valid error code returned by the Amazon Web Services service that
     * hosts the resource that you want to tag.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorCode} will
     * return {@link ErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorCodeAsString}.
     * </p>
     * 
     * @return The code of the common error. Valid values include <code>InternalServiceException</code>,
     *         <code>InvalidParameterException</code>, and any valid error code returned by the Amazon Web Services
     *         service that hosts the resource that you want to tag.
     * @see ErrorCode
     */
    public final ErrorCode errorCode() {
        return ErrorCode.fromValue(errorCode);
    }

    /**
     * <p>
     * The code of the common error. Valid values include <code>InternalServiceException</code>,
     * <code>InvalidParameterException</code>, and any valid error code returned by the Amazon Web Services service that
     * hosts the resource that you want to tag.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorCode} will
     * return {@link ErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorCodeAsString}.
     * </p>
     * 
     * @return The code of the common error. Valid values include <code>InternalServiceException</code>,
     *         <code>InvalidParameterException</code>, and any valid error code returned by the Amazon Web Services
     *         service that hosts the resource that you want to tag.
     * @see ErrorCode
     */
    public final String errorCodeAsString() {
        return errorCode;
    }

    /**
     * <p>
     * The message of the common error.
     * </p>
     * 
     * @return The message of the common error.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusCode());
        hashCode = 31 * hashCode + Objects.hashCode(errorCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FailureInfo)) {
            return false;
        }
        FailureInfo other = (FailureInfo) obj;
        return Objects.equals(statusCode(), other.statusCode()) && Objects.equals(errorCodeAsString(), other.errorCodeAsString())
                && Objects.equals(errorMessage(), other.errorMessage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FailureInfo").add("StatusCode", statusCode()).add("ErrorCode", errorCodeAsString())
                .add("ErrorMessage", errorMessage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StatusCode":
            return Optional.ofNullable(clazz.cast(statusCode()));
        case "ErrorCode":
            return Optional.ofNullable(clazz.cast(errorCodeAsString()));
        case "ErrorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FailureInfo, T> g) {
        return obj -> g.apply((FailureInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FailureInfo> {
        /**
         * <p>
         * The HTTP status code of the common error.
         * </p>
         * 
         * @param statusCode
         *        The HTTP status code of the common error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusCode(Integer statusCode);

        /**
         * <p>
         * The code of the common error. Valid values include <code>InternalServiceException</code>,
         * <code>InvalidParameterException</code>, and any valid error code returned by the Amazon Web Services service
         * that hosts the resource that you want to tag.
         * </p>
         * 
         * @param errorCode
         *        The code of the common error. Valid values include <code>InternalServiceException</code>,
         *        <code>InvalidParameterException</code>, and any valid error code returned by the Amazon Web Services
         *        service that hosts the resource that you want to tag.
         * @see ErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ErrorCode
         */
        Builder errorCode(String errorCode);

        /**
         * <p>
         * The code of the common error. Valid values include <code>InternalServiceException</code>,
         * <code>InvalidParameterException</code>, and any valid error code returned by the Amazon Web Services service
         * that hosts the resource that you want to tag.
         * </p>
         * 
         * @param errorCode
         *        The code of the common error. Valid values include <code>InternalServiceException</code>,
         *        <code>InvalidParameterException</code>, and any valid error code returned by the Amazon Web Services
         *        service that hosts the resource that you want to tag.
         * @see ErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ErrorCode
         */
        Builder errorCode(ErrorCode errorCode);

        /**
         * <p>
         * The message of the common error.
         * </p>
         * 
         * @param errorMessage
         *        The message of the common error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);
    }

    static final class BuilderImpl implements Builder {
        private Integer statusCode;

        private String errorCode;

        private String errorMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(FailureInfo model) {
            statusCode(model.statusCode);
            errorCode(model.errorCode);
            errorMessage(model.errorMessage);
        }

        public final Integer getStatusCode() {
            return statusCode;
        }

        public final void setStatusCode(Integer statusCode) {
            this.statusCode = statusCode;
        }

        @Override
        public final Builder statusCode(Integer statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        public final String getErrorCode() {
            return errorCode;
        }

        public final void setErrorCode(String errorCode) {
            this.errorCode = errorCode;
        }

        @Override
        public final Builder errorCode(String errorCode) {
            this.errorCode = errorCode;
            return this;
        }

        @Override
        public final Builder errorCode(ErrorCode errorCode) {
            this.errorCode(errorCode == null ? null : errorCode.toString());
            return this;
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        @Override
        public FailureInfo build() {
            return new FailureInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
