/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourcegroupstaggingapi.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information that shows whether a resource is compliant with the effective tag policy, including details on any
 * noncompliant tag keys.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComplianceDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<ComplianceDetails.Builder, ComplianceDetails> {
    private static final SdkField<List<String>> NONCOMPLIANT_KEYS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NoncompliantKeys")
            .getter(getter(ComplianceDetails::noncompliantKeys))
            .setter(setter(Builder::noncompliantKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NoncompliantKeys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> KEYS_WITH_NONCOMPLIANT_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("KeysWithNoncompliantValues")
            .getter(getter(ComplianceDetails::keysWithNoncompliantValues))
            .setter(setter(Builder::keysWithNoncompliantValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeysWithNoncompliantValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> COMPLIANCE_STATUS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ComplianceStatus").getter(getter(ComplianceDetails::complianceStatus))
            .setter(setter(Builder::complianceStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComplianceStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NONCOMPLIANT_KEYS_FIELD,
            KEYS_WITH_NONCOMPLIANT_VALUES_FIELD, COMPLIANCE_STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> noncompliantKeys;

    private final List<String> keysWithNoncompliantValues;

    private final Boolean complianceStatus;

    private ComplianceDetails(BuilderImpl builder) {
        this.noncompliantKeys = builder.noncompliantKeys;
        this.keysWithNoncompliantValues = builder.keysWithNoncompliantValues;
        this.complianceStatus = builder.complianceStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the NoncompliantKeys property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNoncompliantKeys() {
        return noncompliantKeys != null && !(noncompliantKeys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * These tag keys on the resource are noncompliant with the effective tag policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNoncompliantKeys} method.
     * </p>
     * 
     * @return These tag keys on the resource are noncompliant with the effective tag policy.
     */
    public final List<String> noncompliantKeys() {
        return noncompliantKeys;
    }

    /**
     * For responses, this returns true if the service returned a value for the KeysWithNoncompliantValues property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasKeysWithNoncompliantValues() {
        return keysWithNoncompliantValues != null && !(keysWithNoncompliantValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * These are keys defined in the effective policy that are on the resource with either incorrect case treatment or
     * noncompliant values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasKeysWithNoncompliantValues} method.
     * </p>
     * 
     * @return These are keys defined in the effective policy that are on the resource with either incorrect case
     *         treatment or noncompliant values.
     */
    public final List<String> keysWithNoncompliantValues() {
        return keysWithNoncompliantValues;
    }

    /**
     * <p>
     * Whether a resource is compliant with the effective tag policy.
     * </p>
     * 
     * @return Whether a resource is compliant with the effective tag policy.
     */
    public final Boolean complianceStatus() {
        return complianceStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasNoncompliantKeys() ? noncompliantKeys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasKeysWithNoncompliantValues() ? keysWithNoncompliantValues() : null);
        hashCode = 31 * hashCode + Objects.hashCode(complianceStatus());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComplianceDetails)) {
            return false;
        }
        ComplianceDetails other = (ComplianceDetails) obj;
        return hasNoncompliantKeys() == other.hasNoncompliantKeys()
                && Objects.equals(noncompliantKeys(), other.noncompliantKeys())
                && hasKeysWithNoncompliantValues() == other.hasKeysWithNoncompliantValues()
                && Objects.equals(keysWithNoncompliantValues(), other.keysWithNoncompliantValues())
                && Objects.equals(complianceStatus(), other.complianceStatus());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ComplianceDetails").add("NoncompliantKeys", hasNoncompliantKeys() ? noncompliantKeys() : null)
                .add("KeysWithNoncompliantValues", hasKeysWithNoncompliantValues() ? keysWithNoncompliantValues() : null)
                .add("ComplianceStatus", complianceStatus()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NoncompliantKeys":
            return Optional.ofNullable(clazz.cast(noncompliantKeys()));
        case "KeysWithNoncompliantValues":
            return Optional.ofNullable(clazz.cast(keysWithNoncompliantValues()));
        case "ComplianceStatus":
            return Optional.ofNullable(clazz.cast(complianceStatus()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ComplianceDetails, T> g) {
        return obj -> g.apply((ComplianceDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComplianceDetails> {
        /**
         * <p>
         * These tag keys on the resource are noncompliant with the effective tag policy.
         * </p>
         * 
         * @param noncompliantKeys
         *        These tag keys on the resource are noncompliant with the effective tag policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder noncompliantKeys(Collection<String> noncompliantKeys);

        /**
         * <p>
         * These tag keys on the resource are noncompliant with the effective tag policy.
         * </p>
         * 
         * @param noncompliantKeys
         *        These tag keys on the resource are noncompliant with the effective tag policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder noncompliantKeys(String... noncompliantKeys);

        /**
         * <p>
         * These are keys defined in the effective policy that are on the resource with either incorrect case treatment
         * or noncompliant values.
         * </p>
         * 
         * @param keysWithNoncompliantValues
         *        These are keys defined in the effective policy that are on the resource with either incorrect case
         *        treatment or noncompliant values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keysWithNoncompliantValues(Collection<String> keysWithNoncompliantValues);

        /**
         * <p>
         * These are keys defined in the effective policy that are on the resource with either incorrect case treatment
         * or noncompliant values.
         * </p>
         * 
         * @param keysWithNoncompliantValues
         *        These are keys defined in the effective policy that are on the resource with either incorrect case
         *        treatment or noncompliant values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keysWithNoncompliantValues(String... keysWithNoncompliantValues);

        /**
         * <p>
         * Whether a resource is compliant with the effective tag policy.
         * </p>
         * 
         * @param complianceStatus
         *        Whether a resource is compliant with the effective tag policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder complianceStatus(Boolean complianceStatus);
    }

    static final class BuilderImpl implements Builder {
        private List<String> noncompliantKeys = DefaultSdkAutoConstructList.getInstance();

        private List<String> keysWithNoncompliantValues = DefaultSdkAutoConstructList.getInstance();

        private Boolean complianceStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(ComplianceDetails model) {
            noncompliantKeys(model.noncompliantKeys);
            keysWithNoncompliantValues(model.keysWithNoncompliantValues);
            complianceStatus(model.complianceStatus);
        }

        public final Collection<String> getNoncompliantKeys() {
            if (noncompliantKeys instanceof SdkAutoConstructList) {
                return null;
            }
            return noncompliantKeys;
        }

        public final void setNoncompliantKeys(Collection<String> noncompliantKeys) {
            this.noncompliantKeys = TagKeyListCopier.copy(noncompliantKeys);
        }

        @Override
        public final Builder noncompliantKeys(Collection<String> noncompliantKeys) {
            this.noncompliantKeys = TagKeyListCopier.copy(noncompliantKeys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder noncompliantKeys(String... noncompliantKeys) {
            noncompliantKeys(Arrays.asList(noncompliantKeys));
            return this;
        }

        public final Collection<String> getKeysWithNoncompliantValues() {
            if (keysWithNoncompliantValues instanceof SdkAutoConstructList) {
                return null;
            }
            return keysWithNoncompliantValues;
        }

        public final void setKeysWithNoncompliantValues(Collection<String> keysWithNoncompliantValues) {
            this.keysWithNoncompliantValues = TagKeyListCopier.copy(keysWithNoncompliantValues);
        }

        @Override
        public final Builder keysWithNoncompliantValues(Collection<String> keysWithNoncompliantValues) {
            this.keysWithNoncompliantValues = TagKeyListCopier.copy(keysWithNoncompliantValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keysWithNoncompliantValues(String... keysWithNoncompliantValues) {
            keysWithNoncompliantValues(Arrays.asList(keysWithNoncompliantValues));
            return this;
        }

        public final Boolean getComplianceStatus() {
            return complianceStatus;
        }

        public final void setComplianceStatus(Boolean complianceStatus) {
            this.complianceStatus = complianceStatus;
        }

        @Override
        public final Builder complianceStatus(Boolean complianceStatus) {
            this.complianceStatus = complianceStatus;
            return this;
        }

        @Override
        public ComplianceDetails build() {
            return new ComplianceDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
