/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53domains.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The TransferDomain request includes the following elements.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TransferDomainRequest extends Route53DomainsRequest implements
        ToCopyableBuilder<TransferDomainRequest.Builder, TransferDomainRequest> {
    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DomainName").getter(getter(TransferDomainRequest::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName").build()).build();

    private static final SdkField<String> IDN_LANG_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IdnLangCode").getter(getter(TransferDomainRequest::idnLangCode)).setter(setter(Builder::idnLangCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdnLangCode").build()).build();

    private static final SdkField<Integer> DURATION_IN_YEARS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("DurationInYears").getter(getter(TransferDomainRequest::durationInYears))
            .setter(setter(Builder::durationInYears))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DurationInYears").build()).build();

    private static final SdkField<List<Nameserver>> NAMESERVERS_FIELD = SdkField
            .<List<Nameserver>> builder(MarshallingType.LIST)
            .memberName("Nameservers")
            .getter(getter(TransferDomainRequest::nameservers))
            .setter(setter(Builder::nameservers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Nameservers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Nameserver> builder(MarshallingType.SDK_POJO)
                                            .constructor(Nameserver::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> AUTH_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthCode").getter(getter(TransferDomainRequest::authCode)).setter(setter(Builder::authCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthCode").build()).build();

    private static final SdkField<Boolean> AUTO_RENEW_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoRenew").getter(getter(TransferDomainRequest::autoRenew)).setter(setter(Builder::autoRenew))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoRenew").build()).build();

    private static final SdkField<ContactDetail> ADMIN_CONTACT_FIELD = SdkField.<ContactDetail> builder(MarshallingType.SDK_POJO)
            .memberName("AdminContact").getter(getter(TransferDomainRequest::adminContact)).setter(setter(Builder::adminContact))
            .constructor(ContactDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdminContact").build()).build();

    private static final SdkField<ContactDetail> REGISTRANT_CONTACT_FIELD = SdkField
            .<ContactDetail> builder(MarshallingType.SDK_POJO).memberName("RegistrantContact")
            .getter(getter(TransferDomainRequest::registrantContact)).setter(setter(Builder::registrantContact))
            .constructor(ContactDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RegistrantContact").build()).build();

    private static final SdkField<ContactDetail> TECH_CONTACT_FIELD = SdkField.<ContactDetail> builder(MarshallingType.SDK_POJO)
            .memberName("TechContact").getter(getter(TransferDomainRequest::techContact)).setter(setter(Builder::techContact))
            .constructor(ContactDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TechContact").build()).build();

    private static final SdkField<Boolean> PRIVACY_PROTECT_ADMIN_CONTACT_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PrivacyProtectAdminContact")
            .getter(getter(TransferDomainRequest::privacyProtectAdminContact))
            .setter(setter(Builder::privacyProtectAdminContact))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivacyProtectAdminContact").build())
            .build();

    private static final SdkField<Boolean> PRIVACY_PROTECT_REGISTRANT_CONTACT_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PrivacyProtectRegistrantContact")
            .getter(getter(TransferDomainRequest::privacyProtectRegistrantContact))
            .setter(setter(Builder::privacyProtectRegistrantContact))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivacyProtectRegistrantContact")
                    .build()).build();

    private static final SdkField<Boolean> PRIVACY_PROTECT_TECH_CONTACT_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("PrivacyProtectTechContact")
            .getter(getter(TransferDomainRequest::privacyProtectTechContact)).setter(setter(Builder::privacyProtectTechContact))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivacyProtectTechContact").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_NAME_FIELD,
            IDN_LANG_CODE_FIELD, DURATION_IN_YEARS_FIELD, NAMESERVERS_FIELD, AUTH_CODE_FIELD, AUTO_RENEW_FIELD,
            ADMIN_CONTACT_FIELD, REGISTRANT_CONTACT_FIELD, TECH_CONTACT_FIELD, PRIVACY_PROTECT_ADMIN_CONTACT_FIELD,
            PRIVACY_PROTECT_REGISTRANT_CONTACT_FIELD, PRIVACY_PROTECT_TECH_CONTACT_FIELD));

    private final String domainName;

    private final String idnLangCode;

    private final Integer durationInYears;

    private final List<Nameserver> nameservers;

    private final String authCode;

    private final Boolean autoRenew;

    private final ContactDetail adminContact;

    private final ContactDetail registrantContact;

    private final ContactDetail techContact;

    private final Boolean privacyProtectAdminContact;

    private final Boolean privacyProtectRegistrantContact;

    private final Boolean privacyProtectTechContact;

    private TransferDomainRequest(BuilderImpl builder) {
        super(builder);
        this.domainName = builder.domainName;
        this.idnLangCode = builder.idnLangCode;
        this.durationInYears = builder.durationInYears;
        this.nameservers = builder.nameservers;
        this.authCode = builder.authCode;
        this.autoRenew = builder.autoRenew;
        this.adminContact = builder.adminContact;
        this.registrantContact = builder.registrantContact;
        this.techContact = builder.techContact;
        this.privacyProtectAdminContact = builder.privacyProtectAdminContact;
        this.privacyProtectRegistrantContact = builder.privacyProtectRegistrantContact;
        this.privacyProtectTechContact = builder.privacyProtectTechContact;
    }

    /**
     * <p>
     * The name of the domain that you want to transfer to Route 53. The top-level domain (TLD), such as .com, must be a
     * TLD that Route 53 supports. For a list of supported TLDs, see <a
     * href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/registrar-tld-list.html">Domains that You Can
     * Register with Amazon Route 53</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * The domain name can contain only the following characters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Letters a through z. Domain names are not case sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * Numbers 0 through 9.
     * </p>
     * </li>
     * <li>
     * <p>
     * Hyphen (-). You can't specify a hyphen at the beginning or end of a label.
     * </p>
     * </li>
     * <li>
     * <p>
     * Period (.) to separate the labels in the name, such as the <code>.</code> in <code>example.com</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of the domain that you want to transfer to Route 53. The top-level domain (TLD), such as .com,
     *         must be a TLD that Route 53 supports. For a list of supported TLDs, see <a
     *         href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/registrar-tld-list.html">Domains that You
     *         Can Register with Amazon Route 53</a> in the <i>Amazon Route 53 Developer Guide</i>.</p>
     *         <p>
     *         The domain name can contain only the following characters:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Letters a through z. Domain names are not case sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Numbers 0 through 9.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Hyphen (-). You can't specify a hyphen at the beginning or end of a label.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Period (.) to separate the labels in the name, such as the <code>.</code> in <code>example.com</code>.
     *         </p>
     *         </li>
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * Reserved for future use.
     * </p>
     * 
     * @return Reserved for future use.
     */
    public final String idnLangCode() {
        return idnLangCode;
    }

    /**
     * <p>
     * The number of years that you want to register the domain for. Domains are registered for a minimum of one year.
     * The maximum period depends on the top-level domain.
     * </p>
     * <p>
     * Default: 1
     * </p>
     * 
     * @return The number of years that you want to register the domain for. Domains are registered for a minimum of one
     *         year. The maximum period depends on the top-level domain.</p>
     *         <p>
     *         Default: 1
     */
    public final Integer durationInYears() {
        return durationInYears;
    }

    /**
     * For responses, this returns true if the service returned a value for the Nameservers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNameservers() {
        return nameservers != null && !(nameservers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains details for the host and glue IP addresses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNameservers} method.
     * </p>
     * 
     * @return Contains details for the host and glue IP addresses.
     */
    public final List<Nameserver> nameservers() {
        return nameservers;
    }

    /**
     * <p>
     * The authorization code for the domain. You get this value from the current registrar.
     * </p>
     * 
     * @return The authorization code for the domain. You get this value from the current registrar.
     */
    public final String authCode() {
        return authCode;
    }

    /**
     * <p>
     * Indicates whether the domain will be automatically renewed (true) or not (false). Autorenewal only takes effect
     * after the account is charged.
     * </p>
     * <p>
     * Default: true
     * </p>
     * 
     * @return Indicates whether the domain will be automatically renewed (true) or not (false). Autorenewal only takes
     *         effect after the account is charged.</p>
     *         <p>
     *         Default: true
     */
    public final Boolean autoRenew() {
        return autoRenew;
    }

    /**
     * <p>
     * Provides detailed contact information.
     * </p>
     * 
     * @return Provides detailed contact information.
     */
    public final ContactDetail adminContact() {
        return adminContact;
    }

    /**
     * <p>
     * Provides detailed contact information.
     * </p>
     * 
     * @return Provides detailed contact information.
     */
    public final ContactDetail registrantContact() {
        return registrantContact;
    }

    /**
     * <p>
     * Provides detailed contact information.
     * </p>
     * 
     * @return Provides detailed contact information.
     */
    public final ContactDetail techContact() {
        return techContact;
    }

    /**
     * <p>
     * Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>, WHOIS
     * ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and .org domains) or
     * for our registrar associate, Gandi (for all other TLDs). If you specify <code>false</code>, WHOIS queries return
     * the information that you entered for the admin contact.
     * </p>
     * <note>
     * <p>
     * You must specify the same privacy setting for the administrative, registrant, and technical contacts.
     * </p>
     * </note>
     * <p>
     * Default: <code>true</code>
     * </p>
     * 
     * @return Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>,
     *         WHOIS ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and .org
     *         domains) or for our registrar associate, Gandi (for all other TLDs). If you specify <code>false</code>,
     *         WHOIS queries return the information that you entered for the admin contact.</p> <note>
     *         <p>
     *         You must specify the same privacy setting for the administrative, registrant, and technical contacts.
     *         </p>
     *         </note>
     *         <p>
     *         Default: <code>true</code>
     */
    public final Boolean privacyProtectAdminContact() {
        return privacyProtectAdminContact;
    }

    /**
     * <p>
     * Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>, WHOIS
     * ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and .org domains) or
     * for our registrar associate, Gandi (for all other TLDs). If you specify <code>false</code>, WHOIS queries return
     * the information that you entered for the registrant contact (domain owner).
     * </p>
     * <note>
     * <p>
     * You must specify the same privacy setting for the administrative, registrant, and technical contacts.
     * </p>
     * </note>
     * <p>
     * Default: <code>true</code>
     * </p>
     * 
     * @return Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>,
     *         WHOIS ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and .org
     *         domains) or for our registrar associate, Gandi (for all other TLDs). If you specify <code>false</code>,
     *         WHOIS queries return the information that you entered for the registrant contact (domain owner).</p>
     *         <note>
     *         <p>
     *         You must specify the same privacy setting for the administrative, registrant, and technical contacts.
     *         </p>
     *         </note>
     *         <p>
     *         Default: <code>true</code>
     */
    public final Boolean privacyProtectRegistrantContact() {
        return privacyProtectRegistrantContact;
    }

    /**
     * <p>
     * Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>, WHOIS
     * ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and .org domains) or
     * for our registrar associate, Gandi (for all other TLDs). If you specify <code>false</code>, WHOIS queries return
     * the information that you entered for the technical contact.
     * </p>
     * <note>
     * <p>
     * You must specify the same privacy setting for the administrative, registrant, and technical contacts.
     * </p>
     * </note>
     * <p>
     * Default: <code>true</code>
     * </p>
     * 
     * @return Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>,
     *         WHOIS ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and .org
     *         domains) or for our registrar associate, Gandi (for all other TLDs). If you specify <code>false</code>,
     *         WHOIS queries return the information that you entered for the technical contact.</p> <note>
     *         <p>
     *         You must specify the same privacy setting for the administrative, registrant, and technical contacts.
     *         </p>
     *         </note>
     *         <p>
     *         Default: <code>true</code>
     */
    public final Boolean privacyProtectTechContact() {
        return privacyProtectTechContact;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(idnLangCode());
        hashCode = 31 * hashCode + Objects.hashCode(durationInYears());
        hashCode = 31 * hashCode + Objects.hashCode(hasNameservers() ? nameservers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(authCode());
        hashCode = 31 * hashCode + Objects.hashCode(autoRenew());
        hashCode = 31 * hashCode + Objects.hashCode(adminContact());
        hashCode = 31 * hashCode + Objects.hashCode(registrantContact());
        hashCode = 31 * hashCode + Objects.hashCode(techContact());
        hashCode = 31 * hashCode + Objects.hashCode(privacyProtectAdminContact());
        hashCode = 31 * hashCode + Objects.hashCode(privacyProtectRegistrantContact());
        hashCode = 31 * hashCode + Objects.hashCode(privacyProtectTechContact());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TransferDomainRequest)) {
            return false;
        }
        TransferDomainRequest other = (TransferDomainRequest) obj;
        return Objects.equals(domainName(), other.domainName()) && Objects.equals(idnLangCode(), other.idnLangCode())
                && Objects.equals(durationInYears(), other.durationInYears()) && hasNameservers() == other.hasNameservers()
                && Objects.equals(nameservers(), other.nameservers()) && Objects.equals(authCode(), other.authCode())
                && Objects.equals(autoRenew(), other.autoRenew()) && Objects.equals(adminContact(), other.adminContact())
                && Objects.equals(registrantContact(), other.registrantContact())
                && Objects.equals(techContact(), other.techContact())
                && Objects.equals(privacyProtectAdminContact(), other.privacyProtectAdminContact())
                && Objects.equals(privacyProtectRegistrantContact(), other.privacyProtectRegistrantContact())
                && Objects.equals(privacyProtectTechContact(), other.privacyProtectTechContact());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TransferDomainRequest").add("DomainName", domainName()).add("IdnLangCode", idnLangCode())
                .add("DurationInYears", durationInYears()).add("Nameservers", hasNameservers() ? nameservers() : null)
                .add("AuthCode", authCode() == null ? null : "*** Sensitive Data Redacted ***").add("AutoRenew", autoRenew())
                .add("AdminContact", adminContact() == null ? null : "*** Sensitive Data Redacted ***")
                .add("RegistrantContact", registrantContact() == null ? null : "*** Sensitive Data Redacted ***")
                .add("TechContact", techContact() == null ? null : "*** Sensitive Data Redacted ***")
                .add("PrivacyProtectAdminContact", privacyProtectAdminContact())
                .add("PrivacyProtectRegistrantContact", privacyProtectRegistrantContact())
                .add("PrivacyProtectTechContact", privacyProtectTechContact()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "IdnLangCode":
            return Optional.ofNullable(clazz.cast(idnLangCode()));
        case "DurationInYears":
            return Optional.ofNullable(clazz.cast(durationInYears()));
        case "Nameservers":
            return Optional.ofNullable(clazz.cast(nameservers()));
        case "AuthCode":
            return Optional.ofNullable(clazz.cast(authCode()));
        case "AutoRenew":
            return Optional.ofNullable(clazz.cast(autoRenew()));
        case "AdminContact":
            return Optional.ofNullable(clazz.cast(adminContact()));
        case "RegistrantContact":
            return Optional.ofNullable(clazz.cast(registrantContact()));
        case "TechContact":
            return Optional.ofNullable(clazz.cast(techContact()));
        case "PrivacyProtectAdminContact":
            return Optional.ofNullable(clazz.cast(privacyProtectAdminContact()));
        case "PrivacyProtectRegistrantContact":
            return Optional.ofNullable(clazz.cast(privacyProtectRegistrantContact()));
        case "PrivacyProtectTechContact":
            return Optional.ofNullable(clazz.cast(privacyProtectTechContact()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TransferDomainRequest, T> g) {
        return obj -> g.apply((TransferDomainRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Route53DomainsRequest.Builder, SdkPojo, CopyableBuilder<Builder, TransferDomainRequest> {
        /**
         * <p>
         * The name of the domain that you want to transfer to Route 53. The top-level domain (TLD), such as .com, must
         * be a TLD that Route 53 supports. For a list of supported TLDs, see <a
         * href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/registrar-tld-list.html">Domains that You Can
         * Register with Amazon Route 53</a> in the <i>Amazon Route 53 Developer Guide</i>.
         * </p>
         * <p>
         * The domain name can contain only the following characters:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Letters a through z. Domain names are not case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Numbers 0 through 9.
         * </p>
         * </li>
         * <li>
         * <p>
         * Hyphen (-). You can't specify a hyphen at the beginning or end of a label.
         * </p>
         * </li>
         * <li>
         * <p>
         * Period (.) to separate the labels in the name, such as the <code>.</code> in <code>example.com</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param domainName
         *        The name of the domain that you want to transfer to Route 53. The top-level domain (TLD), such as
         *        .com, must be a TLD that Route 53 supports. For a list of supported TLDs, see <a
         *        href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/registrar-tld-list.html">Domains that
         *        You Can Register with Amazon Route 53</a> in the <i>Amazon Route 53 Developer Guide</i>.</p>
         *        <p>
         *        The domain name can contain only the following characters:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Letters a through z. Domain names are not case sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Numbers 0 through 9.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Hyphen (-). You can't specify a hyphen at the beginning or end of a label.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Period (.) to separate the labels in the name, such as the <code>.</code> in <code>example.com</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * Reserved for future use.
         * </p>
         * 
         * @param idnLangCode
         *        Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idnLangCode(String idnLangCode);

        /**
         * <p>
         * The number of years that you want to register the domain for. Domains are registered for a minimum of one
         * year. The maximum period depends on the top-level domain.
         * </p>
         * <p>
         * Default: 1
         * </p>
         * 
         * @param durationInYears
         *        The number of years that you want to register the domain for. Domains are registered for a minimum of
         *        one year. The maximum period depends on the top-level domain.</p>
         *        <p>
         *        Default: 1
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder durationInYears(Integer durationInYears);

        /**
         * <p>
         * Contains details for the host and glue IP addresses.
         * </p>
         * 
         * @param nameservers
         *        Contains details for the host and glue IP addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nameservers(Collection<Nameserver> nameservers);

        /**
         * <p>
         * Contains details for the host and glue IP addresses.
         * </p>
         * 
         * @param nameservers
         *        Contains details for the host and glue IP addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nameservers(Nameserver... nameservers);

        /**
         * <p>
         * Contains details for the host and glue IP addresses.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.route53domains.model.Nameserver.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.route53domains.model.Nameserver#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.route53domains.model.Nameserver.Builder#build()} is called immediately
         * and its result is passed to {@link #nameservers(List<Nameserver>)}.
         * 
         * @param nameservers
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.route53domains.model.Nameserver.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nameservers(java.util.Collection<Nameserver>)
         */
        Builder nameservers(Consumer<Nameserver.Builder>... nameservers);

        /**
         * <p>
         * The authorization code for the domain. You get this value from the current registrar.
         * </p>
         * 
         * @param authCode
         *        The authorization code for the domain. You get this value from the current registrar.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authCode(String authCode);

        /**
         * <p>
         * Indicates whether the domain will be automatically renewed (true) or not (false). Autorenewal only takes
         * effect after the account is charged.
         * </p>
         * <p>
         * Default: true
         * </p>
         * 
         * @param autoRenew
         *        Indicates whether the domain will be automatically renewed (true) or not (false). Autorenewal only
         *        takes effect after the account is charged.</p>
         *        <p>
         *        Default: true
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoRenew(Boolean autoRenew);

        /**
         * <p>
         * Provides detailed contact information.
         * </p>
         * 
         * @param adminContact
         *        Provides detailed contact information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adminContact(ContactDetail adminContact);

        /**
         * <p>
         * Provides detailed contact information.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContactDetail.Builder} avoiding the need
         * to create one manually via {@link ContactDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContactDetail.Builder#build()} is called immediately and its
         * result is passed to {@link #adminContact(ContactDetail)}.
         * 
         * @param adminContact
         *        a consumer that will call methods on {@link ContactDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #adminContact(ContactDetail)
         */
        default Builder adminContact(Consumer<ContactDetail.Builder> adminContact) {
            return adminContact(ContactDetail.builder().applyMutation(adminContact).build());
        }

        /**
         * <p>
         * Provides detailed contact information.
         * </p>
         * 
         * @param registrantContact
         *        Provides detailed contact information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registrantContact(ContactDetail registrantContact);

        /**
         * <p>
         * Provides detailed contact information.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContactDetail.Builder} avoiding the need
         * to create one manually via {@link ContactDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContactDetail.Builder#build()} is called immediately and its
         * result is passed to {@link #registrantContact(ContactDetail)}.
         * 
         * @param registrantContact
         *        a consumer that will call methods on {@link ContactDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #registrantContact(ContactDetail)
         */
        default Builder registrantContact(Consumer<ContactDetail.Builder> registrantContact) {
            return registrantContact(ContactDetail.builder().applyMutation(registrantContact).build());
        }

        /**
         * <p>
         * Provides detailed contact information.
         * </p>
         * 
         * @param techContact
         *        Provides detailed contact information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder techContact(ContactDetail techContact);

        /**
         * <p>
         * Provides detailed contact information.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContactDetail.Builder} avoiding the need
         * to create one manually via {@link ContactDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContactDetail.Builder#build()} is called immediately and its
         * result is passed to {@link #techContact(ContactDetail)}.
         * 
         * @param techContact
         *        a consumer that will call methods on {@link ContactDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #techContact(ContactDetail)
         */
        default Builder techContact(Consumer<ContactDetail.Builder> techContact) {
            return techContact(ContactDetail.builder().applyMutation(techContact).build());
        }

        /**
         * <p>
         * Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>, WHOIS
         * ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and .org domains)
         * or for our registrar associate, Gandi (for all other TLDs). If you specify <code>false</code>, WHOIS queries
         * return the information that you entered for the admin contact.
         * </p>
         * <note>
         * <p>
         * You must specify the same privacy setting for the administrative, registrant, and technical contacts.
         * </p>
         * </note>
         * <p>
         * Default: <code>true</code>
         * </p>
         * 
         * @param privacyProtectAdminContact
         *        Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>,
         *        WHOIS ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and
         *        .org domains) or for our registrar associate, Gandi (for all other TLDs). If you specify
         *        <code>false</code>, WHOIS queries return the information that you entered for the admin contact.</p>
         *        <note>
         *        <p>
         *        You must specify the same privacy setting for the administrative, registrant, and technical contacts.
         *        </p>
         *        </note>
         *        <p>
         *        Default: <code>true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privacyProtectAdminContact(Boolean privacyProtectAdminContact);

        /**
         * <p>
         * Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>, WHOIS
         * ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and .org domains)
         * or for our registrar associate, Gandi (for all other TLDs). If you specify <code>false</code>, WHOIS queries
         * return the information that you entered for the registrant contact (domain owner).
         * </p>
         * <note>
         * <p>
         * You must specify the same privacy setting for the administrative, registrant, and technical contacts.
         * </p>
         * </note>
         * <p>
         * Default: <code>true</code>
         * </p>
         * 
         * @param privacyProtectRegistrantContact
         *        Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>,
         *        WHOIS ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and
         *        .org domains) or for our registrar associate, Gandi (for all other TLDs). If you specify
         *        <code>false</code>, WHOIS queries return the information that you entered for the registrant contact
         *        (domain owner).</p> <note>
         *        <p>
         *        You must specify the same privacy setting for the administrative, registrant, and technical contacts.
         *        </p>
         *        </note>
         *        <p>
         *        Default: <code>true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privacyProtectRegistrantContact(Boolean privacyProtectRegistrantContact);

        /**
         * <p>
         * Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>, WHOIS
         * ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and .org domains)
         * or for our registrar associate, Gandi (for all other TLDs). If you specify <code>false</code>, WHOIS queries
         * return the information that you entered for the technical contact.
         * </p>
         * <note>
         * <p>
         * You must specify the same privacy setting for the administrative, registrant, and technical contacts.
         * </p>
         * </note>
         * <p>
         * Default: <code>true</code>
         * </p>
         * 
         * @param privacyProtectTechContact
         *        Whether you want to conceal contact information from WHOIS queries. If you specify <code>true</code>,
         *        WHOIS ("who is") queries return contact information either for Amazon Registrar (for .com, .net, and
         *        .org domains) or for our registrar associate, Gandi (for all other TLDs). If you specify
         *        <code>false</code>, WHOIS queries return the information that you entered for the technical
         *        contact.</p> <note>
         *        <p>
         *        You must specify the same privacy setting for the administrative, registrant, and technical contacts.
         *        </p>
         *        </note>
         *        <p>
         *        Default: <code>true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privacyProtectTechContact(Boolean privacyProtectTechContact);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Route53DomainsRequest.BuilderImpl implements Builder {
        private String domainName;

        private String idnLangCode;

        private Integer durationInYears;

        private List<Nameserver> nameservers = DefaultSdkAutoConstructList.getInstance();

        private String authCode;

        private Boolean autoRenew;

        private ContactDetail adminContact;

        private ContactDetail registrantContact;

        private ContactDetail techContact;

        private Boolean privacyProtectAdminContact;

        private Boolean privacyProtectRegistrantContact;

        private Boolean privacyProtectTechContact;

        private BuilderImpl() {
        }

        private BuilderImpl(TransferDomainRequest model) {
            super(model);
            domainName(model.domainName);
            idnLangCode(model.idnLangCode);
            durationInYears(model.durationInYears);
            nameservers(model.nameservers);
            authCode(model.authCode);
            autoRenew(model.autoRenew);
            adminContact(model.adminContact);
            registrantContact(model.registrantContact);
            techContact(model.techContact);
            privacyProtectAdminContact(model.privacyProtectAdminContact);
            privacyProtectRegistrantContact(model.privacyProtectRegistrantContact);
            privacyProtectTechContact(model.privacyProtectTechContact);
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final String getIdnLangCode() {
            return idnLangCode;
        }

        public final void setIdnLangCode(String idnLangCode) {
            this.idnLangCode = idnLangCode;
        }

        @Override
        public final Builder idnLangCode(String idnLangCode) {
            this.idnLangCode = idnLangCode;
            return this;
        }

        public final Integer getDurationInYears() {
            return durationInYears;
        }

        public final void setDurationInYears(Integer durationInYears) {
            this.durationInYears = durationInYears;
        }

        @Override
        public final Builder durationInYears(Integer durationInYears) {
            this.durationInYears = durationInYears;
            return this;
        }

        public final List<Nameserver.Builder> getNameservers() {
            List<Nameserver.Builder> result = NameserverListCopier.copyToBuilder(this.nameservers);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNameservers(Collection<Nameserver.BuilderImpl> nameservers) {
            this.nameservers = NameserverListCopier.copyFromBuilder(nameservers);
        }

        @Override
        public final Builder nameservers(Collection<Nameserver> nameservers) {
            this.nameservers = NameserverListCopier.copy(nameservers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nameservers(Nameserver... nameservers) {
            nameservers(Arrays.asList(nameservers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nameservers(Consumer<Nameserver.Builder>... nameservers) {
            nameservers(Stream.of(nameservers).map(c -> Nameserver.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getAuthCode() {
            return authCode;
        }

        public final void setAuthCode(String authCode) {
            this.authCode = authCode;
        }

        @Override
        public final Builder authCode(String authCode) {
            this.authCode = authCode;
            return this;
        }

        public final Boolean getAutoRenew() {
            return autoRenew;
        }

        public final void setAutoRenew(Boolean autoRenew) {
            this.autoRenew = autoRenew;
        }

        @Override
        public final Builder autoRenew(Boolean autoRenew) {
            this.autoRenew = autoRenew;
            return this;
        }

        public final ContactDetail.Builder getAdminContact() {
            return adminContact != null ? adminContact.toBuilder() : null;
        }

        public final void setAdminContact(ContactDetail.BuilderImpl adminContact) {
            this.adminContact = adminContact != null ? adminContact.build() : null;
        }

        @Override
        public final Builder adminContact(ContactDetail adminContact) {
            this.adminContact = adminContact;
            return this;
        }

        public final ContactDetail.Builder getRegistrantContact() {
            return registrantContact != null ? registrantContact.toBuilder() : null;
        }

        public final void setRegistrantContact(ContactDetail.BuilderImpl registrantContact) {
            this.registrantContact = registrantContact != null ? registrantContact.build() : null;
        }

        @Override
        public final Builder registrantContact(ContactDetail registrantContact) {
            this.registrantContact = registrantContact;
            return this;
        }

        public final ContactDetail.Builder getTechContact() {
            return techContact != null ? techContact.toBuilder() : null;
        }

        public final void setTechContact(ContactDetail.BuilderImpl techContact) {
            this.techContact = techContact != null ? techContact.build() : null;
        }

        @Override
        public final Builder techContact(ContactDetail techContact) {
            this.techContact = techContact;
            return this;
        }

        public final Boolean getPrivacyProtectAdminContact() {
            return privacyProtectAdminContact;
        }

        public final void setPrivacyProtectAdminContact(Boolean privacyProtectAdminContact) {
            this.privacyProtectAdminContact = privacyProtectAdminContact;
        }

        @Override
        public final Builder privacyProtectAdminContact(Boolean privacyProtectAdminContact) {
            this.privacyProtectAdminContact = privacyProtectAdminContact;
            return this;
        }

        public final Boolean getPrivacyProtectRegistrantContact() {
            return privacyProtectRegistrantContact;
        }

        public final void setPrivacyProtectRegistrantContact(Boolean privacyProtectRegistrantContact) {
            this.privacyProtectRegistrantContact = privacyProtectRegistrantContact;
        }

        @Override
        public final Builder privacyProtectRegistrantContact(Boolean privacyProtectRegistrantContact) {
            this.privacyProtectRegistrantContact = privacyProtectRegistrantContact;
            return this;
        }

        public final Boolean getPrivacyProtectTechContact() {
            return privacyProtectTechContact;
        }

        public final void setPrivacyProtectTechContact(Boolean privacyProtectTechContact) {
            this.privacyProtectTechContact = privacyProtectTechContact;
        }

        @Override
        public final Builder privacyProtectTechContact(Boolean privacyProtectTechContact) {
            this.privacyProtectTechContact = privacyProtectTechContact;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public TransferDomainRequest build() {
            return new TransferDomainRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
