/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53domains.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information for one billing record.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BillingRecord implements SdkPojo, Serializable, ToCopyableBuilder<BillingRecord.Builder, BillingRecord> {
    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DomainName").getter(getter(BillingRecord::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName").build()).build();

    private static final SdkField<String> OPERATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Operation").getter(getter(BillingRecord::operationAsString)).setter(setter(Builder::operation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operation").build()).build();

    private static final SdkField<String> INVOICE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InvoiceId").getter(getter(BillingRecord::invoiceId)).setter(setter(Builder::invoiceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InvoiceId").build()).build();

    private static final SdkField<Instant> BILL_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("BillDate").getter(getter(BillingRecord::billDate)).setter(setter(Builder::billDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BillDate").build()).build();

    private static final SdkField<Double> PRICE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("Price")
            .getter(getter(BillingRecord::price)).setter(setter(Builder::price))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Price").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_NAME_FIELD,
            OPERATION_FIELD, INVOICE_ID_FIELD, BILL_DATE_FIELD, PRICE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String domainName;

    private final String operation;

    private final String invoiceId;

    private final Instant billDate;

    private final Double price;

    private BillingRecord(BuilderImpl builder) {
        this.domainName = builder.domainName;
        this.operation = builder.operation;
        this.invoiceId = builder.invoiceId;
        this.billDate = builder.billDate;
        this.price = builder.price;
    }

    /**
     * <p>
     * The name of the domain that the billing record applies to. If the domain name contains characters other than a-z,
     * 0-9, and - (hyphen), such as an internationalized domain name, then this value is in Punycode. For more
     * information, see <a href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DomainNameFormat.html">DNS
     * Domain Name Format</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     * 
     * @return The name of the domain that the billing record applies to. If the domain name contains characters other
     *         than a-z, 0-9, and - (hyphen), such as an internationalized domain name, then this value is in Punycode.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DomainNameFormat.html">DNS Domain Name
     *         Format</a> in the <i>Amazon Route 53 Developer Guide</i>.
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The operation that you were charged for.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link OperationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The operation that you were charged for.
     * @see OperationType
     */
    public final OperationType operation() {
        return OperationType.fromValue(operation);
    }

    /**
     * <p>
     * The operation that you were charged for.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link OperationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The operation that you were charged for.
     * @see OperationType
     */
    public final String operationAsString() {
        return operation;
    }

    /**
     * <p>
     * The ID of the invoice that is associated with the billing record.
     * </p>
     * 
     * @return The ID of the invoice that is associated with the billing record.
     */
    public final String invoiceId() {
        return invoiceId;
    }

    /**
     * <p>
     * The date that the operation was billed, in Unix format.
     * </p>
     * 
     * @return The date that the operation was billed, in Unix format.
     */
    public final Instant billDate() {
        return billDate;
    }

    /**
     * <p>
     * The price that you were charged for the operation, in US dollars.
     * </p>
     * <p>
     * Example value: 12.0
     * </p>
     * 
     * @return The price that you were charged for the operation, in US dollars.</p>
     *         <p>
     *         Example value: 12.0
     */
    public final Double price() {
        return price;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(operationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(invoiceId());
        hashCode = 31 * hashCode + Objects.hashCode(billDate());
        hashCode = 31 * hashCode + Objects.hashCode(price());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BillingRecord)) {
            return false;
        }
        BillingRecord other = (BillingRecord) obj;
        return Objects.equals(domainName(), other.domainName()) && Objects.equals(operationAsString(), other.operationAsString())
                && Objects.equals(invoiceId(), other.invoiceId()) && Objects.equals(billDate(), other.billDate())
                && Objects.equals(price(), other.price());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BillingRecord").add("DomainName", domainName()).add("Operation", operationAsString())
                .add("InvoiceId", invoiceId()).add("BillDate", billDate()).add("Price", price()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "Operation":
            return Optional.ofNullable(clazz.cast(operationAsString()));
        case "InvoiceId":
            return Optional.ofNullable(clazz.cast(invoiceId()));
        case "BillDate":
            return Optional.ofNullable(clazz.cast(billDate()));
        case "Price":
            return Optional.ofNullable(clazz.cast(price()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BillingRecord, T> g) {
        return obj -> g.apply((BillingRecord) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BillingRecord> {
        /**
         * <p>
         * The name of the domain that the billing record applies to. If the domain name contains characters other than
         * a-z, 0-9, and - (hyphen), such as an internationalized domain name, then this value is in Punycode. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DomainNameFormat.html">DNS Domain Name
         * Format</a> in the <i>Amazon Route 53 Developer Guide</i>.
         * </p>
         * 
         * @param domainName
         *        The name of the domain that the billing record applies to. If the domain name contains characters
         *        other than a-z, 0-9, and - (hyphen), such as an internationalized domain name, then this value is in
         *        Punycode. For more information, see <a
         *        href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DomainNameFormat.html">DNS Domain Name
         *        Format</a> in the <i>Amazon Route 53 Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The operation that you were charged for.
         * </p>
         * 
         * @param operation
         *        The operation that you were charged for.
         * @see OperationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperationType
         */
        Builder operation(String operation);

        /**
         * <p>
         * The operation that you were charged for.
         * </p>
         * 
         * @param operation
         *        The operation that you were charged for.
         * @see OperationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperationType
         */
        Builder operation(OperationType operation);

        /**
         * <p>
         * The ID of the invoice that is associated with the billing record.
         * </p>
         * 
         * @param invoiceId
         *        The ID of the invoice that is associated with the billing record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invoiceId(String invoiceId);

        /**
         * <p>
         * The date that the operation was billed, in Unix format.
         * </p>
         * 
         * @param billDate
         *        The date that the operation was billed, in Unix format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder billDate(Instant billDate);

        /**
         * <p>
         * The price that you were charged for the operation, in US dollars.
         * </p>
         * <p>
         * Example value: 12.0
         * </p>
         * 
         * @param price
         *        The price that you were charged for the operation, in US dollars.</p>
         *        <p>
         *        Example value: 12.0
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder price(Double price);
    }

    static final class BuilderImpl implements Builder {
        private String domainName;

        private String operation;

        private String invoiceId;

        private Instant billDate;

        private Double price;

        private BuilderImpl() {
        }

        private BuilderImpl(BillingRecord model) {
            domainName(model.domainName);
            operation(model.operation);
            invoiceId(model.invoiceId);
            billDate(model.billDate);
            price(model.price);
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final String getOperation() {
            return operation;
        }

        public final void setOperation(String operation) {
            this.operation = operation;
        }

        @Override
        public final Builder operation(String operation) {
            this.operation = operation;
            return this;
        }

        @Override
        public final Builder operation(OperationType operation) {
            this.operation(operation == null ? null : operation.toString());
            return this;
        }

        public final String getInvoiceId() {
            return invoiceId;
        }

        public final void setInvoiceId(String invoiceId) {
            this.invoiceId = invoiceId;
        }

        @Override
        public final Builder invoiceId(String invoiceId) {
            this.invoiceId = invoiceId;
            return this;
        }

        public final Instant getBillDate() {
            return billDate;
        }

        public final void setBillDate(Instant billDate) {
            this.billDate = billDate;
        }

        @Override
        public final Builder billDate(Instant billDate) {
            this.billDate = billDate;
            return this;
        }

        public final Double getPrice() {
            return price;
        }

        public final void setPrice(Double price) {
            this.price = price;
        }

        @Override
        public final Builder price(Double price) {
            this.price = price;
            return this;
        }

        @Override
        public BillingRecord build() {
            return new BillingRecord(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
