/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53resolver.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * In the response to a <a
 * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html"
 * >CreateResolverEndpoint</a>, <a
 * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DeleteResolverEndpoint.html"
 * >DeleteResolverEndpoint</a>, <a
 * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverEndpoint.html"
 * >GetResolverEndpoint</a>, Updates the name, or ResolverEndpointType for an endpoint, or <a
 * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_UpdateResolverEndpoint.html"
 * >UpdateResolverEndpoint</a> request, a complex type that contains settings for an existing inbound or outbound
 * Resolver endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResolverEndpoint implements SdkPojo, Serializable,
        ToCopyableBuilder<ResolverEndpoint.Builder, ResolverEndpoint> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(ResolverEndpoint::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> CREATOR_REQUEST_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CreatorRequestId").getter(getter(ResolverEndpoint::creatorRequestId))
            .setter(setter(Builder::creatorRequestId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatorRequestId").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(ResolverEndpoint::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ResolverEndpoint::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(ResolverEndpoint::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DIRECTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Direction").getter(getter(ResolverEndpoint::directionAsString)).setter(setter(Builder::direction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Direction").build()).build();

    private static final SdkField<Integer> IP_ADDRESS_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("IpAddressCount").getter(getter(ResolverEndpoint::ipAddressCount))
            .setter(setter(Builder::ipAddressCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddressCount").build()).build();

    private static final SdkField<String> HOST_VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HostVPCId").getter(getter(ResolverEndpoint::hostVPCId)).setter(setter(Builder::hostVPCId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HostVPCId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ResolverEndpoint::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage").getter(getter(ResolverEndpoint::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final SdkField<String> CREATION_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CreationTime").getter(getter(ResolverEndpoint::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> MODIFICATION_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModificationTime").getter(getter(ResolverEndpoint::modificationTime))
            .setter(setter(Builder::modificationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModificationTime").build()).build();

    private static final SdkField<String> RESOLVER_ENDPOINT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResolverEndpointType").getter(getter(ResolverEndpoint::resolverEndpointTypeAsString))
            .setter(setter(Builder::resolverEndpointType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResolverEndpointType").build())
            .build();

    private static final SdkField<String> OUTPOST_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutpostArn").getter(getter(ResolverEndpoint::outpostArn)).setter(setter(Builder::outpostArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutpostArn").build()).build();

    private static final SdkField<String> PREFERRED_INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredInstanceType").getter(getter(ResolverEndpoint::preferredInstanceType))
            .setter(setter(Builder::preferredInstanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredInstanceType").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD,
            CREATOR_REQUEST_ID_FIELD, ARN_FIELD, NAME_FIELD, SECURITY_GROUP_IDS_FIELD, DIRECTION_FIELD, IP_ADDRESS_COUNT_FIELD,
            HOST_VPC_ID_FIELD, STATUS_FIELD, STATUS_MESSAGE_FIELD, CREATION_TIME_FIELD, MODIFICATION_TIME_FIELD,
            RESOLVER_ENDPOINT_TYPE_FIELD, OUTPOST_ARN_FIELD, PREFERRED_INSTANCE_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String creatorRequestId;

    private final String arn;

    private final String name;

    private final List<String> securityGroupIds;

    private final String direction;

    private final Integer ipAddressCount;

    private final String hostVPCId;

    private final String status;

    private final String statusMessage;

    private final String creationTime;

    private final String modificationTime;

    private final String resolverEndpointType;

    private final String outpostArn;

    private final String preferredInstanceType;

    private ResolverEndpoint(BuilderImpl builder) {
        this.id = builder.id;
        this.creatorRequestId = builder.creatorRequestId;
        this.arn = builder.arn;
        this.name = builder.name;
        this.securityGroupIds = builder.securityGroupIds;
        this.direction = builder.direction;
        this.ipAddressCount = builder.ipAddressCount;
        this.hostVPCId = builder.hostVPCId;
        this.status = builder.status;
        this.statusMessage = builder.statusMessage;
        this.creationTime = builder.creationTime;
        this.modificationTime = builder.modificationTime;
        this.resolverEndpointType = builder.resolverEndpointType;
        this.outpostArn = builder.outpostArn;
        this.preferredInstanceType = builder.preferredInstanceType;
    }

    /**
     * <p>
     * The ID of the Resolver endpoint.
     * </p>
     * 
     * @return The ID of the Resolver endpoint.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * A unique string that identifies the request that created the Resolver endpoint. The <code>CreatorRequestId</code>
     * allows failed requests to be retried without the risk of running the operation twice.
     * </p>
     * 
     * @return A unique string that identifies the request that created the Resolver endpoint. The
     *         <code>CreatorRequestId</code> allows failed requests to be retried without the risk of running the
     *         operation twice.
     */
    public final String creatorRequestId() {
        return creatorRequestId;
    }

    /**
     * <p>
     * The ARN (Amazon Resource Name) for the Resolver endpoint.
     * </p>
     * 
     * @return The ARN (Amazon Resource Name) for the Resolver endpoint.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The name that you assigned to the Resolver endpoint when you submitted a <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html"
     * >CreateResolverEndpoint</a> request.
     * </p>
     * 
     * @return The name that you assigned to the Resolver endpoint when you submitted a <a
     *         href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html"
     *         >CreateResolverEndpoint</a> request.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ID of one or more security groups that control access to this VPC. The security group must include one or
     * more inbound rules (for inbound endpoints) or outbound rules (for outbound endpoints). Inbound and outbound rules
     * must allow TCP and UDP access. For inbound access, open port 53. For outbound access, open the port that you're
     * using for DNS queries on your network.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return The ID of one or more security groups that control access to this VPC. The security group must include
     *         one or more inbound rules (for inbound endpoints) or outbound rules (for outbound endpoints). Inbound and
     *         outbound rules must allow TCP and UDP access. For inbound access, open port 53. For outbound access, open
     *         the port that you're using for DNS queries on your network.
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * <p>
     * Indicates whether the Resolver endpoint allows inbound or outbound DNS queries:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INBOUND</code>: allows DNS queries to your VPC from your network
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OUTBOUND</code>: allows DNS queries from your VPC to your network
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #direction} will
     * return {@link ResolverEndpointDirection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #directionAsString}.
     * </p>
     * 
     * @return Indicates whether the Resolver endpoint allows inbound or outbound DNS queries:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INBOUND</code>: allows DNS queries to your VPC from your network
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OUTBOUND</code>: allows DNS queries from your VPC to your network
     *         </p>
     *         </li>
     * @see ResolverEndpointDirection
     */
    public final ResolverEndpointDirection direction() {
        return ResolverEndpointDirection.fromValue(direction);
    }

    /**
     * <p>
     * Indicates whether the Resolver endpoint allows inbound or outbound DNS queries:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INBOUND</code>: allows DNS queries to your VPC from your network
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OUTBOUND</code>: allows DNS queries from your VPC to your network
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #direction} will
     * return {@link ResolverEndpointDirection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #directionAsString}.
     * </p>
     * 
     * @return Indicates whether the Resolver endpoint allows inbound or outbound DNS queries:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INBOUND</code>: allows DNS queries to your VPC from your network
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OUTBOUND</code>: allows DNS queries from your VPC to your network
     *         </p>
     *         </li>
     * @see ResolverEndpointDirection
     */
    public final String directionAsString() {
        return direction;
    }

    /**
     * <p>
     * The number of IP addresses that the Resolver endpoint can use for DNS queries.
     * </p>
     * 
     * @return The number of IP addresses that the Resolver endpoint can use for DNS queries.
     */
    public final Integer ipAddressCount() {
        return ipAddressCount;
    }

    /**
     * <p>
     * The ID of the VPC that you want to create the Resolver endpoint in.
     * </p>
     * 
     * @return The ID of the VPC that you want to create the Resolver endpoint in.
     */
    public final String hostVPCId() {
        return hostVPCId;
    }

    /**
     * <p>
     * A code that specifies the current status of the Resolver endpoint. Valid values include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code>: Resolver is creating and configuring one or more Amazon VPC network interfaces for this
     * endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OPERATIONAL</code>: The Amazon VPC network interfaces for this endpoint are correctly configured and able
     * to pass inbound or outbound DNS queries between your network and Resolver.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code>: Resolver is associating or disassociating one or more network interfaces with this
     * endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AUTO_RECOVERING</code>: Resolver is trying to recover one or more of the network interfaces that are
     * associated with this endpoint. During the recovery process, the endpoint functions with limited capacity because
     * of the limit on the number of DNS queries per IP address (per network interface). For the current limit, see <a
     * href
     * ="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities-resolver">
     * Limits on Route 53 Resolver</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTION_NEEDED</code>: This endpoint is unhealthy, and Resolver can't automatically recover it. To resolve
     * the problem, we recommend that you check each IP address that you associated with the endpoint. For each IP
     * address that isn't available, add another IP address and then delete the IP address that isn't available. (An
     * endpoint must always include at least two IP addresses.) A status of <code>ACTION_NEEDED</code> can have a
     * variety of causes. Here are two common causes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * One or more of the network interfaces that are associated with the endpoint were deleted using Amazon VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * The network interface couldn't be created for some reason that's outside the control of Resolver.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code>: Resolver is deleting this endpoint and the associated network interfaces.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResolverEndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return A code that specifies the current status of the Resolver endpoint. Valid values include the
     *         following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code>: Resolver is creating and configuring one or more Amazon VPC network interfaces for
     *         this endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OPERATIONAL</code>: The Amazon VPC network interfaces for this endpoint are correctly configured
     *         and able to pass inbound or outbound DNS queries between your network and Resolver.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code>: Resolver is associating or disassociating one or more network interfaces with this
     *         endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AUTO_RECOVERING</code>: Resolver is trying to recover one or more of the network interfaces that
     *         are associated with this endpoint. During the recovery process, the endpoint functions with limited
     *         capacity because of the limit on the number of DNS queries per IP address (per network interface). For
     *         the current limit, see <a href=
     *         "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities-resolver"
     *         >Limits on Route 53 Resolver</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTION_NEEDED</code>: This endpoint is unhealthy, and Resolver can't automatically recover it. To
     *         resolve the problem, we recommend that you check each IP address that you associated with the endpoint.
     *         For each IP address that isn't available, add another IP address and then delete the IP address that
     *         isn't available. (An endpoint must always include at least two IP addresses.) A status of
     *         <code>ACTION_NEEDED</code> can have a variety of causes. Here are two common causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         One or more of the network interfaces that are associated with the endpoint were deleted using Amazon
     *         VPC.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The network interface couldn't be created for some reason that's outside the control of Resolver.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code>: Resolver is deleting this endpoint and the associated network interfaces.
     *         </p>
     *         </li>
     * @see ResolverEndpointStatus
     */
    public final ResolverEndpointStatus status() {
        return ResolverEndpointStatus.fromValue(status);
    }

    /**
     * <p>
     * A code that specifies the current status of the Resolver endpoint. Valid values include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code>: Resolver is creating and configuring one or more Amazon VPC network interfaces for this
     * endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OPERATIONAL</code>: The Amazon VPC network interfaces for this endpoint are correctly configured and able
     * to pass inbound or outbound DNS queries between your network and Resolver.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code>: Resolver is associating or disassociating one or more network interfaces with this
     * endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AUTO_RECOVERING</code>: Resolver is trying to recover one or more of the network interfaces that are
     * associated with this endpoint. During the recovery process, the endpoint functions with limited capacity because
     * of the limit on the number of DNS queries per IP address (per network interface). For the current limit, see <a
     * href
     * ="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities-resolver">
     * Limits on Route 53 Resolver</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTION_NEEDED</code>: This endpoint is unhealthy, and Resolver can't automatically recover it. To resolve
     * the problem, we recommend that you check each IP address that you associated with the endpoint. For each IP
     * address that isn't available, add another IP address and then delete the IP address that isn't available. (An
     * endpoint must always include at least two IP addresses.) A status of <code>ACTION_NEEDED</code> can have a
     * variety of causes. Here are two common causes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * One or more of the network interfaces that are associated with the endpoint were deleted using Amazon VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * The network interface couldn't be created for some reason that's outside the control of Resolver.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code>: Resolver is deleting this endpoint and the associated network interfaces.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResolverEndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return A code that specifies the current status of the Resolver endpoint. Valid values include the
     *         following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code>: Resolver is creating and configuring one or more Amazon VPC network interfaces for
     *         this endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OPERATIONAL</code>: The Amazon VPC network interfaces for this endpoint are correctly configured
     *         and able to pass inbound or outbound DNS queries between your network and Resolver.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code>: Resolver is associating or disassociating one or more network interfaces with this
     *         endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AUTO_RECOVERING</code>: Resolver is trying to recover one or more of the network interfaces that
     *         are associated with this endpoint. During the recovery process, the endpoint functions with limited
     *         capacity because of the limit on the number of DNS queries per IP address (per network interface). For
     *         the current limit, see <a href=
     *         "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities-resolver"
     *         >Limits on Route 53 Resolver</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTION_NEEDED</code>: This endpoint is unhealthy, and Resolver can't automatically recover it. To
     *         resolve the problem, we recommend that you check each IP address that you associated with the endpoint.
     *         For each IP address that isn't available, add another IP address and then delete the IP address that
     *         isn't available. (An endpoint must always include at least two IP addresses.) A status of
     *         <code>ACTION_NEEDED</code> can have a variety of causes. Here are two common causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         One or more of the network interfaces that are associated with the endpoint were deleted using Amazon
     *         VPC.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The network interface couldn't be created for some reason that's outside the control of Resolver.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code>: Resolver is deleting this endpoint and the associated network interfaces.
     *         </p>
     *         </li>
     * @see ResolverEndpointStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A detailed description of the status of the Resolver endpoint.
     * </p>
     * 
     * @return A detailed description of the status of the Resolver endpoint.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * The date and time that the endpoint was created, in Unix time format and Coordinated Universal Time (UTC).
     * </p>
     * 
     * @return The date and time that the endpoint was created, in Unix time format and Coordinated Universal Time
     *         (UTC).
     */
    public final String creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The date and time that the endpoint was last modified, in Unix time format and Coordinated Universal Time (UTC).
     * </p>
     * 
     * @return The date and time that the endpoint was last modified, in Unix time format and Coordinated Universal Time
     *         (UTC).
     */
    public final String modificationTime() {
        return modificationTime;
    }

    /**
     * <p>
     * The Resolver endpoint IP address type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #resolverEndpointType} will return {@link ResolverEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #resolverEndpointTypeAsString}.
     * </p>
     * 
     * @return The Resolver endpoint IP address type.
     * @see ResolverEndpointType
     */
    public final ResolverEndpointType resolverEndpointType() {
        return ResolverEndpointType.fromValue(resolverEndpointType);
    }

    /**
     * <p>
     * The Resolver endpoint IP address type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #resolverEndpointType} will return {@link ResolverEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #resolverEndpointTypeAsString}.
     * </p>
     * 
     * @return The Resolver endpoint IP address type.
     * @see ResolverEndpointType
     */
    public final String resolverEndpointTypeAsString() {
        return resolverEndpointType;
    }

    /**
     * <p>
     * The ARN (Amazon Resource Name) for the Outpost.
     * </p>
     * 
     * @return The ARN (Amazon Resource Name) for the Outpost.
     */
    public final String outpostArn() {
        return outpostArn;
    }

    /**
     * <p>
     * The Amazon EC2 instance type.
     * </p>
     * 
     * @return The Amazon EC2 instance type.
     */
    public final String preferredInstanceType() {
        return preferredInstanceType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(creatorRequestId());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(directionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddressCount());
        hashCode = 31 * hashCode + Objects.hashCode(hostVPCId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(modificationTime());
        hashCode = 31 * hashCode + Objects.hashCode(resolverEndpointTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(outpostArn());
        hashCode = 31 * hashCode + Objects.hashCode(preferredInstanceType());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResolverEndpoint)) {
            return false;
        }
        ResolverEndpoint other = (ResolverEndpoint) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(creatorRequestId(), other.creatorRequestId())
                && Objects.equals(arn(), other.arn()) && Objects.equals(name(), other.name())
                && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds())
                && Objects.equals(directionAsString(), other.directionAsString())
                && Objects.equals(ipAddressCount(), other.ipAddressCount()) && Objects.equals(hostVPCId(), other.hostVPCId())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusMessage(), other.statusMessage()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(modificationTime(), other.modificationTime())
                && Objects.equals(resolverEndpointTypeAsString(), other.resolverEndpointTypeAsString())
                && Objects.equals(outpostArn(), other.outpostArn())
                && Objects.equals(preferredInstanceType(), other.preferredInstanceType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResolverEndpoint").add("Id", id()).add("CreatorRequestId", creatorRequestId()).add("Arn", arn())
                .add("Name", name()).add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null)
                .add("Direction", directionAsString()).add("IpAddressCount", ipAddressCount()).add("HostVPCId", hostVPCId())
                .add("Status", statusAsString()).add("StatusMessage", statusMessage()).add("CreationTime", creationTime())
                .add("ModificationTime", modificationTime()).add("ResolverEndpointType", resolverEndpointTypeAsString())
                .add("OutpostArn", outpostArn()).add("PreferredInstanceType", preferredInstanceType()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "CreatorRequestId":
            return Optional.ofNullable(clazz.cast(creatorRequestId()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "Direction":
            return Optional.ofNullable(clazz.cast(directionAsString()));
        case "IpAddressCount":
            return Optional.ofNullable(clazz.cast(ipAddressCount()));
        case "HostVPCId":
            return Optional.ofNullable(clazz.cast(hostVPCId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "ModificationTime":
            return Optional.ofNullable(clazz.cast(modificationTime()));
        case "ResolverEndpointType":
            return Optional.ofNullable(clazz.cast(resolverEndpointTypeAsString()));
        case "OutpostArn":
            return Optional.ofNullable(clazz.cast(outpostArn()));
        case "PreferredInstanceType":
            return Optional.ofNullable(clazz.cast(preferredInstanceType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResolverEndpoint, T> g) {
        return obj -> g.apply((ResolverEndpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResolverEndpoint> {
        /**
         * <p>
         * The ID of the Resolver endpoint.
         * </p>
         * 
         * @param id
         *        The ID of the Resolver endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * A unique string that identifies the request that created the Resolver endpoint. The
         * <code>CreatorRequestId</code> allows failed requests to be retried without the risk of running the operation
         * twice.
         * </p>
         * 
         * @param creatorRequestId
         *        A unique string that identifies the request that created the Resolver endpoint. The
         *        <code>CreatorRequestId</code> allows failed requests to be retried without the risk of running the
         *        operation twice.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creatorRequestId(String creatorRequestId);

        /**
         * <p>
         * The ARN (Amazon Resource Name) for the Resolver endpoint.
         * </p>
         * 
         * @param arn
         *        The ARN (Amazon Resource Name) for the Resolver endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The name that you assigned to the Resolver endpoint when you submitted a <a href=
         * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html"
         * >CreateResolverEndpoint</a> request.
         * </p>
         * 
         * @param name
         *        The name that you assigned to the Resolver endpoint when you submitted a <a href=
         *        "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html"
         *        >CreateResolverEndpoint</a> request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of one or more security groups that control access to this VPC. The security group must include one or
         * more inbound rules (for inbound endpoints) or outbound rules (for outbound endpoints). Inbound and outbound
         * rules must allow TCP and UDP access. For inbound access, open port 53. For outbound access, open the port
         * that you're using for DNS queries on your network.
         * </p>
         * 
         * @param securityGroupIds
         *        The ID of one or more security groups that control access to this VPC. The security group must include
         *        one or more inbound rules (for inbound endpoints) or outbound rules (for outbound endpoints). Inbound
         *        and outbound rules must allow TCP and UDP access. For inbound access, open port 53. For outbound
         *        access, open the port that you're using for DNS queries on your network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The ID of one or more security groups that control access to this VPC. The security group must include one or
         * more inbound rules (for inbound endpoints) or outbound rules (for outbound endpoints). Inbound and outbound
         * rules must allow TCP and UDP access. For inbound access, open port 53. For outbound access, open the port
         * that you're using for DNS queries on your network.
         * </p>
         * 
         * @param securityGroupIds
         *        The ID of one or more security groups that control access to this VPC. The security group must include
         *        one or more inbound rules (for inbound endpoints) or outbound rules (for outbound endpoints). Inbound
         *        and outbound rules must allow TCP and UDP access. For inbound access, open port 53. For outbound
         *        access, open the port that you're using for DNS queries on your network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * Indicates whether the Resolver endpoint allows inbound or outbound DNS queries:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INBOUND</code>: allows DNS queries to your VPC from your network
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OUTBOUND</code>: allows DNS queries from your VPC to your network
         * </p>
         * </li>
         * </ul>
         * 
         * @param direction
         *        Indicates whether the Resolver endpoint allows inbound or outbound DNS queries:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INBOUND</code>: allows DNS queries to your VPC from your network
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OUTBOUND</code>: allows DNS queries from your VPC to your network
         *        </p>
         *        </li>
         * @see ResolverEndpointDirection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverEndpointDirection
         */
        Builder direction(String direction);

        /**
         * <p>
         * Indicates whether the Resolver endpoint allows inbound or outbound DNS queries:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INBOUND</code>: allows DNS queries to your VPC from your network
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OUTBOUND</code>: allows DNS queries from your VPC to your network
         * </p>
         * </li>
         * </ul>
         * 
         * @param direction
         *        Indicates whether the Resolver endpoint allows inbound or outbound DNS queries:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INBOUND</code>: allows DNS queries to your VPC from your network
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OUTBOUND</code>: allows DNS queries from your VPC to your network
         *        </p>
         *        </li>
         * @see ResolverEndpointDirection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverEndpointDirection
         */
        Builder direction(ResolverEndpointDirection direction);

        /**
         * <p>
         * The number of IP addresses that the Resolver endpoint can use for DNS queries.
         * </p>
         * 
         * @param ipAddressCount
         *        The number of IP addresses that the Resolver endpoint can use for DNS queries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddressCount(Integer ipAddressCount);

        /**
         * <p>
         * The ID of the VPC that you want to create the Resolver endpoint in.
         * </p>
         * 
         * @param hostVPCId
         *        The ID of the VPC that you want to create the Resolver endpoint in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostVPCId(String hostVPCId);

        /**
         * <p>
         * A code that specifies the current status of the Resolver endpoint. Valid values include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code>: Resolver is creating and configuring one or more Amazon VPC network interfaces for
         * this endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OPERATIONAL</code>: The Amazon VPC network interfaces for this endpoint are correctly configured and
         * able to pass inbound or outbound DNS queries between your network and Resolver.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code>: Resolver is associating or disassociating one or more network interfaces with this
         * endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AUTO_RECOVERING</code>: Resolver is trying to recover one or more of the network interfaces that are
         * associated with this endpoint. During the recovery process, the endpoint functions with limited capacity
         * because of the limit on the number of DNS queries per IP address (per network interface). For the current
         * limit, see <a href=
         * "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities-resolver"
         * >Limits on Route 53 Resolver</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTION_NEEDED</code>: This endpoint is unhealthy, and Resolver can't automatically recover it. To
         * resolve the problem, we recommend that you check each IP address that you associated with the endpoint. For
         * each IP address that isn't available, add another IP address and then delete the IP address that isn't
         * available. (An endpoint must always include at least two IP addresses.) A status of
         * <code>ACTION_NEEDED</code> can have a variety of causes. Here are two common causes:
         * </p>
         * <ul>
         * <li>
         * <p>
         * One or more of the network interfaces that are associated with the endpoint were deleted using Amazon VPC.
         * </p>
         * </li>
         * <li>
         * <p>
         * The network interface couldn't be created for some reason that's outside the control of Resolver.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code>: Resolver is deleting this endpoint and the associated network interfaces.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        A code that specifies the current status of the Resolver endpoint. Valid values include the
         *        following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code>: Resolver is creating and configuring one or more Amazon VPC network interfaces
         *        for this endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OPERATIONAL</code>: The Amazon VPC network interfaces for this endpoint are correctly configured
         *        and able to pass inbound or outbound DNS queries between your network and Resolver.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code>: Resolver is associating or disassociating one or more network interfaces with
         *        this endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AUTO_RECOVERING</code>: Resolver is trying to recover one or more of the network interfaces that
         *        are associated with this endpoint. During the recovery process, the endpoint functions with limited
         *        capacity because of the limit on the number of DNS queries per IP address (per network interface). For
         *        the current limit, see <a href=
         *        "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities-resolver"
         *        >Limits on Route 53 Resolver</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTION_NEEDED</code>: This endpoint is unhealthy, and Resolver can't automatically recover it.
         *        To resolve the problem, we recommend that you check each IP address that you associated with the
         *        endpoint. For each IP address that isn't available, add another IP address and then delete the IP
         *        address that isn't available. (An endpoint must always include at least two IP addresses.) A status of
         *        <code>ACTION_NEEDED</code> can have a variety of causes. Here are two common causes:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        One or more of the network interfaces that are associated with the endpoint were deleted using Amazon
         *        VPC.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The network interface couldn't be created for some reason that's outside the control of Resolver.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code>: Resolver is deleting this endpoint and the associated network interfaces.
         *        </p>
         *        </li>
         * @see ResolverEndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverEndpointStatus
         */
        Builder status(String status);

        /**
         * <p>
         * A code that specifies the current status of the Resolver endpoint. Valid values include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code>: Resolver is creating and configuring one or more Amazon VPC network interfaces for
         * this endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OPERATIONAL</code>: The Amazon VPC network interfaces for this endpoint are correctly configured and
         * able to pass inbound or outbound DNS queries between your network and Resolver.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code>: Resolver is associating or disassociating one or more network interfaces with this
         * endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AUTO_RECOVERING</code>: Resolver is trying to recover one or more of the network interfaces that are
         * associated with this endpoint. During the recovery process, the endpoint functions with limited capacity
         * because of the limit on the number of DNS queries per IP address (per network interface). For the current
         * limit, see <a href=
         * "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities-resolver"
         * >Limits on Route 53 Resolver</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTION_NEEDED</code>: This endpoint is unhealthy, and Resolver can't automatically recover it. To
         * resolve the problem, we recommend that you check each IP address that you associated with the endpoint. For
         * each IP address that isn't available, add another IP address and then delete the IP address that isn't
         * available. (An endpoint must always include at least two IP addresses.) A status of
         * <code>ACTION_NEEDED</code> can have a variety of causes. Here are two common causes:
         * </p>
         * <ul>
         * <li>
         * <p>
         * One or more of the network interfaces that are associated with the endpoint were deleted using Amazon VPC.
         * </p>
         * </li>
         * <li>
         * <p>
         * The network interface couldn't be created for some reason that's outside the control of Resolver.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code>: Resolver is deleting this endpoint and the associated network interfaces.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        A code that specifies the current status of the Resolver endpoint. Valid values include the
         *        following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code>: Resolver is creating and configuring one or more Amazon VPC network interfaces
         *        for this endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OPERATIONAL</code>: The Amazon VPC network interfaces for this endpoint are correctly configured
         *        and able to pass inbound or outbound DNS queries between your network and Resolver.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code>: Resolver is associating or disassociating one or more network interfaces with
         *        this endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AUTO_RECOVERING</code>: Resolver is trying to recover one or more of the network interfaces that
         *        are associated with this endpoint. During the recovery process, the endpoint functions with limited
         *        capacity because of the limit on the number of DNS queries per IP address (per network interface). For
         *        the current limit, see <a href=
         *        "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities-resolver"
         *        >Limits on Route 53 Resolver</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTION_NEEDED</code>: This endpoint is unhealthy, and Resolver can't automatically recover it.
         *        To resolve the problem, we recommend that you check each IP address that you associated with the
         *        endpoint. For each IP address that isn't available, add another IP address and then delete the IP
         *        address that isn't available. (An endpoint must always include at least two IP addresses.) A status of
         *        <code>ACTION_NEEDED</code> can have a variety of causes. Here are two common causes:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        One or more of the network interfaces that are associated with the endpoint were deleted using Amazon
         *        VPC.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The network interface couldn't be created for some reason that's outside the control of Resolver.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code>: Resolver is deleting this endpoint and the associated network interfaces.
         *        </p>
         *        </li>
         * @see ResolverEndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverEndpointStatus
         */
        Builder status(ResolverEndpointStatus status);

        /**
         * <p>
         * A detailed description of the status of the Resolver endpoint.
         * </p>
         * 
         * @param statusMessage
         *        A detailed description of the status of the Resolver endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * The date and time that the endpoint was created, in Unix time format and Coordinated Universal Time (UTC).
         * </p>
         * 
         * @param creationTime
         *        The date and time that the endpoint was created, in Unix time format and Coordinated Universal Time
         *        (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(String creationTime);

        /**
         * <p>
         * The date and time that the endpoint was last modified, in Unix time format and Coordinated Universal Time
         * (UTC).
         * </p>
         * 
         * @param modificationTime
         *        The date and time that the endpoint was last modified, in Unix time format and Coordinated Universal
         *        Time (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modificationTime(String modificationTime);

        /**
         * <p>
         * The Resolver endpoint IP address type.
         * </p>
         * 
         * @param resolverEndpointType
         *        The Resolver endpoint IP address type.
         * @see ResolverEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverEndpointType
         */
        Builder resolverEndpointType(String resolverEndpointType);

        /**
         * <p>
         * The Resolver endpoint IP address type.
         * </p>
         * 
         * @param resolverEndpointType
         *        The Resolver endpoint IP address type.
         * @see ResolverEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverEndpointType
         */
        Builder resolverEndpointType(ResolverEndpointType resolverEndpointType);

        /**
         * <p>
         * The ARN (Amazon Resource Name) for the Outpost.
         * </p>
         * 
         * @param outpostArn
         *        The ARN (Amazon Resource Name) for the Outpost.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outpostArn(String outpostArn);

        /**
         * <p>
         * The Amazon EC2 instance type.
         * </p>
         * 
         * @param preferredInstanceType
         *        The Amazon EC2 instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredInstanceType(String preferredInstanceType);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String creatorRequestId;

        private String arn;

        private String name;

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private String direction;

        private Integer ipAddressCount;

        private String hostVPCId;

        private String status;

        private String statusMessage;

        private String creationTime;

        private String modificationTime;

        private String resolverEndpointType;

        private String outpostArn;

        private String preferredInstanceType;

        private BuilderImpl() {
        }

        private BuilderImpl(ResolverEndpoint model) {
            id(model.id);
            creatorRequestId(model.creatorRequestId);
            arn(model.arn);
            name(model.name);
            securityGroupIds(model.securityGroupIds);
            direction(model.direction);
            ipAddressCount(model.ipAddressCount);
            hostVPCId(model.hostVPCId);
            status(model.status);
            statusMessage(model.statusMessage);
            creationTime(model.creationTime);
            modificationTime(model.modificationTime);
            resolverEndpointType(model.resolverEndpointType);
            outpostArn(model.outpostArn);
            preferredInstanceType(model.preferredInstanceType);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getCreatorRequestId() {
            return creatorRequestId;
        }

        public final void setCreatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
        }

        @Override
        public final Builder creatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final String getDirection() {
            return direction;
        }

        public final void setDirection(String direction) {
            this.direction = direction;
        }

        @Override
        public final Builder direction(String direction) {
            this.direction = direction;
            return this;
        }

        @Override
        public final Builder direction(ResolverEndpointDirection direction) {
            this.direction(direction == null ? null : direction.toString());
            return this;
        }

        public final Integer getIpAddressCount() {
            return ipAddressCount;
        }

        public final void setIpAddressCount(Integer ipAddressCount) {
            this.ipAddressCount = ipAddressCount;
        }

        @Override
        public final Builder ipAddressCount(Integer ipAddressCount) {
            this.ipAddressCount = ipAddressCount;
            return this;
        }

        public final String getHostVPCId() {
            return hostVPCId;
        }

        public final void setHostVPCId(String hostVPCId) {
            this.hostVPCId = hostVPCId;
        }

        @Override
        public final Builder hostVPCId(String hostVPCId) {
            this.hostVPCId = hostVPCId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ResolverEndpointStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final String getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(String creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(String creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getModificationTime() {
            return modificationTime;
        }

        public final void setModificationTime(String modificationTime) {
            this.modificationTime = modificationTime;
        }

        @Override
        public final Builder modificationTime(String modificationTime) {
            this.modificationTime = modificationTime;
            return this;
        }

        public final String getResolverEndpointType() {
            return resolverEndpointType;
        }

        public final void setResolverEndpointType(String resolverEndpointType) {
            this.resolverEndpointType = resolverEndpointType;
        }

        @Override
        public final Builder resolverEndpointType(String resolverEndpointType) {
            this.resolverEndpointType = resolverEndpointType;
            return this;
        }

        @Override
        public final Builder resolverEndpointType(ResolverEndpointType resolverEndpointType) {
            this.resolverEndpointType(resolverEndpointType == null ? null : resolverEndpointType.toString());
            return this;
        }

        public final String getOutpostArn() {
            return outpostArn;
        }

        public final void setOutpostArn(String outpostArn) {
            this.outpostArn = outpostArn;
        }

        @Override
        public final Builder outpostArn(String outpostArn) {
            this.outpostArn = outpostArn;
            return this;
        }

        public final String getPreferredInstanceType() {
            return preferredInstanceType;
        }

        public final void setPreferredInstanceType(String preferredInstanceType) {
            this.preferredInstanceType = preferredInstanceType;
        }

        @Override
        public final Builder preferredInstanceType(String preferredInstanceType) {
            this.preferredInstanceType = preferredInstanceType;
            return this;
        }

        @Override
        public ResolverEndpoint build() {
            return new ResolverEndpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
