/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Container for the objects to delete.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Delete implements SdkPojo, Serializable, ToCopyableBuilder<Delete.Builder, Delete> {
    private static final SdkField<List<ObjectIdentifier>> OBJECTS_FIELD = SdkField
            .<List<ObjectIdentifier>> builder(MarshallingType.LIST)
            .getter(getter(Delete::objects))
            .setter(setter(Builder::objects))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Object")
                    .unmarshallLocationName("Object").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ObjectIdentifier> builder(MarshallingType.SDK_POJO)
                                            .constructor(ObjectIdentifier::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .isFlattened(true).build()).build();

    private static final SdkField<Boolean> QUIET_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(Delete::quiet))
            .setter(setter(Builder::quiet))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Quiet")
                    .unmarshallLocationName("Quiet").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OBJECTS_FIELD, QUIET_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<ObjectIdentifier> objects;

    private final Boolean quiet;

    private Delete(BuilderImpl builder) {
        this.objects = builder.objects;
        this.quiet = builder.quiet;
    }

    /**
     * Returns true if the Objects property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasObjects() {
        return objects != null && !(objects instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The objects to delete.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasObjects()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The objects to delete.
     */
    public List<ObjectIdentifier> objects() {
        return objects;
    }

    /**
     * <p>
     * Element to enable quiet mode for the request. When you add this element, you must set its value to true.
     * </p>
     * 
     * @return Element to enable quiet mode for the request. When you add this element, you must set its value to true.
     */
    public Boolean quiet() {
        return quiet;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(objects());
        hashCode = 31 * hashCode + Objects.hashCode(quiet());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Delete)) {
            return false;
        }
        Delete other = (Delete) obj;
        return Objects.equals(objects(), other.objects()) && Objects.equals(quiet(), other.quiet());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Delete").add("Objects", objects()).add("Quiet", quiet()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Objects":
            return Optional.ofNullable(clazz.cast(objects()));
        case "Quiet":
            return Optional.ofNullable(clazz.cast(quiet()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Delete, T> g) {
        return obj -> g.apply((Delete) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Delete> {
        /**
         * <p>
         * The objects to delete.
         * </p>
         * 
         * @param objects
         *        The objects to delete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objects(Collection<ObjectIdentifier> objects);

        /**
         * <p>
         * The objects to delete.
         * </p>
         * 
         * @param objects
         *        The objects to delete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objects(ObjectIdentifier... objects);

        /**
         * <p>
         * The objects to delete.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ObjectIdentifier>.Builder} avoiding the
         * need to create one manually via {@link List<ObjectIdentifier>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ObjectIdentifier>.Builder#build()} is called immediately and
         * its result is passed to {@link #objects(List<ObjectIdentifier>)}.
         * 
         * @param objects
         *        a consumer that will call methods on {@link List<ObjectIdentifier>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #objects(List<ObjectIdentifier>)
         */
        Builder objects(Consumer<ObjectIdentifier.Builder>... objects);

        /**
         * <p>
         * Element to enable quiet mode for the request. When you add this element, you must set its value to true.
         * </p>
         * 
         * @param quiet
         *        Element to enable quiet mode for the request. When you add this element, you must set its value to
         *        true.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quiet(Boolean quiet);
    }

    static final class BuilderImpl implements Builder {
        private List<ObjectIdentifier> objects = DefaultSdkAutoConstructList.getInstance();

        private Boolean quiet;

        private BuilderImpl() {
        }

        private BuilderImpl(Delete model) {
            objects(model.objects);
            quiet(model.quiet);
        }

        public final Collection<ObjectIdentifier.Builder> getObjects() {
            return objects != null ? objects.stream().map(ObjectIdentifier::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder objects(Collection<ObjectIdentifier> objects) {
            this.objects = ObjectIdentifierListCopier.copy(objects);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder objects(ObjectIdentifier... objects) {
            objects(Arrays.asList(objects));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder objects(Consumer<ObjectIdentifier.Builder>... objects) {
            objects(Stream.of(objects).map(c -> ObjectIdentifier.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setObjects(Collection<ObjectIdentifier.BuilderImpl> objects) {
            this.objects = ObjectIdentifierListCopier.copyFromBuilder(objects);
        }

        public final Boolean getQuiet() {
            return quiet;
        }

        @Override
        public final Builder quiet(Boolean quiet) {
            this.quiet = quiet;
            return this;
        }

        public final void setQuiet(Boolean quiet) {
            this.quiet = quiet;
        }

        @Override
        public Delete build() {
            return new Delete(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
