/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the configuration parameters for a POST Object restore job. S3 Batch Operations passes every object to the
 * underlying <code>RestoreObject</code> API operation. For more information about the parameters for this operation,
 * see <a href=
 * "https://docs.aws.amazon.com/AmazonS3/latest/API/RESTObjectPOSTrestore.html#RESTObjectPOSTrestore-restore-request"
 * >RestoreObject</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3InitiateRestoreObjectOperation implements SdkPojo, Serializable,
        ToCopyableBuilder<S3InitiateRestoreObjectOperation.Builder, S3InitiateRestoreObjectOperation> {
    private static final SdkField<Integer> EXPIRATION_IN_DAYS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("ExpirationInDays")
            .getter(getter(S3InitiateRestoreObjectOperation::expirationInDays))
            .setter(setter(Builder::expirationInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpirationInDays")
                    .unmarshallLocationName("ExpirationInDays").build()).build();

    private static final SdkField<String> GLACIER_JOB_TIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("GlacierJobTier")
            .getter(getter(S3InitiateRestoreObjectOperation::glacierJobTierAsString))
            .setter(setter(Builder::glacierJobTier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GlacierJobTier")
                    .unmarshallLocationName("GlacierJobTier").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPIRATION_IN_DAYS_FIELD,
            GLACIER_JOB_TIER_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer expirationInDays;

    private final String glacierJobTier;

    private S3InitiateRestoreObjectOperation(BuilderImpl builder) {
        this.expirationInDays = builder.expirationInDays;
        this.glacierJobTier = builder.glacierJobTier;
    }

    /**
     * <p>
     * This argument specifies how long the S3 Glacier or S3 Glacier Deep Archive object remains available in Amazon S3.
     * S3 Initiate Restore Object jobs that target S3 Glacier and S3 Glacier Deep Archive objects require
     * <code>ExpirationInDays</code> set to 1 or greater.
     * </p>
     * <p>
     * Conversely, do <i>not</i> set <code>ExpirationInDays</code> when creating S3 Initiate Restore Object jobs that
     * target S3 Intelligent-Tiering Archive Access and Deep Archive Access tier objects. Objects in S3
     * Intelligent-Tiering archive access tiers are not subject to restore expiry, so specifying
     * <code>ExpirationInDays</code> results in restore request failure.
     * </p>
     * <p>
     * S3 Batch Operations jobs can operate either on S3 Glacier and S3 Glacier Deep Archive storage class objects or on
     * S3 Intelligent-Tiering Archive Access and Deep Archive Access storage tier objects, but not both types in the
     * same job. If you need to restore objects of both types you <i>must</i> create separate Batch Operations jobs.
     * </p>
     * 
     * @return This argument specifies how long the S3 Glacier or S3 Glacier Deep Archive object remains available in
     *         Amazon S3. S3 Initiate Restore Object jobs that target S3 Glacier and S3 Glacier Deep Archive objects
     *         require <code>ExpirationInDays</code> set to 1 or greater.</p>
     *         <p>
     *         Conversely, do <i>not</i> set <code>ExpirationInDays</code> when creating S3 Initiate Restore Object jobs
     *         that target S3 Intelligent-Tiering Archive Access and Deep Archive Access tier objects. Objects in S3
     *         Intelligent-Tiering archive access tiers are not subject to restore expiry, so specifying
     *         <code>ExpirationInDays</code> results in restore request failure.
     *         </p>
     *         <p>
     *         S3 Batch Operations jobs can operate either on S3 Glacier and S3 Glacier Deep Archive storage class
     *         objects or on S3 Intelligent-Tiering Archive Access and Deep Archive Access storage tier objects, but not
     *         both types in the same job. If you need to restore objects of both types you <i>must</i> create separate
     *         Batch Operations jobs.
     */
    public final Integer expirationInDays() {
        return expirationInDays;
    }

    /**
     * <p>
     * S3 Batch Operations supports <code>STANDARD</code> and <code>BULK</code> retrieval tiers, but not the
     * <code>EXPEDITED</code> retrieval tier.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #glacierJobTier}
     * will return {@link S3GlacierJobTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #glacierJobTierAsString}.
     * </p>
     * 
     * @return S3 Batch Operations supports <code>STANDARD</code> and <code>BULK</code> retrieval tiers, but not the
     *         <code>EXPEDITED</code> retrieval tier.
     * @see S3GlacierJobTier
     */
    public final S3GlacierJobTier glacierJobTier() {
        return S3GlacierJobTier.fromValue(glacierJobTier);
    }

    /**
     * <p>
     * S3 Batch Operations supports <code>STANDARD</code> and <code>BULK</code> retrieval tiers, but not the
     * <code>EXPEDITED</code> retrieval tier.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #glacierJobTier}
     * will return {@link S3GlacierJobTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #glacierJobTierAsString}.
     * </p>
     * 
     * @return S3 Batch Operations supports <code>STANDARD</code> and <code>BULK</code> retrieval tiers, but not the
     *         <code>EXPEDITED</code> retrieval tier.
     * @see S3GlacierJobTier
     */
    public final String glacierJobTierAsString() {
        return glacierJobTier;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(expirationInDays());
        hashCode = 31 * hashCode + Objects.hashCode(glacierJobTierAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3InitiateRestoreObjectOperation)) {
            return false;
        }
        S3InitiateRestoreObjectOperation other = (S3InitiateRestoreObjectOperation) obj;
        return Objects.equals(expirationInDays(), other.expirationInDays())
                && Objects.equals(glacierJobTierAsString(), other.glacierJobTierAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3InitiateRestoreObjectOperation").add("ExpirationInDays", expirationInDays())
                .add("GlacierJobTier", glacierJobTierAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ExpirationInDays":
            return Optional.ofNullable(clazz.cast(expirationInDays()));
        case "GlacierJobTier":
            return Optional.ofNullable(clazz.cast(glacierJobTierAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3InitiateRestoreObjectOperation, T> g) {
        return obj -> g.apply((S3InitiateRestoreObjectOperation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3InitiateRestoreObjectOperation> {
        /**
         * <p>
         * This argument specifies how long the S3 Glacier or S3 Glacier Deep Archive object remains available in Amazon
         * S3. S3 Initiate Restore Object jobs that target S3 Glacier and S3 Glacier Deep Archive objects require
         * <code>ExpirationInDays</code> set to 1 or greater.
         * </p>
         * <p>
         * Conversely, do <i>not</i> set <code>ExpirationInDays</code> when creating S3 Initiate Restore Object jobs
         * that target S3 Intelligent-Tiering Archive Access and Deep Archive Access tier objects. Objects in S3
         * Intelligent-Tiering archive access tiers are not subject to restore expiry, so specifying
         * <code>ExpirationInDays</code> results in restore request failure.
         * </p>
         * <p>
         * S3 Batch Operations jobs can operate either on S3 Glacier and S3 Glacier Deep Archive storage class objects
         * or on S3 Intelligent-Tiering Archive Access and Deep Archive Access storage tier objects, but not both types
         * in the same job. If you need to restore objects of both types you <i>must</i> create separate Batch
         * Operations jobs.
         * </p>
         * 
         * @param expirationInDays
         *        This argument specifies how long the S3 Glacier or S3 Glacier Deep Archive object remains available in
         *        Amazon S3. S3 Initiate Restore Object jobs that target S3 Glacier and S3 Glacier Deep Archive objects
         *        require <code>ExpirationInDays</code> set to 1 or greater.</p>
         *        <p>
         *        Conversely, do <i>not</i> set <code>ExpirationInDays</code> when creating S3 Initiate Restore Object
         *        jobs that target S3 Intelligent-Tiering Archive Access and Deep Archive Access tier objects. Objects
         *        in S3 Intelligent-Tiering archive access tiers are not subject to restore expiry, so specifying
         *        <code>ExpirationInDays</code> results in restore request failure.
         *        </p>
         *        <p>
         *        S3 Batch Operations jobs can operate either on S3 Glacier and S3 Glacier Deep Archive storage class
         *        objects or on S3 Intelligent-Tiering Archive Access and Deep Archive Access storage tier objects, but
         *        not both types in the same job. If you need to restore objects of both types you <i>must</i> create
         *        separate Batch Operations jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expirationInDays(Integer expirationInDays);

        /**
         * <p>
         * S3 Batch Operations supports <code>STANDARD</code> and <code>BULK</code> retrieval tiers, but not the
         * <code>EXPEDITED</code> retrieval tier.
         * </p>
         * 
         * @param glacierJobTier
         *        S3 Batch Operations supports <code>STANDARD</code> and <code>BULK</code> retrieval tiers, but not the
         *        <code>EXPEDITED</code> retrieval tier.
         * @see S3GlacierJobTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3GlacierJobTier
         */
        Builder glacierJobTier(String glacierJobTier);

        /**
         * <p>
         * S3 Batch Operations supports <code>STANDARD</code> and <code>BULK</code> retrieval tiers, but not the
         * <code>EXPEDITED</code> retrieval tier.
         * </p>
         * 
         * @param glacierJobTier
         *        S3 Batch Operations supports <code>STANDARD</code> and <code>BULK</code> retrieval tiers, but not the
         *        <code>EXPEDITED</code> retrieval tier.
         * @see S3GlacierJobTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3GlacierJobTier
         */
        Builder glacierJobTier(S3GlacierJobTier glacierJobTier);
    }

    static final class BuilderImpl implements Builder {
        private Integer expirationInDays;

        private String glacierJobTier;

        private BuilderImpl() {
        }

        private BuilderImpl(S3InitiateRestoreObjectOperation model) {
            expirationInDays(model.expirationInDays);
            glacierJobTier(model.glacierJobTier);
        }

        public final Integer getExpirationInDays() {
            return expirationInDays;
        }

        public final void setExpirationInDays(Integer expirationInDays) {
            this.expirationInDays = expirationInDays;
        }

        @Override
        public final Builder expirationInDays(Integer expirationInDays) {
            this.expirationInDays = expirationInDays;
            return this;
        }

        public final String getGlacierJobTier() {
            return glacierJobTier;
        }

        public final void setGlacierJobTier(String glacierJobTier) {
            this.glacierJobTier = glacierJobTier;
        }

        @Override
        public final Builder glacierJobTier(String glacierJobTier) {
            this.glacierJobTier = glacierJobTier;
            return this;
        }

        @Override
        public final Builder glacierJobTier(S3GlacierJobTier glacierJobTier) {
            this.glacierJobTier(glacierJobTier == null ? null : glacierJobTier.toString());
            return this;
        }

        @Override
        public S3InitiateRestoreObjectOperation build() {
            return new S3InitiateRestoreObjectOperation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
