/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateModelPackageRequest extends SageMakerRequest implements
        ToCopyableBuilder<CreateModelPackageRequest.Builder, CreateModelPackageRequest> {
    private static final SdkField<String> MODEL_PACKAGE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelPackageName").getter(getter(CreateModelPackageRequest::modelPackageName))
            .setter(setter(Builder::modelPackageName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelPackageName").build()).build();

    private static final SdkField<String> MODEL_PACKAGE_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelPackageGroupName").getter(getter(CreateModelPackageRequest::modelPackageGroupName))
            .setter(setter(Builder::modelPackageGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelPackageGroupName").build())
            .build();

    private static final SdkField<String> MODEL_PACKAGE_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelPackageDescription").getter(getter(CreateModelPackageRequest::modelPackageDescription))
            .setter(setter(Builder::modelPackageDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelPackageDescription").build())
            .build();

    private static final SdkField<InferenceSpecification> INFERENCE_SPECIFICATION_FIELD = SdkField
            .<InferenceSpecification> builder(MarshallingType.SDK_POJO).memberName("InferenceSpecification")
            .getter(getter(CreateModelPackageRequest::inferenceSpecification)).setter(setter(Builder::inferenceSpecification))
            .constructor(InferenceSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InferenceSpecification").build())
            .build();

    private static final SdkField<ModelPackageValidationSpecification> VALIDATION_SPECIFICATION_FIELD = SdkField
            .<ModelPackageValidationSpecification> builder(MarshallingType.SDK_POJO).memberName("ValidationSpecification")
            .getter(getter(CreateModelPackageRequest::validationSpecification)).setter(setter(Builder::validationSpecification))
            .constructor(ModelPackageValidationSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidationSpecification").build())
            .build();

    private static final SdkField<SourceAlgorithmSpecification> SOURCE_ALGORITHM_SPECIFICATION_FIELD = SdkField
            .<SourceAlgorithmSpecification> builder(MarshallingType.SDK_POJO)
            .memberName("SourceAlgorithmSpecification")
            .getter(getter(CreateModelPackageRequest::sourceAlgorithmSpecification))
            .setter(setter(Builder::sourceAlgorithmSpecification))
            .constructor(SourceAlgorithmSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceAlgorithmSpecification")
                    .build()).build();

    private static final SdkField<Boolean> CERTIFY_FOR_MARKETPLACE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CertifyForMarketplace").getter(getter(CreateModelPackageRequest::certifyForMarketplace))
            .setter(setter(Builder::certifyForMarketplace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertifyForMarketplace").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateModelPackageRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> MODEL_APPROVAL_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelApprovalStatus").getter(getter(CreateModelPackageRequest::modelApprovalStatusAsString))
            .setter(setter(Builder::modelApprovalStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelApprovalStatus").build())
            .build();

    private static final SdkField<MetadataProperties> METADATA_PROPERTIES_FIELD = SdkField
            .<MetadataProperties> builder(MarshallingType.SDK_POJO).memberName("MetadataProperties")
            .getter(getter(CreateModelPackageRequest::metadataProperties)).setter(setter(Builder::metadataProperties))
            .constructor(MetadataProperties::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetadataProperties").build())
            .build();

    private static final SdkField<ModelMetrics> MODEL_METRICS_FIELD = SdkField.<ModelMetrics> builder(MarshallingType.SDK_POJO)
            .memberName("ModelMetrics").getter(getter(CreateModelPackageRequest::modelMetrics))
            .setter(setter(Builder::modelMetrics)).constructor(ModelMetrics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelMetrics").build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(CreateModelPackageRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<Map<String, String>> CUSTOMER_METADATA_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("CustomerMetadataProperties")
            .getter(getter(CreateModelPackageRequest::customerMetadataProperties))
            .setter(setter(Builder::customerMetadataProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerMetadataProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<DriftCheckBaselines> DRIFT_CHECK_BASELINES_FIELD = SdkField
            .<DriftCheckBaselines> builder(MarshallingType.SDK_POJO).memberName("DriftCheckBaselines")
            .getter(getter(CreateModelPackageRequest::driftCheckBaselines)).setter(setter(Builder::driftCheckBaselines))
            .constructor(DriftCheckBaselines::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DriftCheckBaselines").build())
            .build();

    private static final SdkField<String> DOMAIN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Domain")
            .getter(getter(CreateModelPackageRequest::domain)).setter(setter(Builder::domain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Domain").build()).build();

    private static final SdkField<String> TASK_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Task")
            .getter(getter(CreateModelPackageRequest::task)).setter(setter(Builder::task))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Task").build()).build();

    private static final SdkField<String> SAMPLE_PAYLOAD_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SamplePayloadUrl").getter(getter(CreateModelPackageRequest::samplePayloadUrl))
            .setter(setter(Builder::samplePayloadUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SamplePayloadUrl").build()).build();

    private static final SdkField<List<AdditionalInferenceSpecificationDefinition>> ADDITIONAL_INFERENCE_SPECIFICATIONS_FIELD = SdkField
            .<List<AdditionalInferenceSpecificationDefinition>> builder(MarshallingType.LIST)
            .memberName("AdditionalInferenceSpecifications")
            .getter(getter(CreateModelPackageRequest::additionalInferenceSpecifications))
            .setter(setter(Builder::additionalInferenceSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalInferenceSpecifications")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AdditionalInferenceSpecificationDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(AdditionalInferenceSpecificationDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MODEL_PACKAGE_NAME_FIELD,
            MODEL_PACKAGE_GROUP_NAME_FIELD, MODEL_PACKAGE_DESCRIPTION_FIELD, INFERENCE_SPECIFICATION_FIELD,
            VALIDATION_SPECIFICATION_FIELD, SOURCE_ALGORITHM_SPECIFICATION_FIELD, CERTIFY_FOR_MARKETPLACE_FIELD, TAGS_FIELD,
            MODEL_APPROVAL_STATUS_FIELD, METADATA_PROPERTIES_FIELD, MODEL_METRICS_FIELD, CLIENT_TOKEN_FIELD,
            CUSTOMER_METADATA_PROPERTIES_FIELD, DRIFT_CHECK_BASELINES_FIELD, DOMAIN_FIELD, TASK_FIELD, SAMPLE_PAYLOAD_URL_FIELD,
            ADDITIONAL_INFERENCE_SPECIFICATIONS_FIELD));

    private final String modelPackageName;

    private final String modelPackageGroupName;

    private final String modelPackageDescription;

    private final InferenceSpecification inferenceSpecification;

    private final ModelPackageValidationSpecification validationSpecification;

    private final SourceAlgorithmSpecification sourceAlgorithmSpecification;

    private final Boolean certifyForMarketplace;

    private final List<Tag> tags;

    private final String modelApprovalStatus;

    private final MetadataProperties metadataProperties;

    private final ModelMetrics modelMetrics;

    private final String clientToken;

    private final Map<String, String> customerMetadataProperties;

    private final DriftCheckBaselines driftCheckBaselines;

    private final String domain;

    private final String task;

    private final String samplePayloadUrl;

    private final List<AdditionalInferenceSpecificationDefinition> additionalInferenceSpecifications;

    private CreateModelPackageRequest(BuilderImpl builder) {
        super(builder);
        this.modelPackageName = builder.modelPackageName;
        this.modelPackageGroupName = builder.modelPackageGroupName;
        this.modelPackageDescription = builder.modelPackageDescription;
        this.inferenceSpecification = builder.inferenceSpecification;
        this.validationSpecification = builder.validationSpecification;
        this.sourceAlgorithmSpecification = builder.sourceAlgorithmSpecification;
        this.certifyForMarketplace = builder.certifyForMarketplace;
        this.tags = builder.tags;
        this.modelApprovalStatus = builder.modelApprovalStatus;
        this.metadataProperties = builder.metadataProperties;
        this.modelMetrics = builder.modelMetrics;
        this.clientToken = builder.clientToken;
        this.customerMetadataProperties = builder.customerMetadataProperties;
        this.driftCheckBaselines = builder.driftCheckBaselines;
        this.domain = builder.domain;
        this.task = builder.task;
        this.samplePayloadUrl = builder.samplePayloadUrl;
        this.additionalInferenceSpecifications = builder.additionalInferenceSpecifications;
    }

    /**
     * <p>
     * The name of the model package. The name must have 1 to 63 characters. Valid characters are a-z, A-Z, 0-9, and -
     * (hyphen).
     * </p>
     * <p>
     * This parameter is required for unversioned models. It is not applicable to versioned models.
     * </p>
     * 
     * @return The name of the model package. The name must have 1 to 63 characters. Valid characters are a-z, A-Z, 0-9,
     *         and - (hyphen).</p>
     *         <p>
     *         This parameter is required for unversioned models. It is not applicable to versioned models.
     */
    public final String modelPackageName() {
        return modelPackageName;
    }

    /**
     * <p>
     * The name or Amazon Resource Name (ARN) of the model package group that this model version belongs to.
     * </p>
     * <p>
     * This parameter is required for versioned models, and does not apply to unversioned models.
     * </p>
     * 
     * @return The name or Amazon Resource Name (ARN) of the model package group that this model version belongs to.</p>
     *         <p>
     *         This parameter is required for versioned models, and does not apply to unversioned models.
     */
    public final String modelPackageGroupName() {
        return modelPackageGroupName;
    }

    /**
     * <p>
     * A description of the model package.
     * </p>
     * 
     * @return A description of the model package.
     */
    public final String modelPackageDescription() {
        return modelPackageDescription;
    }

    /**
     * <p>
     * Specifies details about inference jobs that can be run with models based on this model package, including the
     * following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The Amazon ECR paths of containers that contain the inference code and model artifacts.
     * </p>
     * </li>
     * <li>
     * <p>
     * The instance types that the model package supports for transform jobs and real-time endpoints used for inference.
     * </p>
     * </li>
     * <li>
     * <p>
     * The input and output content formats that the model package supports for inference.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies details about inference jobs that can be run with models based on this model package, including
     *         the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The Amazon ECR paths of containers that contain the inference code and model artifacts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The instance types that the model package supports for transform jobs and real-time endpoints used for
     *         inference.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The input and output content formats that the model package supports for inference.
     *         </p>
     *         </li>
     */
    public final InferenceSpecification inferenceSpecification() {
        return inferenceSpecification;
    }

    /**
     * <p>
     * Specifies configurations for one or more transform jobs that SageMaker runs to test the model package.
     * </p>
     * 
     * @return Specifies configurations for one or more transform jobs that SageMaker runs to test the model package.
     */
    public final ModelPackageValidationSpecification validationSpecification() {
        return validationSpecification;
    }

    /**
     * <p>
     * Details about the algorithm that was used to create the model package.
     * </p>
     * 
     * @return Details about the algorithm that was used to create the model package.
     */
    public final SourceAlgorithmSpecification sourceAlgorithmSpecification() {
        return sourceAlgorithmSpecification;
    }

    /**
     * <p>
     * Whether to certify the model package for listing on Amazon Web Services Marketplace.
     * </p>
     * <p>
     * This parameter is optional for unversioned models, and does not apply to versioned models.
     * </p>
     * 
     * @return Whether to certify the model package for listing on Amazon Web Services Marketplace.</p>
     *         <p>
     *         This parameter is optional for unversioned models, and does not apply to versioned models.
     */
    public final Boolean certifyForMarketplace() {
        return certifyForMarketplace;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of key value pairs associated with the model. For more information, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services resources</a>
     * in the <i>Amazon Web Services General Reference Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of key value pairs associated with the model. For more information, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
     *         resources</a> in the <i>Amazon Web Services General Reference Guide</i>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Whether the model is approved for deployment.
     * </p>
     * <p>
     * This parameter is optional for versioned models, and does not apply to unversioned models.
     * </p>
     * <p>
     * For versioned models, the value of this parameter must be set to <code>Approved</code> to deploy the model.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #modelApprovalStatus} will return {@link ModelApprovalStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #modelApprovalStatusAsString}.
     * </p>
     * 
     * @return Whether the model is approved for deployment.</p>
     *         <p>
     *         This parameter is optional for versioned models, and does not apply to unversioned models.
     *         </p>
     *         <p>
     *         For versioned models, the value of this parameter must be set to <code>Approved</code> to deploy the
     *         model.
     * @see ModelApprovalStatus
     */
    public final ModelApprovalStatus modelApprovalStatus() {
        return ModelApprovalStatus.fromValue(modelApprovalStatus);
    }

    /**
     * <p>
     * Whether the model is approved for deployment.
     * </p>
     * <p>
     * This parameter is optional for versioned models, and does not apply to unversioned models.
     * </p>
     * <p>
     * For versioned models, the value of this parameter must be set to <code>Approved</code> to deploy the model.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #modelApprovalStatus} will return {@link ModelApprovalStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #modelApprovalStatusAsString}.
     * </p>
     * 
     * @return Whether the model is approved for deployment.</p>
     *         <p>
     *         This parameter is optional for versioned models, and does not apply to unversioned models.
     *         </p>
     *         <p>
     *         For versioned models, the value of this parameter must be set to <code>Approved</code> to deploy the
     *         model.
     * @see ModelApprovalStatus
     */
    public final String modelApprovalStatusAsString() {
        return modelApprovalStatus;
    }

    /**
     * Returns the value of the MetadataProperties property for this object.
     * 
     * @return The value of the MetadataProperties property for this object.
     */
    public final MetadataProperties metadataProperties() {
        return metadataProperties;
    }

    /**
     * <p>
     * A structure that contains model metrics reports.
     * </p>
     * 
     * @return A structure that contains model metrics reports.
     */
    public final ModelMetrics modelMetrics() {
        return modelMetrics;
    }

    /**
     * <p>
     * A unique token that guarantees that the call to this API is idempotent.
     * </p>
     * 
     * @return A unique token that guarantees that the call to this API is idempotent.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomerMetadataProperties property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCustomerMetadataProperties() {
        return customerMetadataProperties != null && !(customerMetadataProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The metadata properties associated with the model package versions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomerMetadataProperties} method.
     * </p>
     * 
     * @return The metadata properties associated with the model package versions.
     */
    public final Map<String, String> customerMetadataProperties() {
        return customerMetadataProperties;
    }

    /**
     * <p>
     * Represents the drift check baselines that can be used when the model monitor is set using the model package. For
     * more information, see the topic on <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/dg/pipelines-quality-clarify-baseline-lifecycle.html#pipelines-quality-clarify-baseline-drift-detection"
     * >Drift Detection against Previous Baselines in SageMaker Pipelines</a> in the <i>Amazon SageMaker Developer
     * Guide</i>.
     * </p>
     * 
     * @return Represents the drift check baselines that can be used when the model monitor is set using the model
     *         package. For more information, see the topic on <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/pipelines-quality-clarify-baseline-lifecycle.html#pipelines-quality-clarify-baseline-drift-detection"
     *         >Drift Detection against Previous Baselines in SageMaker Pipelines</a> in the <i>Amazon SageMaker
     *         Developer Guide</i>.
     */
    public final DriftCheckBaselines driftCheckBaselines() {
        return driftCheckBaselines;
    }

    /**
     * <p>
     * The machine learning domain of your model package and its components. Common machine learning domains include
     * computer vision and natural language processing.
     * </p>
     * 
     * @return The machine learning domain of your model package and its components. Common machine learning domains
     *         include computer vision and natural language processing.
     */
    public final String domain() {
        return domain;
    }

    /**
     * <p>
     * The machine learning task your model package accomplishes. Common machine learning tasks include object detection
     * and image classification. The following tasks are supported by Inference Recommender:
     * <code>"IMAGE_CLASSIFICATION"</code> | <code>"OBJECT_DETECTION"</code> | <code>"TEXT_GENERATION"</code> |
     * <code>"IMAGE_SEGMENTATION"</code> | <code>"FILL_MASK"</code> | <code>"CLASSIFICATION"</code> |
     * <code>"REGRESSION"</code> | <code>"OTHER"</code>.
     * </p>
     * <p>
     * Specify "OTHER" if none of the tasks listed fit your use case.
     * </p>
     * 
     * @return The machine learning task your model package accomplishes. Common machine learning tasks include object
     *         detection and image classification. The following tasks are supported by Inference Recommender:
     *         <code>"IMAGE_CLASSIFICATION"</code> | <code>"OBJECT_DETECTION"</code> | <code>"TEXT_GENERATION"</code> |
     *         <code>"IMAGE_SEGMENTATION"</code> | <code>"FILL_MASK"</code> | <code>"CLASSIFICATION"</code> |
     *         <code>"REGRESSION"</code> | <code>"OTHER"</code>.</p>
     *         <p>
     *         Specify "OTHER" if none of the tasks listed fit your use case.
     */
    public final String task() {
        return task;
    }

    /**
     * <p>
     * The Amazon Simple Storage Service (Amazon S3) path where the sample payload are stored. This path must point to a
     * single gzip compressed tar archive (.tar.gz suffix).
     * </p>
     * 
     * @return The Amazon Simple Storage Service (Amazon S3) path where the sample payload are stored. This path must
     *         point to a single gzip compressed tar archive (.tar.gz suffix).
     */
    public final String samplePayloadUrl() {
        return samplePayloadUrl;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdditionalInferenceSpecifications
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasAdditionalInferenceSpecifications() {
        return additionalInferenceSpecifications != null && !(additionalInferenceSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of additional Inference Specification objects. Each additional Inference Specification specifies
     * artifacts based on this model package that can be used on inference endpoints. Generally used with SageMaker Neo
     * to store the compiled artifacts.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdditionalInferenceSpecifications}
     * method.
     * </p>
     * 
     * @return An array of additional Inference Specification objects. Each additional Inference Specification specifies
     *         artifacts based on this model package that can be used on inference endpoints. Generally used with
     *         SageMaker Neo to store the compiled artifacts.
     */
    public final List<AdditionalInferenceSpecificationDefinition> additionalInferenceSpecifications() {
        return additionalInferenceSpecifications;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(modelPackageName());
        hashCode = 31 * hashCode + Objects.hashCode(modelPackageGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(modelPackageDescription());
        hashCode = 31 * hashCode + Objects.hashCode(inferenceSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(validationSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(sourceAlgorithmSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(certifyForMarketplace());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(modelApprovalStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(metadataProperties());
        hashCode = 31 * hashCode + Objects.hashCode(modelMetrics());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomerMetadataProperties() ? customerMetadataProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(driftCheckBaselines());
        hashCode = 31 * hashCode + Objects.hashCode(domain());
        hashCode = 31 * hashCode + Objects.hashCode(task());
        hashCode = 31 * hashCode + Objects.hashCode(samplePayloadUrl());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasAdditionalInferenceSpecifications() ? additionalInferenceSpecifications() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateModelPackageRequest)) {
            return false;
        }
        CreateModelPackageRequest other = (CreateModelPackageRequest) obj;
        return Objects.equals(modelPackageName(), other.modelPackageName())
                && Objects.equals(modelPackageGroupName(), other.modelPackageGroupName())
                && Objects.equals(modelPackageDescription(), other.modelPackageDescription())
                && Objects.equals(inferenceSpecification(), other.inferenceSpecification())
                && Objects.equals(validationSpecification(), other.validationSpecification())
                && Objects.equals(sourceAlgorithmSpecification(), other.sourceAlgorithmSpecification())
                && Objects.equals(certifyForMarketplace(), other.certifyForMarketplace()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(modelApprovalStatusAsString(), other.modelApprovalStatusAsString())
                && Objects.equals(metadataProperties(), other.metadataProperties())
                && Objects.equals(modelMetrics(), other.modelMetrics()) && Objects.equals(clientToken(), other.clientToken())
                && hasCustomerMetadataProperties() == other.hasCustomerMetadataProperties()
                && Objects.equals(customerMetadataProperties(), other.customerMetadataProperties())
                && Objects.equals(driftCheckBaselines(), other.driftCheckBaselines()) && Objects.equals(domain(), other.domain())
                && Objects.equals(task(), other.task()) && Objects.equals(samplePayloadUrl(), other.samplePayloadUrl())
                && hasAdditionalInferenceSpecifications() == other.hasAdditionalInferenceSpecifications()
                && Objects.equals(additionalInferenceSpecifications(), other.additionalInferenceSpecifications());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("CreateModelPackageRequest")
                .add("ModelPackageName", modelPackageName())
                .add("ModelPackageGroupName", modelPackageGroupName())
                .add("ModelPackageDescription", modelPackageDescription())
                .add("InferenceSpecification", inferenceSpecification())
                .add("ValidationSpecification", validationSpecification())
                .add("SourceAlgorithmSpecification", sourceAlgorithmSpecification())
                .add("CertifyForMarketplace", certifyForMarketplace())
                .add("Tags", hasTags() ? tags() : null)
                .add("ModelApprovalStatus", modelApprovalStatusAsString())
                .add("MetadataProperties", metadataProperties())
                .add("ModelMetrics", modelMetrics())
                .add("ClientToken", clientToken())
                .add("CustomerMetadataProperties", hasCustomerMetadataProperties() ? customerMetadataProperties() : null)
                .add("DriftCheckBaselines", driftCheckBaselines())
                .add("Domain", domain())
                .add("Task", task())
                .add("SamplePayloadUrl", samplePayloadUrl())
                .add("AdditionalInferenceSpecifications",
                        hasAdditionalInferenceSpecifications() ? additionalInferenceSpecifications() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ModelPackageName":
            return Optional.ofNullable(clazz.cast(modelPackageName()));
        case "ModelPackageGroupName":
            return Optional.ofNullable(clazz.cast(modelPackageGroupName()));
        case "ModelPackageDescription":
            return Optional.ofNullable(clazz.cast(modelPackageDescription()));
        case "InferenceSpecification":
            return Optional.ofNullable(clazz.cast(inferenceSpecification()));
        case "ValidationSpecification":
            return Optional.ofNullable(clazz.cast(validationSpecification()));
        case "SourceAlgorithmSpecification":
            return Optional.ofNullable(clazz.cast(sourceAlgorithmSpecification()));
        case "CertifyForMarketplace":
            return Optional.ofNullable(clazz.cast(certifyForMarketplace()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "ModelApprovalStatus":
            return Optional.ofNullable(clazz.cast(modelApprovalStatusAsString()));
        case "MetadataProperties":
            return Optional.ofNullable(clazz.cast(metadataProperties()));
        case "ModelMetrics":
            return Optional.ofNullable(clazz.cast(modelMetrics()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "CustomerMetadataProperties":
            return Optional.ofNullable(clazz.cast(customerMetadataProperties()));
        case "DriftCheckBaselines":
            return Optional.ofNullable(clazz.cast(driftCheckBaselines()));
        case "Domain":
            return Optional.ofNullable(clazz.cast(domain()));
        case "Task":
            return Optional.ofNullable(clazz.cast(task()));
        case "SamplePayloadUrl":
            return Optional.ofNullable(clazz.cast(samplePayloadUrl()));
        case "AdditionalInferenceSpecifications":
            return Optional.ofNullable(clazz.cast(additionalInferenceSpecifications()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateModelPackageRequest, T> g) {
        return obj -> g.apply((CreateModelPackageRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SageMakerRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateModelPackageRequest> {
        /**
         * <p>
         * The name of the model package. The name must have 1 to 63 characters. Valid characters are a-z, A-Z, 0-9, and
         * - (hyphen).
         * </p>
         * <p>
         * This parameter is required for unversioned models. It is not applicable to versioned models.
         * </p>
         * 
         * @param modelPackageName
         *        The name of the model package. The name must have 1 to 63 characters. Valid characters are a-z, A-Z,
         *        0-9, and - (hyphen).</p>
         *        <p>
         *        This parameter is required for unversioned models. It is not applicable to versioned models.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelPackageName(String modelPackageName);

        /**
         * <p>
         * The name or Amazon Resource Name (ARN) of the model package group that this model version belongs to.
         * </p>
         * <p>
         * This parameter is required for versioned models, and does not apply to unversioned models.
         * </p>
         * 
         * @param modelPackageGroupName
         *        The name or Amazon Resource Name (ARN) of the model package group that this model version belongs
         *        to.</p>
         *        <p>
         *        This parameter is required for versioned models, and does not apply to unversioned models.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelPackageGroupName(String modelPackageGroupName);

        /**
         * <p>
         * A description of the model package.
         * </p>
         * 
         * @param modelPackageDescription
         *        A description of the model package.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelPackageDescription(String modelPackageDescription);

        /**
         * <p>
         * Specifies details about inference jobs that can be run with models based on this model package, including the
         * following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The Amazon ECR paths of containers that contain the inference code and model artifacts.
         * </p>
         * </li>
         * <li>
         * <p>
         * The instance types that the model package supports for transform jobs and real-time endpoints used for
         * inference.
         * </p>
         * </li>
         * <li>
         * <p>
         * The input and output content formats that the model package supports for inference.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferenceSpecification
         *        Specifies details about inference jobs that can be run with models based on this model package,
         *        including the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The Amazon ECR paths of containers that contain the inference code and model artifacts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The instance types that the model package supports for transform jobs and real-time endpoints used for
         *        inference.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The input and output content formats that the model package supports for inference.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferenceSpecification(InferenceSpecification inferenceSpecification);

        /**
         * <p>
         * Specifies details about inference jobs that can be run with models based on this model package, including the
         * following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The Amazon ECR paths of containers that contain the inference code and model artifacts.
         * </p>
         * </li>
         * <li>
         * <p>
         * The instance types that the model package supports for transform jobs and real-time endpoints used for
         * inference.
         * </p>
         * </li>
         * <li>
         * <p>
         * The input and output content formats that the model package supports for inference.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link InferenceSpecification.Builder} avoiding
         * the need to create one manually via {@link InferenceSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InferenceSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #inferenceSpecification(InferenceSpecification)}.
         * 
         * @param inferenceSpecification
         *        a consumer that will call methods on {@link InferenceSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inferenceSpecification(InferenceSpecification)
         */
        default Builder inferenceSpecification(Consumer<InferenceSpecification.Builder> inferenceSpecification) {
            return inferenceSpecification(InferenceSpecification.builder().applyMutation(inferenceSpecification).build());
        }

        /**
         * <p>
         * Specifies configurations for one or more transform jobs that SageMaker runs to test the model package.
         * </p>
         * 
         * @param validationSpecification
         *        Specifies configurations for one or more transform jobs that SageMaker runs to test the model package.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationSpecification(ModelPackageValidationSpecification validationSpecification);

        /**
         * <p>
         * Specifies configurations for one or more transform jobs that SageMaker runs to test the model package.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ModelPackageValidationSpecification.Builder} avoiding the need to create one manually via
         * {@link ModelPackageValidationSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ModelPackageValidationSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #validationSpecification(ModelPackageValidationSpecification)}.
         * 
         * @param validationSpecification
         *        a consumer that will call methods on {@link ModelPackageValidationSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validationSpecification(ModelPackageValidationSpecification)
         */
        default Builder validationSpecification(Consumer<ModelPackageValidationSpecification.Builder> validationSpecification) {
            return validationSpecification(ModelPackageValidationSpecification.builder().applyMutation(validationSpecification)
                    .build());
        }

        /**
         * <p>
         * Details about the algorithm that was used to create the model package.
         * </p>
         * 
         * @param sourceAlgorithmSpecification
         *        Details about the algorithm that was used to create the model package.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceAlgorithmSpecification(SourceAlgorithmSpecification sourceAlgorithmSpecification);

        /**
         * <p>
         * Details about the algorithm that was used to create the model package.
         * </p>
         * This is a convenience method that creates an instance of the {@link SourceAlgorithmSpecification.Builder}
         * avoiding the need to create one manually via {@link SourceAlgorithmSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SourceAlgorithmSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #sourceAlgorithmSpecification(SourceAlgorithmSpecification)}.
         * 
         * @param sourceAlgorithmSpecification
         *        a consumer that will call methods on {@link SourceAlgorithmSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceAlgorithmSpecification(SourceAlgorithmSpecification)
         */
        default Builder sourceAlgorithmSpecification(Consumer<SourceAlgorithmSpecification.Builder> sourceAlgorithmSpecification) {
            return sourceAlgorithmSpecification(SourceAlgorithmSpecification.builder()
                    .applyMutation(sourceAlgorithmSpecification).build());
        }

        /**
         * <p>
         * Whether to certify the model package for listing on Amazon Web Services Marketplace.
         * </p>
         * <p>
         * This parameter is optional for unversioned models, and does not apply to versioned models.
         * </p>
         * 
         * @param certifyForMarketplace
         *        Whether to certify the model package for listing on Amazon Web Services Marketplace.</p>
         *        <p>
         *        This parameter is optional for unversioned models, and does not apply to versioned models.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certifyForMarketplace(Boolean certifyForMarketplace);

        /**
         * <p>
         * A list of key value pairs associated with the model. For more information, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
         * resources</a> in the <i>Amazon Web Services General Reference Guide</i>.
         * </p>
         * 
         * @param tags
         *        A list of key value pairs associated with the model. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
         *        resources</a> in the <i>Amazon Web Services General Reference Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of key value pairs associated with the model. For more information, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
         * resources</a> in the <i>Amazon Web Services General Reference Guide</i>.
         * </p>
         * 
         * @param tags
         *        A list of key value pairs associated with the model. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
         *        resources</a> in the <i>Amazon Web Services General Reference Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of key value pairs associated with the model. For more information, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
         * resources</a> in the <i>Amazon Web Services General Reference Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.sagemaker.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Whether the model is approved for deployment.
         * </p>
         * <p>
         * This parameter is optional for versioned models, and does not apply to unversioned models.
         * </p>
         * <p>
         * For versioned models, the value of this parameter must be set to <code>Approved</code> to deploy the model.
         * </p>
         * 
         * @param modelApprovalStatus
         *        Whether the model is approved for deployment.</p>
         *        <p>
         *        This parameter is optional for versioned models, and does not apply to unversioned models.
         *        </p>
         *        <p>
         *        For versioned models, the value of this parameter must be set to <code>Approved</code> to deploy the
         *        model.
         * @see ModelApprovalStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ModelApprovalStatus
         */
        Builder modelApprovalStatus(String modelApprovalStatus);

        /**
         * <p>
         * Whether the model is approved for deployment.
         * </p>
         * <p>
         * This parameter is optional for versioned models, and does not apply to unversioned models.
         * </p>
         * <p>
         * For versioned models, the value of this parameter must be set to <code>Approved</code> to deploy the model.
         * </p>
         * 
         * @param modelApprovalStatus
         *        Whether the model is approved for deployment.</p>
         *        <p>
         *        This parameter is optional for versioned models, and does not apply to unversioned models.
         *        </p>
         *        <p>
         *        For versioned models, the value of this parameter must be set to <code>Approved</code> to deploy the
         *        model.
         * @see ModelApprovalStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ModelApprovalStatus
         */
        Builder modelApprovalStatus(ModelApprovalStatus modelApprovalStatus);

        /**
         * Sets the value of the MetadataProperties property for this object.
         *
         * @param metadataProperties
         *        The new value for the MetadataProperties property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metadataProperties(MetadataProperties metadataProperties);

        /**
         * Sets the value of the MetadataProperties property for this object.
         *
         * This is a convenience method that creates an instance of the {@link MetadataProperties.Builder} avoiding the
         * need to create one manually via {@link MetadataProperties#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MetadataProperties.Builder#build()} is called immediately and its
         * result is passed to {@link #metadataProperties(MetadataProperties)}.
         * 
         * @param metadataProperties
         *        a consumer that will call methods on {@link MetadataProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metadataProperties(MetadataProperties)
         */
        default Builder metadataProperties(Consumer<MetadataProperties.Builder> metadataProperties) {
            return metadataProperties(MetadataProperties.builder().applyMutation(metadataProperties).build());
        }

        /**
         * <p>
         * A structure that contains model metrics reports.
         * </p>
         * 
         * @param modelMetrics
         *        A structure that contains model metrics reports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelMetrics(ModelMetrics modelMetrics);

        /**
         * <p>
         * A structure that contains model metrics reports.
         * </p>
         * This is a convenience method that creates an instance of the {@link ModelMetrics.Builder} avoiding the need
         * to create one manually via {@link ModelMetrics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ModelMetrics.Builder#build()} is called immediately and its
         * result is passed to {@link #modelMetrics(ModelMetrics)}.
         * 
         * @param modelMetrics
         *        a consumer that will call methods on {@link ModelMetrics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelMetrics(ModelMetrics)
         */
        default Builder modelMetrics(Consumer<ModelMetrics.Builder> modelMetrics) {
            return modelMetrics(ModelMetrics.builder().applyMutation(modelMetrics).build());
        }

        /**
         * <p>
         * A unique token that guarantees that the call to this API is idempotent.
         * </p>
         * 
         * @param clientToken
         *        A unique token that guarantees that the call to this API is idempotent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * The metadata properties associated with the model package versions.
         * </p>
         * 
         * @param customerMetadataProperties
         *        The metadata properties associated with the model package versions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerMetadataProperties(Map<String, String> customerMetadataProperties);

        /**
         * <p>
         * Represents the drift check baselines that can be used when the model monitor is set using the model package.
         * For more information, see the topic on <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/pipelines-quality-clarify-baseline-lifecycle.html#pipelines-quality-clarify-baseline-drift-detection"
         * >Drift Detection against Previous Baselines in SageMaker Pipelines</a> in the <i>Amazon SageMaker Developer
         * Guide</i>.
         * </p>
         * 
         * @param driftCheckBaselines
         *        Represents the drift check baselines that can be used when the model monitor is set using the model
         *        package. For more information, see the topic on <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/pipelines-quality-clarify-baseline-lifecycle.html#pipelines-quality-clarify-baseline-drift-detection"
         *        >Drift Detection against Previous Baselines in SageMaker Pipelines</a> in the <i>Amazon SageMaker
         *        Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder driftCheckBaselines(DriftCheckBaselines driftCheckBaselines);

        /**
         * <p>
         * Represents the drift check baselines that can be used when the model monitor is set using the model package.
         * For more information, see the topic on <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/pipelines-quality-clarify-baseline-lifecycle.html#pipelines-quality-clarify-baseline-drift-detection"
         * >Drift Detection against Previous Baselines in SageMaker Pipelines</a> in the <i>Amazon SageMaker Developer
         * Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link DriftCheckBaselines.Builder} avoiding the
         * need to create one manually via {@link DriftCheckBaselines#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DriftCheckBaselines.Builder#build()} is called immediately and
         * its result is passed to {@link #driftCheckBaselines(DriftCheckBaselines)}.
         * 
         * @param driftCheckBaselines
         *        a consumer that will call methods on {@link DriftCheckBaselines.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #driftCheckBaselines(DriftCheckBaselines)
         */
        default Builder driftCheckBaselines(Consumer<DriftCheckBaselines.Builder> driftCheckBaselines) {
            return driftCheckBaselines(DriftCheckBaselines.builder().applyMutation(driftCheckBaselines).build());
        }

        /**
         * <p>
         * The machine learning domain of your model package and its components. Common machine learning domains include
         * computer vision and natural language processing.
         * </p>
         * 
         * @param domain
         *        The machine learning domain of your model package and its components. Common machine learning domains
         *        include computer vision and natural language processing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domain(String domain);

        /**
         * <p>
         * The machine learning task your model package accomplishes. Common machine learning tasks include object
         * detection and image classification. The following tasks are supported by Inference Recommender:
         * <code>"IMAGE_CLASSIFICATION"</code> | <code>"OBJECT_DETECTION"</code> | <code>"TEXT_GENERATION"</code> |
         * <code>"IMAGE_SEGMENTATION"</code> | <code>"FILL_MASK"</code> | <code>"CLASSIFICATION"</code> |
         * <code>"REGRESSION"</code> | <code>"OTHER"</code>.
         * </p>
         * <p>
         * Specify "OTHER" if none of the tasks listed fit your use case.
         * </p>
         * 
         * @param task
         *        The machine learning task your model package accomplishes. Common machine learning tasks include
         *        object detection and image classification. The following tasks are supported by Inference Recommender:
         *        <code>"IMAGE_CLASSIFICATION"</code> | <code>"OBJECT_DETECTION"</code> | <code>"TEXT_GENERATION"</code>
         *        |<code>"IMAGE_SEGMENTATION"</code> | <code>"FILL_MASK"</code> | <code>"CLASSIFICATION"</code> |
         *        <code>"REGRESSION"</code> | <code>"OTHER"</code>.</p>
         *        <p>
         *        Specify "OTHER" if none of the tasks listed fit your use case.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder task(String task);

        /**
         * <p>
         * The Amazon Simple Storage Service (Amazon S3) path where the sample payload are stored. This path must point
         * to a single gzip compressed tar archive (.tar.gz suffix).
         * </p>
         * 
         * @param samplePayloadUrl
         *        The Amazon Simple Storage Service (Amazon S3) path where the sample payload are stored. This path must
         *        point to a single gzip compressed tar archive (.tar.gz suffix).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder samplePayloadUrl(String samplePayloadUrl);

        /**
         * <p>
         * An array of additional Inference Specification objects. Each additional Inference Specification specifies
         * artifacts based on this model package that can be used on inference endpoints. Generally used with SageMaker
         * Neo to store the compiled artifacts.
         * </p>
         * 
         * @param additionalInferenceSpecifications
         *        An array of additional Inference Specification objects. Each additional Inference Specification
         *        specifies artifacts based on this model package that can be used on inference endpoints. Generally
         *        used with SageMaker Neo to store the compiled artifacts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalInferenceSpecifications(
                Collection<AdditionalInferenceSpecificationDefinition> additionalInferenceSpecifications);

        /**
         * <p>
         * An array of additional Inference Specification objects. Each additional Inference Specification specifies
         * artifacts based on this model package that can be used on inference endpoints. Generally used with SageMaker
         * Neo to store the compiled artifacts.
         * </p>
         * 
         * @param additionalInferenceSpecifications
         *        An array of additional Inference Specification objects. Each additional Inference Specification
         *        specifies artifacts based on this model package that can be used on inference endpoints. Generally
         *        used with SageMaker Neo to store the compiled artifacts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalInferenceSpecifications(AdditionalInferenceSpecificationDefinition... additionalInferenceSpecifications);

        /**
         * <p>
         * An array of additional Inference Specification objects. Each additional Inference Specification specifies
         * artifacts based on this model package that can be used on inference endpoints. Generally used with SageMaker
         * Neo to store the compiled artifacts.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.AdditionalInferenceSpecificationDefinition.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.sagemaker.model.AdditionalInferenceSpecificationDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.AdditionalInferenceSpecificationDefinition.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #additionalInferenceSpecifications(List<AdditionalInferenceSpecificationDefinition>)}.
         * 
         * @param additionalInferenceSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.AdditionalInferenceSpecificationDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #additionalInferenceSpecifications(java.util.Collection<AdditionalInferenceSpecificationDefinition>)
         */
        Builder additionalInferenceSpecifications(
                Consumer<AdditionalInferenceSpecificationDefinition.Builder>... additionalInferenceSpecifications);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SageMakerRequest.BuilderImpl implements Builder {
        private String modelPackageName;

        private String modelPackageGroupName;

        private String modelPackageDescription;

        private InferenceSpecification inferenceSpecification;

        private ModelPackageValidationSpecification validationSpecification;

        private SourceAlgorithmSpecification sourceAlgorithmSpecification;

        private Boolean certifyForMarketplace;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String modelApprovalStatus;

        private MetadataProperties metadataProperties;

        private ModelMetrics modelMetrics;

        private String clientToken;

        private Map<String, String> customerMetadataProperties = DefaultSdkAutoConstructMap.getInstance();

        private DriftCheckBaselines driftCheckBaselines;

        private String domain;

        private String task;

        private String samplePayloadUrl;

        private List<AdditionalInferenceSpecificationDefinition> additionalInferenceSpecifications = DefaultSdkAutoConstructList
                .getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateModelPackageRequest model) {
            super(model);
            modelPackageName(model.modelPackageName);
            modelPackageGroupName(model.modelPackageGroupName);
            modelPackageDescription(model.modelPackageDescription);
            inferenceSpecification(model.inferenceSpecification);
            validationSpecification(model.validationSpecification);
            sourceAlgorithmSpecification(model.sourceAlgorithmSpecification);
            certifyForMarketplace(model.certifyForMarketplace);
            tags(model.tags);
            modelApprovalStatus(model.modelApprovalStatus);
            metadataProperties(model.metadataProperties);
            modelMetrics(model.modelMetrics);
            clientToken(model.clientToken);
            customerMetadataProperties(model.customerMetadataProperties);
            driftCheckBaselines(model.driftCheckBaselines);
            domain(model.domain);
            task(model.task);
            samplePayloadUrl(model.samplePayloadUrl);
            additionalInferenceSpecifications(model.additionalInferenceSpecifications);
        }

        public final String getModelPackageName() {
            return modelPackageName;
        }

        public final void setModelPackageName(String modelPackageName) {
            this.modelPackageName = modelPackageName;
        }

        @Override
        public final Builder modelPackageName(String modelPackageName) {
            this.modelPackageName = modelPackageName;
            return this;
        }

        public final String getModelPackageGroupName() {
            return modelPackageGroupName;
        }

        public final void setModelPackageGroupName(String modelPackageGroupName) {
            this.modelPackageGroupName = modelPackageGroupName;
        }

        @Override
        public final Builder modelPackageGroupName(String modelPackageGroupName) {
            this.modelPackageGroupName = modelPackageGroupName;
            return this;
        }

        public final String getModelPackageDescription() {
            return modelPackageDescription;
        }

        public final void setModelPackageDescription(String modelPackageDescription) {
            this.modelPackageDescription = modelPackageDescription;
        }

        @Override
        public final Builder modelPackageDescription(String modelPackageDescription) {
            this.modelPackageDescription = modelPackageDescription;
            return this;
        }

        public final InferenceSpecification.Builder getInferenceSpecification() {
            return inferenceSpecification != null ? inferenceSpecification.toBuilder() : null;
        }

        public final void setInferenceSpecification(InferenceSpecification.BuilderImpl inferenceSpecification) {
            this.inferenceSpecification = inferenceSpecification != null ? inferenceSpecification.build() : null;
        }

        @Override
        public final Builder inferenceSpecification(InferenceSpecification inferenceSpecification) {
            this.inferenceSpecification = inferenceSpecification;
            return this;
        }

        public final ModelPackageValidationSpecification.Builder getValidationSpecification() {
            return validationSpecification != null ? validationSpecification.toBuilder() : null;
        }

        public final void setValidationSpecification(ModelPackageValidationSpecification.BuilderImpl validationSpecification) {
            this.validationSpecification = validationSpecification != null ? validationSpecification.build() : null;
        }

        @Override
        public final Builder validationSpecification(ModelPackageValidationSpecification validationSpecification) {
            this.validationSpecification = validationSpecification;
            return this;
        }

        public final SourceAlgorithmSpecification.Builder getSourceAlgorithmSpecification() {
            return sourceAlgorithmSpecification != null ? sourceAlgorithmSpecification.toBuilder() : null;
        }

        public final void setSourceAlgorithmSpecification(SourceAlgorithmSpecification.BuilderImpl sourceAlgorithmSpecification) {
            this.sourceAlgorithmSpecification = sourceAlgorithmSpecification != null ? sourceAlgorithmSpecification.build()
                    : null;
        }

        @Override
        public final Builder sourceAlgorithmSpecification(SourceAlgorithmSpecification sourceAlgorithmSpecification) {
            this.sourceAlgorithmSpecification = sourceAlgorithmSpecification;
            return this;
        }

        public final Boolean getCertifyForMarketplace() {
            return certifyForMarketplace;
        }

        public final void setCertifyForMarketplace(Boolean certifyForMarketplace) {
            this.certifyForMarketplace = certifyForMarketplace;
        }

        @Override
        public final Builder certifyForMarketplace(Boolean certifyForMarketplace) {
            this.certifyForMarketplace = certifyForMarketplace;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getModelApprovalStatus() {
            return modelApprovalStatus;
        }

        public final void setModelApprovalStatus(String modelApprovalStatus) {
            this.modelApprovalStatus = modelApprovalStatus;
        }

        @Override
        public final Builder modelApprovalStatus(String modelApprovalStatus) {
            this.modelApprovalStatus = modelApprovalStatus;
            return this;
        }

        @Override
        public final Builder modelApprovalStatus(ModelApprovalStatus modelApprovalStatus) {
            this.modelApprovalStatus(modelApprovalStatus == null ? null : modelApprovalStatus.toString());
            return this;
        }

        public final MetadataProperties.Builder getMetadataProperties() {
            return metadataProperties != null ? metadataProperties.toBuilder() : null;
        }

        public final void setMetadataProperties(MetadataProperties.BuilderImpl metadataProperties) {
            this.metadataProperties = metadataProperties != null ? metadataProperties.build() : null;
        }

        @Override
        public final Builder metadataProperties(MetadataProperties metadataProperties) {
            this.metadataProperties = metadataProperties;
            return this;
        }

        public final ModelMetrics.Builder getModelMetrics() {
            return modelMetrics != null ? modelMetrics.toBuilder() : null;
        }

        public final void setModelMetrics(ModelMetrics.BuilderImpl modelMetrics) {
            this.modelMetrics = modelMetrics != null ? modelMetrics.build() : null;
        }

        @Override
        public final Builder modelMetrics(ModelMetrics modelMetrics) {
            this.modelMetrics = modelMetrics;
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final Map<String, String> getCustomerMetadataProperties() {
            if (customerMetadataProperties instanceof SdkAutoConstructMap) {
                return null;
            }
            return customerMetadataProperties;
        }

        public final void setCustomerMetadataProperties(Map<String, String> customerMetadataProperties) {
            this.customerMetadataProperties = CustomerMetadataMapCopier.copy(customerMetadataProperties);
        }

        @Override
        public final Builder customerMetadataProperties(Map<String, String> customerMetadataProperties) {
            this.customerMetadataProperties = CustomerMetadataMapCopier.copy(customerMetadataProperties);
            return this;
        }

        public final DriftCheckBaselines.Builder getDriftCheckBaselines() {
            return driftCheckBaselines != null ? driftCheckBaselines.toBuilder() : null;
        }

        public final void setDriftCheckBaselines(DriftCheckBaselines.BuilderImpl driftCheckBaselines) {
            this.driftCheckBaselines = driftCheckBaselines != null ? driftCheckBaselines.build() : null;
        }

        @Override
        public final Builder driftCheckBaselines(DriftCheckBaselines driftCheckBaselines) {
            this.driftCheckBaselines = driftCheckBaselines;
            return this;
        }

        public final String getDomain() {
            return domain;
        }

        public final void setDomain(String domain) {
            this.domain = domain;
        }

        @Override
        public final Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        public final String getTask() {
            return task;
        }

        public final void setTask(String task) {
            this.task = task;
        }

        @Override
        public final Builder task(String task) {
            this.task = task;
            return this;
        }

        public final String getSamplePayloadUrl() {
            return samplePayloadUrl;
        }

        public final void setSamplePayloadUrl(String samplePayloadUrl) {
            this.samplePayloadUrl = samplePayloadUrl;
        }

        @Override
        public final Builder samplePayloadUrl(String samplePayloadUrl) {
            this.samplePayloadUrl = samplePayloadUrl;
            return this;
        }

        public final List<AdditionalInferenceSpecificationDefinition.Builder> getAdditionalInferenceSpecifications() {
            List<AdditionalInferenceSpecificationDefinition.Builder> result = AdditionalInferenceSpecificationsCopier
                    .copyToBuilder(this.additionalInferenceSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAdditionalInferenceSpecifications(
                Collection<AdditionalInferenceSpecificationDefinition.BuilderImpl> additionalInferenceSpecifications) {
            this.additionalInferenceSpecifications = AdditionalInferenceSpecificationsCopier
                    .copyFromBuilder(additionalInferenceSpecifications);
        }

        @Override
        public final Builder additionalInferenceSpecifications(
                Collection<AdditionalInferenceSpecificationDefinition> additionalInferenceSpecifications) {
            this.additionalInferenceSpecifications = AdditionalInferenceSpecificationsCopier
                    .copy(additionalInferenceSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder additionalInferenceSpecifications(
                AdditionalInferenceSpecificationDefinition... additionalInferenceSpecifications) {
            additionalInferenceSpecifications(Arrays.asList(additionalInferenceSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder additionalInferenceSpecifications(
                Consumer<AdditionalInferenceSpecificationDefinition.Builder>... additionalInferenceSpecifications) {
            additionalInferenceSpecifications(Stream.of(additionalInferenceSpecifications)
                    .map(c -> AdditionalInferenceSpecificationDefinition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateModelPackageRequest build() {
            return new CreateModelPackageRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
