/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A collection of settings that apply to users of Amazon SageMaker Studio. These settings are specified when the
 * <code>CreateUserProfile</code> API is called, and as <code>DefaultUserSettings</code> when the
 * <code>CreateDomain</code> API is called.
 * </p>
 * <p>
 * <code>SecurityGroups</code> is aggregated when specified in both calls. For all other settings in
 * <code>UserSettings</code>, the values specified in <code>CreateUserProfile</code> take precedence over those
 * specified in <code>CreateDomain</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UserSettings implements SdkPojo, Serializable, ToCopyableBuilder<UserSettings.Builder, UserSettings> {
    private static final SdkField<String> EXECUTION_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionRole").getter(getter(UserSettings::executionRole)).setter(setter(Builder::executionRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionRole").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroups")
            .getter(getter(UserSettings::securityGroups))
            .setter(setter(Builder::securityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<SharingSettings> SHARING_SETTINGS_FIELD = SdkField
            .<SharingSettings> builder(MarshallingType.SDK_POJO).memberName("SharingSettings")
            .getter(getter(UserSettings::sharingSettings)).setter(setter(Builder::sharingSettings))
            .constructor(SharingSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SharingSettings").build()).build();

    private static final SdkField<JupyterServerAppSettings> JUPYTER_SERVER_APP_SETTINGS_FIELD = SdkField
            .<JupyterServerAppSettings> builder(MarshallingType.SDK_POJO).memberName("JupyterServerAppSettings")
            .getter(getter(UserSettings::jupyterServerAppSettings)).setter(setter(Builder::jupyterServerAppSettings))
            .constructor(JupyterServerAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JupyterServerAppSettings").build())
            .build();

    private static final SdkField<KernelGatewayAppSettings> KERNEL_GATEWAY_APP_SETTINGS_FIELD = SdkField
            .<KernelGatewayAppSettings> builder(MarshallingType.SDK_POJO).memberName("KernelGatewayAppSettings")
            .getter(getter(UserSettings::kernelGatewayAppSettings)).setter(setter(Builder::kernelGatewayAppSettings))
            .constructor(KernelGatewayAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KernelGatewayAppSettings").build())
            .build();

    private static final SdkField<TensorBoardAppSettings> TENSOR_BOARD_APP_SETTINGS_FIELD = SdkField
            .<TensorBoardAppSettings> builder(MarshallingType.SDK_POJO).memberName("TensorBoardAppSettings")
            .getter(getter(UserSettings::tensorBoardAppSettings)).setter(setter(Builder::tensorBoardAppSettings))
            .constructor(TensorBoardAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TensorBoardAppSettings").build())
            .build();

    private static final SdkField<RStudioServerProAppSettings> R_STUDIO_SERVER_PRO_APP_SETTINGS_FIELD = SdkField
            .<RStudioServerProAppSettings> builder(MarshallingType.SDK_POJO)
            .memberName("RStudioServerProAppSettings")
            .getter(getter(UserSettings::rStudioServerProAppSettings))
            .setter(setter(Builder::rStudioServerProAppSettings))
            .constructor(RStudioServerProAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RStudioServerProAppSettings")
                    .build()).build();

    private static final SdkField<RSessionAppSettings> R_SESSION_APP_SETTINGS_FIELD = SdkField
            .<RSessionAppSettings> builder(MarshallingType.SDK_POJO).memberName("RSessionAppSettings")
            .getter(getter(UserSettings::rSessionAppSettings)).setter(setter(Builder::rSessionAppSettings))
            .constructor(RSessionAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RSessionAppSettings").build())
            .build();

    private static final SdkField<CanvasAppSettings> CANVAS_APP_SETTINGS_FIELD = SdkField
            .<CanvasAppSettings> builder(MarshallingType.SDK_POJO).memberName("CanvasAppSettings")
            .getter(getter(UserSettings::canvasAppSettings)).setter(setter(Builder::canvasAppSettings))
            .constructor(CanvasAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CanvasAppSettings").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXECUTION_ROLE_FIELD,
            SECURITY_GROUPS_FIELD, SHARING_SETTINGS_FIELD, JUPYTER_SERVER_APP_SETTINGS_FIELD, KERNEL_GATEWAY_APP_SETTINGS_FIELD,
            TENSOR_BOARD_APP_SETTINGS_FIELD, R_STUDIO_SERVER_PRO_APP_SETTINGS_FIELD, R_SESSION_APP_SETTINGS_FIELD,
            CANVAS_APP_SETTINGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String executionRole;

    private final List<String> securityGroups;

    private final SharingSettings sharingSettings;

    private final JupyterServerAppSettings jupyterServerAppSettings;

    private final KernelGatewayAppSettings kernelGatewayAppSettings;

    private final TensorBoardAppSettings tensorBoardAppSettings;

    private final RStudioServerProAppSettings rStudioServerProAppSettings;

    private final RSessionAppSettings rSessionAppSettings;

    private final CanvasAppSettings canvasAppSettings;

    private UserSettings(BuilderImpl builder) {
        this.executionRole = builder.executionRole;
        this.securityGroups = builder.securityGroups;
        this.sharingSettings = builder.sharingSettings;
        this.jupyterServerAppSettings = builder.jupyterServerAppSettings;
        this.kernelGatewayAppSettings = builder.kernelGatewayAppSettings;
        this.tensorBoardAppSettings = builder.tensorBoardAppSettings;
        this.rStudioServerProAppSettings = builder.rStudioServerProAppSettings;
        this.rSessionAppSettings = builder.rSessionAppSettings;
        this.canvasAppSettings = builder.canvasAppSettings;
    }

    /**
     * <p>
     * The execution role for the user.
     * </p>
     * 
     * @return The execution role for the user.
     */
    public final String executionRole() {
        return executionRole;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroups() {
        return securityGroups != null && !(securityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The security groups for the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.
     * </p>
     * <p>
     * Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
     * <code>PublicInternetOnly</code>.
     * </p>
     * <p>
     * Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>VpcOnly</code>.
     * </p>
     * <p>
     * Amazon SageMaker adds a security group to allow NFS traffic from SageMaker Studio. Therefore, the number of
     * security groups that you can specify is one less than the maximum number shown.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroups} method.
     * </p>
     * 
     * @return The security groups for the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.</p>
     *         <p>
     *         Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
     *         <code>PublicInternetOnly</code>.
     *         </p>
     *         <p>
     *         Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>VpcOnly</code>
     *         .
     *         </p>
     *         <p>
     *         Amazon SageMaker adds a security group to allow NFS traffic from SageMaker Studio. Therefore, the number
     *         of security groups that you can specify is one less than the maximum number shown.
     */
    public final List<String> securityGroups() {
        return securityGroups;
    }

    /**
     * <p>
     * Specifies options for sharing SageMaker Studio notebooks.
     * </p>
     * 
     * @return Specifies options for sharing SageMaker Studio notebooks.
     */
    public final SharingSettings sharingSettings() {
        return sharingSettings;
    }

    /**
     * <p>
     * The Jupyter server's app settings.
     * </p>
     * 
     * @return The Jupyter server's app settings.
     */
    public final JupyterServerAppSettings jupyterServerAppSettings() {
        return jupyterServerAppSettings;
    }

    /**
     * <p>
     * The kernel gateway app settings.
     * </p>
     * 
     * @return The kernel gateway app settings.
     */
    public final KernelGatewayAppSettings kernelGatewayAppSettings() {
        return kernelGatewayAppSettings;
    }

    /**
     * <p>
     * The TensorBoard app settings.
     * </p>
     * 
     * @return The TensorBoard app settings.
     */
    public final TensorBoardAppSettings tensorBoardAppSettings() {
        return tensorBoardAppSettings;
    }

    /**
     * <p>
     * A collection of settings that configure user interaction with the <code>RStudioServerPro</code> app.
     * </p>
     * 
     * @return A collection of settings that configure user interaction with the <code>RStudioServerPro</code> app.
     */
    public final RStudioServerProAppSettings rStudioServerProAppSettings() {
        return rStudioServerProAppSettings;
    }

    /**
     * <p>
     * A collection of settings that configure the <code>RSessionGateway</code> app.
     * </p>
     * 
     * @return A collection of settings that configure the <code>RSessionGateway</code> app.
     */
    public final RSessionAppSettings rSessionAppSettings() {
        return rSessionAppSettings;
    }

    /**
     * <p>
     * The Canvas app settings.
     * </p>
     * 
     * @return The Canvas app settings.
     */
    public final CanvasAppSettings canvasAppSettings() {
        return canvasAppSettings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(executionRole());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroups() ? securityGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sharingSettings());
        hashCode = 31 * hashCode + Objects.hashCode(jupyterServerAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(kernelGatewayAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(tensorBoardAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(rStudioServerProAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(rSessionAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(canvasAppSettings());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UserSettings)) {
            return false;
        }
        UserSettings other = (UserSettings) obj;
        return Objects.equals(executionRole(), other.executionRole()) && hasSecurityGroups() == other.hasSecurityGroups()
                && Objects.equals(securityGroups(), other.securityGroups())
                && Objects.equals(sharingSettings(), other.sharingSettings())
                && Objects.equals(jupyterServerAppSettings(), other.jupyterServerAppSettings())
                && Objects.equals(kernelGatewayAppSettings(), other.kernelGatewayAppSettings())
                && Objects.equals(tensorBoardAppSettings(), other.tensorBoardAppSettings())
                && Objects.equals(rStudioServerProAppSettings(), other.rStudioServerProAppSettings())
                && Objects.equals(rSessionAppSettings(), other.rSessionAppSettings())
                && Objects.equals(canvasAppSettings(), other.canvasAppSettings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UserSettings").add("ExecutionRole", executionRole())
                .add("SecurityGroups", hasSecurityGroups() ? securityGroups() : null).add("SharingSettings", sharingSettings())
                .add("JupyterServerAppSettings", jupyterServerAppSettings())
                .add("KernelGatewayAppSettings", kernelGatewayAppSettings())
                .add("TensorBoardAppSettings", tensorBoardAppSettings())
                .add("RStudioServerProAppSettings", rStudioServerProAppSettings())
                .add("RSessionAppSettings", rSessionAppSettings()).add("CanvasAppSettings", canvasAppSettings()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ExecutionRole":
            return Optional.ofNullable(clazz.cast(executionRole()));
        case "SecurityGroups":
            return Optional.ofNullable(clazz.cast(securityGroups()));
        case "SharingSettings":
            return Optional.ofNullable(clazz.cast(sharingSettings()));
        case "JupyterServerAppSettings":
            return Optional.ofNullable(clazz.cast(jupyterServerAppSettings()));
        case "KernelGatewayAppSettings":
            return Optional.ofNullable(clazz.cast(kernelGatewayAppSettings()));
        case "TensorBoardAppSettings":
            return Optional.ofNullable(clazz.cast(tensorBoardAppSettings()));
        case "RStudioServerProAppSettings":
            return Optional.ofNullable(clazz.cast(rStudioServerProAppSettings()));
        case "RSessionAppSettings":
            return Optional.ofNullable(clazz.cast(rSessionAppSettings()));
        case "CanvasAppSettings":
            return Optional.ofNullable(clazz.cast(canvasAppSettings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UserSettings, T> g) {
        return obj -> g.apply((UserSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UserSettings> {
        /**
         * <p>
         * The execution role for the user.
         * </p>
         * 
         * @param executionRole
         *        The execution role for the user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRole(String executionRole);

        /**
         * <p>
         * The security groups for the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.
         * </p>
         * <p>
         * Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         * <code>PublicInternetOnly</code>.
         * </p>
         * <p>
         * Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>VpcOnly</code>.
         * </p>
         * <p>
         * Amazon SageMaker adds a security group to allow NFS traffic from SageMaker Studio. Therefore, the number of
         * security groups that you can specify is one less than the maximum number shown.
         * </p>
         * 
         * @param securityGroups
         *        The security groups for the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.</p>
         *        <p>
         *        Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         *        <code>PublicInternetOnly</code>.
         *        </p>
         *        <p>
         *        Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         *        <code>VpcOnly</code>.
         *        </p>
         *        <p>
         *        Amazon SageMaker adds a security group to allow NFS traffic from SageMaker Studio. Therefore, the
         *        number of security groups that you can specify is one less than the maximum number shown.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(Collection<String> securityGroups);

        /**
         * <p>
         * The security groups for the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.
         * </p>
         * <p>
         * Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         * <code>PublicInternetOnly</code>.
         * </p>
         * <p>
         * Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>VpcOnly</code>.
         * </p>
         * <p>
         * Amazon SageMaker adds a security group to allow NFS traffic from SageMaker Studio. Therefore, the number of
         * security groups that you can specify is one less than the maximum number shown.
         * </p>
         * 
         * @param securityGroups
         *        The security groups for the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.</p>
         *        <p>
         *        Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         *        <code>PublicInternetOnly</code>.
         *        </p>
         *        <p>
         *        Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         *        <code>VpcOnly</code>.
         *        </p>
         *        <p>
         *        Amazon SageMaker adds a security group to allow NFS traffic from SageMaker Studio. Therefore, the
         *        number of security groups that you can specify is one less than the maximum number shown.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(String... securityGroups);

        /**
         * <p>
         * Specifies options for sharing SageMaker Studio notebooks.
         * </p>
         * 
         * @param sharingSettings
         *        Specifies options for sharing SageMaker Studio notebooks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sharingSettings(SharingSettings sharingSettings);

        /**
         * <p>
         * Specifies options for sharing SageMaker Studio notebooks.
         * </p>
         * This is a convenience method that creates an instance of the {@link SharingSettings.Builder} avoiding the
         * need to create one manually via {@link SharingSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SharingSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #sharingSettings(SharingSettings)}.
         * 
         * @param sharingSettings
         *        a consumer that will call methods on {@link SharingSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sharingSettings(SharingSettings)
         */
        default Builder sharingSettings(Consumer<SharingSettings.Builder> sharingSettings) {
            return sharingSettings(SharingSettings.builder().applyMutation(sharingSettings).build());
        }

        /**
         * <p>
         * The Jupyter server's app settings.
         * </p>
         * 
         * @param jupyterServerAppSettings
         *        The Jupyter server's app settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jupyterServerAppSettings(JupyterServerAppSettings jupyterServerAppSettings);

        /**
         * <p>
         * The Jupyter server's app settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link JupyterServerAppSettings.Builder}
         * avoiding the need to create one manually via {@link JupyterServerAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JupyterServerAppSettings.Builder#build()} is called immediately
         * and its result is passed to {@link #jupyterServerAppSettings(JupyterServerAppSettings)}.
         * 
         * @param jupyterServerAppSettings
         *        a consumer that will call methods on {@link JupyterServerAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jupyterServerAppSettings(JupyterServerAppSettings)
         */
        default Builder jupyterServerAppSettings(Consumer<JupyterServerAppSettings.Builder> jupyterServerAppSettings) {
            return jupyterServerAppSettings(JupyterServerAppSettings.builder().applyMutation(jupyterServerAppSettings).build());
        }

        /**
         * <p>
         * The kernel gateway app settings.
         * </p>
         * 
         * @param kernelGatewayAppSettings
         *        The kernel gateway app settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kernelGatewayAppSettings(KernelGatewayAppSettings kernelGatewayAppSettings);

        /**
         * <p>
         * The kernel gateway app settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link KernelGatewayAppSettings.Builder}
         * avoiding the need to create one manually via {@link KernelGatewayAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KernelGatewayAppSettings.Builder#build()} is called immediately
         * and its result is passed to {@link #kernelGatewayAppSettings(KernelGatewayAppSettings)}.
         * 
         * @param kernelGatewayAppSettings
         *        a consumer that will call methods on {@link KernelGatewayAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kernelGatewayAppSettings(KernelGatewayAppSettings)
         */
        default Builder kernelGatewayAppSettings(Consumer<KernelGatewayAppSettings.Builder> kernelGatewayAppSettings) {
            return kernelGatewayAppSettings(KernelGatewayAppSettings.builder().applyMutation(kernelGatewayAppSettings).build());
        }

        /**
         * <p>
         * The TensorBoard app settings.
         * </p>
         * 
         * @param tensorBoardAppSettings
         *        The TensorBoard app settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tensorBoardAppSettings(TensorBoardAppSettings tensorBoardAppSettings);

        /**
         * <p>
         * The TensorBoard app settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link TensorBoardAppSettings.Builder} avoiding
         * the need to create one manually via {@link TensorBoardAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TensorBoardAppSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #tensorBoardAppSettings(TensorBoardAppSettings)}.
         * 
         * @param tensorBoardAppSettings
         *        a consumer that will call methods on {@link TensorBoardAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tensorBoardAppSettings(TensorBoardAppSettings)
         */
        default Builder tensorBoardAppSettings(Consumer<TensorBoardAppSettings.Builder> tensorBoardAppSettings) {
            return tensorBoardAppSettings(TensorBoardAppSettings.builder().applyMutation(tensorBoardAppSettings).build());
        }

        /**
         * <p>
         * A collection of settings that configure user interaction with the <code>RStudioServerPro</code> app.
         * </p>
         * 
         * @param rStudioServerProAppSettings
         *        A collection of settings that configure user interaction with the <code>RStudioServerPro</code> app.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rStudioServerProAppSettings(RStudioServerProAppSettings rStudioServerProAppSettings);

        /**
         * <p>
         * A collection of settings that configure user interaction with the <code>RStudioServerPro</code> app.
         * </p>
         * This is a convenience method that creates an instance of the {@link RStudioServerProAppSettings.Builder}
         * avoiding the need to create one manually via {@link RStudioServerProAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RStudioServerProAppSettings.Builder#build()} is called
         * immediately and its result is passed to {@link #rStudioServerProAppSettings(RStudioServerProAppSettings)}.
         * 
         * @param rStudioServerProAppSettings
         *        a consumer that will call methods on {@link RStudioServerProAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rStudioServerProAppSettings(RStudioServerProAppSettings)
         */
        default Builder rStudioServerProAppSettings(Consumer<RStudioServerProAppSettings.Builder> rStudioServerProAppSettings) {
            return rStudioServerProAppSettings(RStudioServerProAppSettings.builder().applyMutation(rStudioServerProAppSettings)
                    .build());
        }

        /**
         * <p>
         * A collection of settings that configure the <code>RSessionGateway</code> app.
         * </p>
         * 
         * @param rSessionAppSettings
         *        A collection of settings that configure the <code>RSessionGateway</code> app.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rSessionAppSettings(RSessionAppSettings rSessionAppSettings);

        /**
         * <p>
         * A collection of settings that configure the <code>RSessionGateway</code> app.
         * </p>
         * This is a convenience method that creates an instance of the {@link RSessionAppSettings.Builder} avoiding the
         * need to create one manually via {@link RSessionAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RSessionAppSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #rSessionAppSettings(RSessionAppSettings)}.
         * 
         * @param rSessionAppSettings
         *        a consumer that will call methods on {@link RSessionAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rSessionAppSettings(RSessionAppSettings)
         */
        default Builder rSessionAppSettings(Consumer<RSessionAppSettings.Builder> rSessionAppSettings) {
            return rSessionAppSettings(RSessionAppSettings.builder().applyMutation(rSessionAppSettings).build());
        }

        /**
         * <p>
         * The Canvas app settings.
         * </p>
         * 
         * @param canvasAppSettings
         *        The Canvas app settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder canvasAppSettings(CanvasAppSettings canvasAppSettings);

        /**
         * <p>
         * The Canvas app settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link CanvasAppSettings.Builder} avoiding the
         * need to create one manually via {@link CanvasAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CanvasAppSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #canvasAppSettings(CanvasAppSettings)}.
         * 
         * @param canvasAppSettings
         *        a consumer that will call methods on {@link CanvasAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #canvasAppSettings(CanvasAppSettings)
         */
        default Builder canvasAppSettings(Consumer<CanvasAppSettings.Builder> canvasAppSettings) {
            return canvasAppSettings(CanvasAppSettings.builder().applyMutation(canvasAppSettings).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String executionRole;

        private List<String> securityGroups = DefaultSdkAutoConstructList.getInstance();

        private SharingSettings sharingSettings;

        private JupyterServerAppSettings jupyterServerAppSettings;

        private KernelGatewayAppSettings kernelGatewayAppSettings;

        private TensorBoardAppSettings tensorBoardAppSettings;

        private RStudioServerProAppSettings rStudioServerProAppSettings;

        private RSessionAppSettings rSessionAppSettings;

        private CanvasAppSettings canvasAppSettings;

        private BuilderImpl() {
        }

        private BuilderImpl(UserSettings model) {
            executionRole(model.executionRole);
            securityGroups(model.securityGroups);
            sharingSettings(model.sharingSettings);
            jupyterServerAppSettings(model.jupyterServerAppSettings);
            kernelGatewayAppSettings(model.kernelGatewayAppSettings);
            tensorBoardAppSettings(model.tensorBoardAppSettings);
            rStudioServerProAppSettings(model.rStudioServerProAppSettings);
            rSessionAppSettings(model.rSessionAppSettings);
            canvasAppSettings(model.canvasAppSettings);
        }

        public final String getExecutionRole() {
            return executionRole;
        }

        public final void setExecutionRole(String executionRole) {
            this.executionRole = executionRole;
        }

        @Override
        public final Builder executionRole(String executionRole) {
            this.executionRole = executionRole;
            return this;
        }

        public final Collection<String> getSecurityGroups() {
            if (securityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroups;
        }

        public final void setSecurityGroups(Collection<String> securityGroups) {
            this.securityGroups = SecurityGroupIdsCopier.copy(securityGroups);
        }

        @Override
        public final Builder securityGroups(Collection<String> securityGroups) {
            this.securityGroups = SecurityGroupIdsCopier.copy(securityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(String... securityGroups) {
            securityGroups(Arrays.asList(securityGroups));
            return this;
        }

        public final SharingSettings.Builder getSharingSettings() {
            return sharingSettings != null ? sharingSettings.toBuilder() : null;
        }

        public final void setSharingSettings(SharingSettings.BuilderImpl sharingSettings) {
            this.sharingSettings = sharingSettings != null ? sharingSettings.build() : null;
        }

        @Override
        public final Builder sharingSettings(SharingSettings sharingSettings) {
            this.sharingSettings = sharingSettings;
            return this;
        }

        public final JupyterServerAppSettings.Builder getJupyterServerAppSettings() {
            return jupyterServerAppSettings != null ? jupyterServerAppSettings.toBuilder() : null;
        }

        public final void setJupyterServerAppSettings(JupyterServerAppSettings.BuilderImpl jupyterServerAppSettings) {
            this.jupyterServerAppSettings = jupyterServerAppSettings != null ? jupyterServerAppSettings.build() : null;
        }

        @Override
        public final Builder jupyterServerAppSettings(JupyterServerAppSettings jupyterServerAppSettings) {
            this.jupyterServerAppSettings = jupyterServerAppSettings;
            return this;
        }

        public final KernelGatewayAppSettings.Builder getKernelGatewayAppSettings() {
            return kernelGatewayAppSettings != null ? kernelGatewayAppSettings.toBuilder() : null;
        }

        public final void setKernelGatewayAppSettings(KernelGatewayAppSettings.BuilderImpl kernelGatewayAppSettings) {
            this.kernelGatewayAppSettings = kernelGatewayAppSettings != null ? kernelGatewayAppSettings.build() : null;
        }

        @Override
        public final Builder kernelGatewayAppSettings(KernelGatewayAppSettings kernelGatewayAppSettings) {
            this.kernelGatewayAppSettings = kernelGatewayAppSettings;
            return this;
        }

        public final TensorBoardAppSettings.Builder getTensorBoardAppSettings() {
            return tensorBoardAppSettings != null ? tensorBoardAppSettings.toBuilder() : null;
        }

        public final void setTensorBoardAppSettings(TensorBoardAppSettings.BuilderImpl tensorBoardAppSettings) {
            this.tensorBoardAppSettings = tensorBoardAppSettings != null ? tensorBoardAppSettings.build() : null;
        }

        @Override
        public final Builder tensorBoardAppSettings(TensorBoardAppSettings tensorBoardAppSettings) {
            this.tensorBoardAppSettings = tensorBoardAppSettings;
            return this;
        }

        public final RStudioServerProAppSettings.Builder getRStudioServerProAppSettings() {
            return rStudioServerProAppSettings != null ? rStudioServerProAppSettings.toBuilder() : null;
        }

        public final void setRStudioServerProAppSettings(RStudioServerProAppSettings.BuilderImpl rStudioServerProAppSettings) {
            this.rStudioServerProAppSettings = rStudioServerProAppSettings != null ? rStudioServerProAppSettings.build() : null;
        }

        @Override
        public final Builder rStudioServerProAppSettings(RStudioServerProAppSettings rStudioServerProAppSettings) {
            this.rStudioServerProAppSettings = rStudioServerProAppSettings;
            return this;
        }

        public final RSessionAppSettings.Builder getRSessionAppSettings() {
            return rSessionAppSettings != null ? rSessionAppSettings.toBuilder() : null;
        }

        public final void setRSessionAppSettings(RSessionAppSettings.BuilderImpl rSessionAppSettings) {
            this.rSessionAppSettings = rSessionAppSettings != null ? rSessionAppSettings.build() : null;
        }

        @Override
        public final Builder rSessionAppSettings(RSessionAppSettings rSessionAppSettings) {
            this.rSessionAppSettings = rSessionAppSettings;
            return this;
        }

        public final CanvasAppSettings.Builder getCanvasAppSettings() {
            return canvasAppSettings != null ? canvasAppSettings.toBuilder() : null;
        }

        public final void setCanvasAppSettings(CanvasAppSettings.BuilderImpl canvasAppSettings) {
            this.canvasAppSettings = canvasAppSettings != null ? canvasAppSettings.build() : null;
        }

        @Override
        public final Builder canvasAppSettings(CanvasAppSettings canvasAppSettings) {
            this.canvasAppSettings = canvasAppSettings;
            return this;
        }

        @Override
        public UserSettings build() {
            return new UserSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
