/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemakerruntime;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sagemakerruntime.model.InternalDependencyException;
import software.amazon.awssdk.services.sagemakerruntime.model.InternalFailureException;
import software.amazon.awssdk.services.sagemakerruntime.model.InvokeEndpointAsyncRequest;
import software.amazon.awssdk.services.sagemakerruntime.model.InvokeEndpointAsyncResponse;
import software.amazon.awssdk.services.sagemakerruntime.model.InvokeEndpointRequest;
import software.amazon.awssdk.services.sagemakerruntime.model.InvokeEndpointResponse;
import software.amazon.awssdk.services.sagemakerruntime.model.ModelErrorException;
import software.amazon.awssdk.services.sagemakerruntime.model.ModelNotReadyException;
import software.amazon.awssdk.services.sagemakerruntime.model.SageMakerRuntimeException;
import software.amazon.awssdk.services.sagemakerruntime.model.ServiceUnavailableException;
import software.amazon.awssdk.services.sagemakerruntime.model.ValidationErrorException;
import software.amazon.awssdk.services.sagemakerruntime.transform.InvokeEndpointAsyncRequestMarshaller;
import software.amazon.awssdk.services.sagemakerruntime.transform.InvokeEndpointRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SageMakerRuntimeAsyncClient}.
 *
 * @see SageMakerRuntimeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSageMakerRuntimeAsyncClient implements SageMakerRuntimeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSageMakerRuntimeAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSageMakerRuntimeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * After you deploy a model into production using Amazon SageMaker hosting services, your client applications use
     * this API to get inferences from the model hosted at the specified endpoint.
     * </p>
     * <p>
     * For an overview of Amazon SageMaker, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/how-it-works.html">How It Works</a>.
     * </p>
     * <p>
     * Amazon SageMaker strips all POST headers except those supported by the API. Amazon SageMaker might add additional
     * headers. You should not rely on the behavior of headers outside those enumerated in the request syntax.
     * </p>
     * <p>
     * Calls to <code>InvokeEndpoint</code> are authenticated by using Amazon Web Services Signature Version 4. For
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/sig-v4-authenticating-requests.html">Authenticating
     * Requests (Amazon Web Services Signature Version 4)</a> in the <i>Amazon S3 API Reference</i>.
     * </p>
     * <p>
     * A customer's model containers must respond to requests within 60 seconds. The model itself can have a maximum
     * processing time of 60 seconds before responding to invocations. If your model is going to take 50-60 seconds of
     * processing time, the SDK socket timeout should be set to be 70 seconds.
     * </p>
     * <note>
     * <p>
     * Endpoints are scoped to an individual account, and are not public. The URL does not contain the account ID, but
     * Amazon SageMaker determines the account ID from the authentication token that is supplied by the caller.
     * </p>
     * </note>
     *
     * @param invokeEndpointRequest
     * @return A Java Future containing the result of the InvokeEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Try your call again.</li>
     *         <li>ValidationErrorException Inspect your request and try again.</li>
     *         <li>ModelErrorException Model (owned by the customer in the container) returned 4xx or 5xx error code.</li>
     *         <li>InternalDependencyException Your request caused an exception with an internal dependency. Contact
     *         customer support.</li>
     *         <li>ModelNotReadyException Either a serverless endpoint variant's resources are still being provisioned,
     *         or a multi-model endpoint is still downloading or loading the target model. Wait and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SageMakerRuntimeAsyncClient.InvokeEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.sagemaker-2017-05-13/InvokeEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<InvokeEndpointResponse> invokeEndpoint(InvokeEndpointRequest invokeEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<InvokeEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InvokeEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InvokeEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InvokeEndpointRequest, InvokeEndpointResponse>()
                            .withOperationName("InvokeEndpoint")
                            .withMarshaller(new InvokeEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(invokeEndpointRequest));
            CompletableFuture<InvokeEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * After you deploy a model into production using Amazon SageMaker hosting services, your client applications use
     * this API to get inferences from the model hosted at the specified endpoint in an asynchronous manner.
     * </p>
     * <p>
     * Inference requests sent to this API are enqueued for asynchronous processing. The processing of the inference
     * request may or may not complete before the you receive a response from this API. The response from this API will
     * not contain the result of the inference request but contain information about where you can locate it.
     * </p>
     * <p>
     * Amazon SageMaker strips all <code>POST</code> headers except those supported by the API. Amazon SageMaker might
     * add additional headers. You should not rely on the behavior of headers outside those enumerated in the request
     * syntax.
     * </p>
     * <p>
     * Calls to <code>InvokeEndpointAsync</code> are authenticated by using Amazon Web Services Signature Version 4. For
     * information, see <a href=
     * "https://docs.aws.amazon.com/https:/docs.aws.amazon.com/AmazonS3/latest/API/sig-v4-authenticating-requests.html"
     * >Authenticating Requests (Amazon Web Services Signature Version 4)</a> in the <i>Amazon S3 API Reference</i>.
     * </p>
     *
     * @param invokeEndpointAsyncRequest
     * @return A Java Future containing the result of the InvokeEndpointAsync operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Try your call again.</li>
     *         <li>ValidationErrorException Inspect your request and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SageMakerRuntimeAsyncClient.InvokeEndpointAsync
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.sagemaker-2017-05-13/InvokeEndpointAsync"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<InvokeEndpointAsyncResponse> invokeEndpointAsync(
            InvokeEndpointAsyncRequest invokeEndpointAsyncRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeEndpointAsyncRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeEndpointAsync");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InvokeEndpointAsyncResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InvokeEndpointAsyncResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InvokeEndpointAsyncResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InvokeEndpointAsyncRequest, InvokeEndpointAsyncResponse>()
                            .withOperationName("InvokeEndpointAsync")
                            .withMarshaller(new InvokeEndpointAsyncRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(invokeEndpointAsyncRequest));
            CompletableFuture<InvokeEndpointAsyncResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SageMakerRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailure")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ModelError")
                                .exceptionBuilderSupplier(ModelErrorException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalDependencyException")
                                .exceptionBuilderSupplier(InternalDependencyException::builder).httpStatusCode(530).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationError")
                                .exceptionBuilderSupplier(ValidationErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ModelNotReadyException")
                                .exceptionBuilderSupplier(ModelNotReadyException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailable")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
