/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides metadata for the Amazon CodeGuru detector associated with a finding. This field pertains to findings that
 * relate to Lambda functions. Amazon Inspector identifies policy violations and vulnerabilities in Lambda function code
 * based on internal detectors developed in collaboration with Amazon CodeGuru. Security Hub receives those findings.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GeneratorDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<GeneratorDetails.Builder, GeneratorDetails> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(GeneratorDetails::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(GeneratorDetails::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<String>> LABELS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Labels")
            .getter(getter(GeneratorDetails::labels))
            .setter(setter(Builder::labels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Labels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            LABELS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final List<String> labels;

    private GeneratorDetails(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.labels = builder.labels;
    }

    /**
     * <p>
     * The name of the detector used to identify the code vulnerability.
     * </p>
     * 
     * @return The name of the detector used to identify the code vulnerability.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The description of the detector used to identify the code vulnerability.
     * </p>
     * 
     * @return The description of the detector used to identify the code vulnerability.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the Labels property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasLabels() {
        return labels != null && !(labels instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of tags used to identify the detector associated with the finding.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLabels} method.
     * </p>
     * 
     * @return An array of tags used to identify the detector associated with the finding.
     */
    public final List<String> labels() {
        return labels;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasLabels() ? labels() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GeneratorDetails)) {
            return false;
        }
        GeneratorDetails other = (GeneratorDetails) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && hasLabels() == other.hasLabels() && Objects.equals(labels(), other.labels());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GeneratorDetails").add("Name", name()).add("Description", description())
                .add("Labels", hasLabels() ? labels() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Labels":
            return Optional.ofNullable(clazz.cast(labels()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GeneratorDetails, T> g) {
        return obj -> g.apply((GeneratorDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GeneratorDetails> {
        /**
         * <p>
         * The name of the detector used to identify the code vulnerability.
         * </p>
         * 
         * @param name
         *        The name of the detector used to identify the code vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description of the detector used to identify the code vulnerability.
         * </p>
         * 
         * @param description
         *        The description of the detector used to identify the code vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * An array of tags used to identify the detector associated with the finding.
         * </p>
         * 
         * @param labels
         *        An array of tags used to identify the detector associated with the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labels(Collection<String> labels);

        /**
         * <p>
         * An array of tags used to identify the detector associated with the finding.
         * </p>
         * 
         * @param labels
         *        An array of tags used to identify the detector associated with the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labels(String... labels);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private List<String> labels = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GeneratorDetails model) {
            name(model.name);
            description(model.description);
            labels(model.labels);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getLabels() {
            if (labels instanceof SdkAutoConstructList) {
                return null;
            }
            return labels;
        }

        public final void setLabels(Collection<String> labels) {
            this.labels = TypeListCopier.copy(labels);
        }

        @Override
        public final Builder labels(Collection<String> labels) {
            this.labels = TypeListCopier.copy(labels);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder labels(String... labels) {
            labels(Arrays.asList(labels));
            return this;
        }

        @Override
        public GeneratorDetails build() {
            return new GeneratorDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
