/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.serverlessapplicationrepository;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.BadRequestException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ConflictException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateApplicationRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateApplicationResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateApplicationVersionRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateApplicationVersionResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateCloudFormationChangeSetRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateCloudFormationChangeSetResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateCloudFormationTemplateRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateCloudFormationTemplateResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ForbiddenException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetApplicationPolicyRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetApplicationPolicyResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetApplicationRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetApplicationResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetCloudFormationTemplateRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetCloudFormationTemplateResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.InternalServerErrorException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationDependenciesRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationDependenciesResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationVersionsRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationVersionsResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationsRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationsResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.NotFoundException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.PutApplicationPolicyRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.PutApplicationPolicyResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ServerlessApplicationRepositoryException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ServerlessApplicationRepositoryRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.TooManyRequestsException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.UnshareApplicationRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.UnshareApplicationResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.paginators.ListApplicationDependenciesPublisher;
import software.amazon.awssdk.services.serverlessapplicationrepository.paginators.ListApplicationVersionsPublisher;
import software.amazon.awssdk.services.serverlessapplicationrepository.paginators.ListApplicationsPublisher;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.CreateApplicationVersionRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.CreateCloudFormationChangeSetRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.CreateCloudFormationTemplateRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.GetApplicationPolicyRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.GetCloudFormationTemplateRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.ListApplicationDependenciesRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.ListApplicationVersionsRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.PutApplicationPolicyRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.UnshareApplicationRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ServerlessApplicationRepositoryAsyncClient}.
 *
 * @see ServerlessApplicationRepositoryAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultServerlessApplicationRepositoryAsyncClient implements ServerlessApplicationRepositoryAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultServerlessApplicationRepositoryAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultServerlessApplicationRepositoryAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an application, optionally including an AWS SAM file to create the first application version in the same
     * call.
     * </p>
     *
     * @param createApplicationRequest
     * @return A Java Future containing the result of the CreateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.CreateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/CreateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApplicationResponse> createApplication(CreateApplicationRequest createApplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                            .withOperationName("CreateApplication")
                            .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createApplicationRequest));
            CompletableFuture<CreateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an application version.
     * </p>
     *
     * @param createApplicationVersionRequest
     * @return A Java Future containing the result of the CreateApplicationVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.CreateApplicationVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/CreateApplicationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApplicationVersionResponse> createApplicationVersion(
            CreateApplicationVersionRequest createApplicationVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplicationVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApplicationVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApplicationVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApplicationVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationVersionRequest, CreateApplicationVersionResponse>()
                            .withOperationName("CreateApplicationVersion")
                            .withMarshaller(new CreateApplicationVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createApplicationVersionRequest));
            CompletableFuture<CreateApplicationVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AWS CloudFormation change set for the given application.
     * </p>
     *
     * @param createCloudFormationChangeSetRequest
     * @return A Java Future containing the result of the CreateCloudFormationChangeSet operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.CreateCloudFormationChangeSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/CreateCloudFormationChangeSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCloudFormationChangeSetResponse> createCloudFormationChangeSet(
            CreateCloudFormationChangeSetRequest createCloudFormationChangeSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createCloudFormationChangeSetRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCloudFormationChangeSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCloudFormationChangeSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCloudFormationChangeSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCloudFormationChangeSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCloudFormationChangeSetRequest, CreateCloudFormationChangeSetResponse>()
                            .withOperationName("CreateCloudFormationChangeSet")
                            .withMarshaller(new CreateCloudFormationChangeSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createCloudFormationChangeSetRequest));
            CompletableFuture<CreateCloudFormationChangeSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AWS CloudFormation template.
     * </p>
     *
     * @param createCloudFormationTemplateRequest
     * @return A Java Future containing the result of the CreateCloudFormationTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.CreateCloudFormationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/CreateCloudFormationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCloudFormationTemplateResponse> createCloudFormationTemplate(
            CreateCloudFormationTemplateRequest createCloudFormationTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCloudFormationTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCloudFormationTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCloudFormationTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCloudFormationTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCloudFormationTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCloudFormationTemplateRequest, CreateCloudFormationTemplateResponse>()
                            .withOperationName("CreateCloudFormationTemplate")
                            .withMarshaller(new CreateCloudFormationTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createCloudFormationTemplateRequest));
            CompletableFuture<CreateCloudFormationTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified application.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return A Java Future containing the result of the DeleteApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.DeleteApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/DeleteApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationResponse> deleteApplication(DeleteApplicationRequest deleteApplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                            .withOperationName("DeleteApplication")
                            .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteApplicationRequest));
            CompletableFuture<DeleteApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified application.
     * </p>
     *
     * @param getApplicationRequest
     * @return A Java Future containing the result of the GetApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.GetApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/GetApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetApplicationResponse> getApplication(GetApplicationRequest getApplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                            .withOperationName("GetApplication")
                            .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getApplicationRequest));
            CompletableFuture<GetApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the policy for the application.
     * </p>
     *
     * @param getApplicationPolicyRequest
     * @return A Java Future containing the result of the GetApplicationPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.GetApplicationPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/GetApplicationPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetApplicationPolicyResponse> getApplicationPolicy(
            GetApplicationPolicyRequest getApplicationPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplicationPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetApplicationPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetApplicationPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetApplicationPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetApplicationPolicyRequest, GetApplicationPolicyResponse>()
                            .withOperationName("GetApplicationPolicy")
                            .withMarshaller(new GetApplicationPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getApplicationPolicyRequest));
            CompletableFuture<GetApplicationPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified AWS CloudFormation template.
     * </p>
     *
     * @param getCloudFormationTemplateRequest
     * @return A Java Future containing the result of the GetCloudFormationTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.GetCloudFormationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/GetCloudFormationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCloudFormationTemplateResponse> getCloudFormationTemplate(
            GetCloudFormationTemplateRequest getCloudFormationTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCloudFormationTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCloudFormationTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCloudFormationTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCloudFormationTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCloudFormationTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCloudFormationTemplateRequest, GetCloudFormationTemplateResponse>()
                            .withOperationName("GetCloudFormationTemplate")
                            .withMarshaller(new GetCloudFormationTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCloudFormationTemplateRequest));
            CompletableFuture<GetCloudFormationTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of applications nested in the containing application.
     * </p>
     *
     * @param listApplicationDependenciesRequest
     * @return A Java Future containing the result of the ListApplicationDependencies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.ListApplicationDependencies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/ListApplicationDependencies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationDependenciesResponse> listApplicationDependencies(
            ListApplicationDependenciesRequest listApplicationDependenciesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationDependenciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationDependencies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationDependenciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationDependenciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationDependenciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationDependenciesRequest, ListApplicationDependenciesResponse>()
                            .withOperationName("ListApplicationDependencies")
                            .withMarshaller(new ListApplicationDependenciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listApplicationDependenciesRequest));
            CompletableFuture<ListApplicationDependenciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of applications nested in the containing application.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplicationDependencies(software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationDependenciesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.serverlessapplicationrepository.paginators.ListApplicationDependenciesPublisher publisher = client.listApplicationDependenciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.serverlessapplicationrepository.paginators.ListApplicationDependenciesPublisher publisher = client.listApplicationDependenciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationDependenciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationDependenciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplicationDependencies(software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationDependenciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listApplicationDependenciesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.ListApplicationDependencies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/ListApplicationDependencies"
     *      target="_top">AWS API Documentation</a>
     */
    public ListApplicationDependenciesPublisher listApplicationDependenciesPaginator(
            ListApplicationDependenciesRequest listApplicationDependenciesRequest) {
        return new ListApplicationDependenciesPublisher(this, applyPaginatorUserAgent(listApplicationDependenciesRequest));
    }

    /**
     * <p>
     * Lists versions for the specified application.
     * </p>
     *
     * @param listApplicationVersionsRequest
     * @return A Java Future containing the result of the ListApplicationVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.ListApplicationVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/ListApplicationVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationVersionsResponse> listApplicationVersions(
            ListApplicationVersionsRequest listApplicationVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationVersionsRequest, ListApplicationVersionsResponse>()
                            .withOperationName("ListApplicationVersions")
                            .withMarshaller(new ListApplicationVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listApplicationVersionsRequest));
            CompletableFuture<ListApplicationVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists versions for the specified application.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplicationVersions(software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationVersionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.serverlessapplicationrepository.paginators.ListApplicationVersionsPublisher publisher = client.listApplicationVersionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.serverlessapplicationrepository.paginators.ListApplicationVersionsPublisher publisher = client.listApplicationVersionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationVersionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationVersionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplicationVersions(software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listApplicationVersionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.ListApplicationVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/ListApplicationVersions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListApplicationVersionsPublisher listApplicationVersionsPaginator(
            ListApplicationVersionsRequest listApplicationVersionsRequest) {
        return new ListApplicationVersionsPublisher(this, applyPaginatorUserAgent(listApplicationVersionsRequest));
    }

    /**
     * <p>
     * Lists applications owned by the requester.
     * </p>
     *
     * @param listApplicationsRequest
     * @return A Java Future containing the result of the ListApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/ListApplications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationsResponse> listApplications(ListApplicationsRequest listApplicationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                            .withOperationName("ListApplications")
                            .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listApplicationsRequest));
            CompletableFuture<ListApplicationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists applications owned by the requester.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplications(software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.serverlessapplicationrepository.paginators.ListApplicationsPublisher publisher = client.listApplicationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.serverlessapplicationrepository.paginators.ListApplicationsPublisher publisher = client.listApplicationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplications(software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listApplicationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/ListApplications"
     *      target="_top">AWS API Documentation</a>
     */
    public ListApplicationsPublisher listApplicationsPaginator(ListApplicationsRequest listApplicationsRequest) {
        return new ListApplicationsPublisher(this, applyPaginatorUserAgent(listApplicationsRequest));
    }

    /**
     * <p>
     * Sets the permission policy for an application. For the list of actions supported for this operation, see <a href=
     * "https://docs.aws.amazon.com/serverlessrepo/latest/devguide/access-control-resource-based.html#application-permissions"
     * >Application Permissions</a> .
     * </p>
     *
     * @param putApplicationPolicyRequest
     * @return A Java Future containing the result of the PutApplicationPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.PutApplicationPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/PutApplicationPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutApplicationPolicyResponse> putApplicationPolicy(
            PutApplicationPolicyRequest putApplicationPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putApplicationPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutApplicationPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutApplicationPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutApplicationPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutApplicationPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutApplicationPolicyRequest, PutApplicationPolicyResponse>()
                            .withOperationName("PutApplicationPolicy")
                            .withMarshaller(new PutApplicationPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putApplicationPolicyRequest));
            CompletableFuture<PutApplicationPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Unshares an application from an AWS Organization.
     * </p>
     * <p>
     * This operation can be called only from the organization's master account.
     * </p>
     *
     * @param unshareApplicationRequest
     * @return A Java Future containing the result of the UnshareApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.UnshareApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/UnshareApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UnshareApplicationResponse> unshareApplication(UnshareApplicationRequest unshareApplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unshareApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnshareApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UnshareApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UnshareApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UnshareApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UnshareApplicationRequest, UnshareApplicationResponse>()
                            .withOperationName("UnshareApplication")
                            .withMarshaller(new UnshareApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(unshareApplicationRequest));
            CompletableFuture<UnshareApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified application.
     * </p>
     *
     * @param updateApplicationRequest
     * @return A Java Future containing the result of the UpdateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException One of the parameters in the request is invalid.</li>
     *         <li>InternalServerErrorException The AWS Serverless Application Repository service encountered an
     *         internal error.</li>
     *         <li>ForbiddenException The client is not authenticated.</li>
     *         <li>NotFoundException The resource (for example, an access policy statement) specified in the request
     *         doesn't exist.</li>
     *         <li>TooManyRequestsException The client is sending more than the allowed number of requests per unit of
     *         time.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServerlessApplicationRepositoryException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample ServerlessApplicationRepositoryAsyncClient.UpdateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/UpdateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApplicationResponse> updateApplication(UpdateApplicationRequest updateApplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                            .withOperationName("UpdateApplication")
                            .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateApplicationRequest));
            CompletableFuture<UpdateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ServerlessApplicationRepositoryException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends ServerlessApplicationRepositoryRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
