/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowdevicemanagement;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.snowdevicemanagement.internal.SnowDeviceManagementServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.snowdevicemanagement.model.AccessDeniedException;
import software.amazon.awssdk.services.snowdevicemanagement.model.CancelTaskRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.CancelTaskResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.CreateTaskRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.CreateTaskResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeDeviceEc2InstancesRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeDeviceEc2InstancesResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeDeviceRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeDeviceResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeExecutionRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeExecutionResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeTaskRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeTaskResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.InternalServerException;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListDeviceResourcesRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListDeviceResourcesResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListDevicesRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListDevicesResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListExecutionsRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListExecutionsResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListTasksRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListTasksResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ResourceNotFoundException;
import software.amazon.awssdk.services.snowdevicemanagement.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.snowdevicemanagement.model.SnowDeviceManagementException;
import software.amazon.awssdk.services.snowdevicemanagement.model.TagResourceRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.TagResourceResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ThrottlingException;
import software.amazon.awssdk.services.snowdevicemanagement.model.UntagResourceRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.UntagResourceResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ValidationException;
import software.amazon.awssdk.services.snowdevicemanagement.transform.CancelTaskRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.CreateTaskRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.DescribeDeviceEc2InstancesRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.DescribeDeviceRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.DescribeExecutionRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.DescribeTaskRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.ListDeviceResourcesRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.ListDevicesRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.ListExecutionsRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.ListTasksRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SnowDeviceManagementClient}.
 *
 * @see SnowDeviceManagementClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSnowDeviceManagementClient implements SnowDeviceManagementClient {
    private static final Logger log = Logger.loggerFor(DefaultSnowDeviceManagementClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSnowDeviceManagementClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Sends a cancel request for a specified task. You can cancel a task only if it's still in a <code>QUEUED</code>
     * state. Tasks that are already running can't be cancelled.
     * </p>
     * <note>
     * <p>
     * A task might still run if it's processed from the queue before the <code>CancelTask</code> operation changes the
     * task's state.
     * </p>
     * </note>
     *
     * @param cancelTaskRequest
     * @return Result of the CancelTask operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.CancelTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/CancelTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelTaskResponse cancelTask(CancelTaskRequest cancelTaskRequest) throws ThrottlingException,
            InternalServerException, ResourceNotFoundException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelTaskRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelTask");

            return clientHandler.execute(new ClientExecutionParams<CancelTaskRequest, CancelTaskResponse>()
                    .withOperationName("CancelTask").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(cancelTaskRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Instructs one or more devices to start a task, such as unlocking or rebooting.
     * </p>
     *
     * @param createTaskRequest
     * @return Result of the CreateTask operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.CreateTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/CreateTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateTaskResponse createTask(CreateTaskRequest createTaskRequest) throws ServiceQuotaExceededException,
            ThrottlingException, InternalServerException, ResourceNotFoundException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTaskRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTask");

            return clientHandler.execute(new ClientExecutionParams<CreateTaskRequest, CreateTaskResponse>()
                    .withOperationName("CreateTask").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createTaskRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Checks device-specific information, such as the device type, software version, IP addresses, and lock status.
     * </p>
     *
     * @param describeDeviceRequest
     * @return Result of the DescribeDevice operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.DescribeDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/DescribeDevice"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDeviceResponse describeDevice(DescribeDeviceRequest describeDeviceRequest) throws ThrottlingException,
            InternalServerException, ResourceNotFoundException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDeviceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDevice");

            return clientHandler.execute(new ClientExecutionParams<DescribeDeviceRequest, DescribeDeviceResponse>()
                    .withOperationName("DescribeDevice").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeDeviceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDeviceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Checks the current state of the Amazon EC2 instances. The output is similar to <code>describeDevice</code>, but
     * the results are sourced from the device cache in the Amazon Web Services Cloud and include a subset of the
     * available fields.
     * </p>
     *
     * @param describeDeviceEc2InstancesRequest
     * @return Result of the DescribeDeviceEc2Instances operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.DescribeDeviceEc2Instances
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/DescribeDeviceEc2Instances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDeviceEc2InstancesResponse describeDeviceEc2Instances(
            DescribeDeviceEc2InstancesRequest describeDeviceEc2InstancesRequest) throws ThrottlingException,
            InternalServerException, ResourceNotFoundException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDeviceEc2InstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDeviceEc2InstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDeviceEc2InstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeviceEc2InstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDeviceEc2Instances");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeDeviceEc2InstancesRequest, DescribeDeviceEc2InstancesResponse>()
                            .withOperationName("DescribeDeviceEc2Instances").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeDeviceEc2InstancesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeDeviceEc2InstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Checks the status of a remote task running on one or more target devices.
     * </p>
     *
     * @param describeExecutionRequest
     * @return Result of the DescribeExecution operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.DescribeExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/DescribeExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeExecutionResponse describeExecution(DescribeExecutionRequest describeExecutionRequest)
            throws ThrottlingException, InternalServerException, ResourceNotFoundException, ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeExecutionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeExecution");

            return clientHandler.execute(new ClientExecutionParams<DescribeExecutionRequest, DescribeExecutionResponse>()
                    .withOperationName("DescribeExecution").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeExecutionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Checks the metadata for a given task on a device.
     * </p>
     *
     * @param describeTaskRequest
     * @return Result of the DescribeTask operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.DescribeTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/DescribeTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTaskResponse describeTask(DescribeTaskRequest describeTaskRequest) throws ThrottlingException,
            InternalServerException, ResourceNotFoundException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTaskRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTask");

            return clientHandler.execute(new ClientExecutionParams<DescribeTaskRequest, DescribeTaskResponse>()
                    .withOperationName("DescribeTask").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the Amazon Web Services resources available for a device. Currently, Amazon EC2 instances are
     * the only supported resource type.
     * </p>
     *
     * @param listDeviceResourcesRequest
     * @return Result of the ListDeviceResources operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.ListDeviceResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListDeviceResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDeviceResourcesResponse listDeviceResources(ListDeviceResourcesRequest listDeviceResourcesRequest)
            throws ThrottlingException, InternalServerException, ResourceNotFoundException, ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDeviceResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDeviceResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDeviceResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeviceResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeviceResources");

            return clientHandler.execute(new ClientExecutionParams<ListDeviceResourcesRequest, ListDeviceResourcesResponse>()
                    .withOperationName("ListDeviceResources").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDeviceResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDeviceResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all devices on your Amazon Web Services account that have Amazon Web Services Snow Device
     * Management enabled in the Amazon Web Services Region where the command is run.
     * </p>
     *
     * @param listDevicesRequest
     * @return Result of the ListDevices operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.ListDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListDevices"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDevicesResponse listDevices(ListDevicesRequest listDevicesRequest) throws ThrottlingException,
            InternalServerException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDevicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDevicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDevicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDevices");

            return clientHandler.execute(new ClientExecutionParams<ListDevicesRequest, ListDevicesResponse>()
                    .withOperationName("ListDevices").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listDevicesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDevicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the status of tasks for one or more target devices.
     * </p>
     *
     * @param listExecutionsRequest
     * @return Result of the ListExecutions operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.ListExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListExecutionsResponse listExecutions(ListExecutionsRequest listExecutionsRequest) throws ThrottlingException,
            InternalServerException, ResourceNotFoundException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListExecutionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExecutionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExecutions");

            return clientHandler.execute(new ClientExecutionParams<ListExecutionsRequest, ListExecutionsResponse>()
                    .withOperationName("ListExecutions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listExecutionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListExecutionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tags for a managed device or task.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tasks that can be filtered by state.
     * </p>
     *
     * @param listTasksRequest
     * @return Result of the ListTasks operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.ListTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTasksResponse listTasks(ListTasksRequest listTasksRequest) throws ThrottlingException, InternalServerException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTasksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTasksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTasksRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTasks");

            return clientHandler.execute(new ClientExecutionParams<ListTasksRequest, ListTasksResponse>()
                    .withOperationName("ListTasks").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listTasksRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTasksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or replaces tags on a device or task.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from a device or task.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowDeviceManagementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowDeviceManagementClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            SnowDeviceManagementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        SnowDeviceManagementServiceClientConfigurationBuilder serviceConfigBuilder = new SnowDeviceManagementServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SnowDeviceManagementException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final SnowDeviceManagementServiceClientConfiguration serviceClientConfiguration() {
        return new SnowDeviceManagementServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
