/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmincidents;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ssmincidents.model.AccessDeniedException;
import software.amazon.awssdk.services.ssmincidents.model.ConflictException;
import software.amazon.awssdk.services.ssmincidents.model.CreateReplicationSetRequest;
import software.amazon.awssdk.services.ssmincidents.model.CreateReplicationSetResponse;
import software.amazon.awssdk.services.ssmincidents.model.CreateResponsePlanRequest;
import software.amazon.awssdk.services.ssmincidents.model.CreateResponsePlanResponse;
import software.amazon.awssdk.services.ssmincidents.model.CreateTimelineEventRequest;
import software.amazon.awssdk.services.ssmincidents.model.CreateTimelineEventResponse;
import software.amazon.awssdk.services.ssmincidents.model.DeleteIncidentRecordRequest;
import software.amazon.awssdk.services.ssmincidents.model.DeleteIncidentRecordResponse;
import software.amazon.awssdk.services.ssmincidents.model.DeleteReplicationSetRequest;
import software.amazon.awssdk.services.ssmincidents.model.DeleteReplicationSetResponse;
import software.amazon.awssdk.services.ssmincidents.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.ssmincidents.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.ssmincidents.model.DeleteResponsePlanRequest;
import software.amazon.awssdk.services.ssmincidents.model.DeleteResponsePlanResponse;
import software.amazon.awssdk.services.ssmincidents.model.DeleteTimelineEventRequest;
import software.amazon.awssdk.services.ssmincidents.model.DeleteTimelineEventResponse;
import software.amazon.awssdk.services.ssmincidents.model.GetIncidentRecordRequest;
import software.amazon.awssdk.services.ssmincidents.model.GetIncidentRecordResponse;
import software.amazon.awssdk.services.ssmincidents.model.GetReplicationSetRequest;
import software.amazon.awssdk.services.ssmincidents.model.GetReplicationSetResponse;
import software.amazon.awssdk.services.ssmincidents.model.GetResourcePoliciesRequest;
import software.amazon.awssdk.services.ssmincidents.model.GetResourcePoliciesResponse;
import software.amazon.awssdk.services.ssmincidents.model.GetResponsePlanRequest;
import software.amazon.awssdk.services.ssmincidents.model.GetResponsePlanResponse;
import software.amazon.awssdk.services.ssmincidents.model.GetTimelineEventRequest;
import software.amazon.awssdk.services.ssmincidents.model.GetTimelineEventResponse;
import software.amazon.awssdk.services.ssmincidents.model.InternalServerException;
import software.amazon.awssdk.services.ssmincidents.model.ListIncidentRecordsRequest;
import software.amazon.awssdk.services.ssmincidents.model.ListIncidentRecordsResponse;
import software.amazon.awssdk.services.ssmincidents.model.ListRelatedItemsRequest;
import software.amazon.awssdk.services.ssmincidents.model.ListRelatedItemsResponse;
import software.amazon.awssdk.services.ssmincidents.model.ListReplicationSetsRequest;
import software.amazon.awssdk.services.ssmincidents.model.ListReplicationSetsResponse;
import software.amazon.awssdk.services.ssmincidents.model.ListResponsePlansRequest;
import software.amazon.awssdk.services.ssmincidents.model.ListResponsePlansResponse;
import software.amazon.awssdk.services.ssmincidents.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ssmincidents.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ssmincidents.model.ListTimelineEventsRequest;
import software.amazon.awssdk.services.ssmincidents.model.ListTimelineEventsResponse;
import software.amazon.awssdk.services.ssmincidents.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.ssmincidents.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.ssmincidents.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ssmincidents.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ssmincidents.model.SsmIncidentsException;
import software.amazon.awssdk.services.ssmincidents.model.SsmIncidentsRequest;
import software.amazon.awssdk.services.ssmincidents.model.StartIncidentRequest;
import software.amazon.awssdk.services.ssmincidents.model.StartIncidentResponse;
import software.amazon.awssdk.services.ssmincidents.model.TagResourceRequest;
import software.amazon.awssdk.services.ssmincidents.model.TagResourceResponse;
import software.amazon.awssdk.services.ssmincidents.model.ThrottlingException;
import software.amazon.awssdk.services.ssmincidents.model.UntagResourceRequest;
import software.amazon.awssdk.services.ssmincidents.model.UntagResourceResponse;
import software.amazon.awssdk.services.ssmincidents.model.UpdateDeletionProtectionRequest;
import software.amazon.awssdk.services.ssmincidents.model.UpdateDeletionProtectionResponse;
import software.amazon.awssdk.services.ssmincidents.model.UpdateIncidentRecordRequest;
import software.amazon.awssdk.services.ssmincidents.model.UpdateIncidentRecordResponse;
import software.amazon.awssdk.services.ssmincidents.model.UpdateRelatedItemsRequest;
import software.amazon.awssdk.services.ssmincidents.model.UpdateRelatedItemsResponse;
import software.amazon.awssdk.services.ssmincidents.model.UpdateReplicationSetRequest;
import software.amazon.awssdk.services.ssmincidents.model.UpdateReplicationSetResponse;
import software.amazon.awssdk.services.ssmincidents.model.UpdateResponsePlanRequest;
import software.amazon.awssdk.services.ssmincidents.model.UpdateResponsePlanResponse;
import software.amazon.awssdk.services.ssmincidents.model.UpdateTimelineEventRequest;
import software.amazon.awssdk.services.ssmincidents.model.UpdateTimelineEventResponse;
import software.amazon.awssdk.services.ssmincidents.model.ValidationException;
import software.amazon.awssdk.services.ssmincidents.paginators.GetResourcePoliciesIterable;
import software.amazon.awssdk.services.ssmincidents.paginators.ListIncidentRecordsIterable;
import software.amazon.awssdk.services.ssmincidents.paginators.ListRelatedItemsIterable;
import software.amazon.awssdk.services.ssmincidents.paginators.ListReplicationSetsIterable;
import software.amazon.awssdk.services.ssmincidents.paginators.ListResponsePlansIterable;
import software.amazon.awssdk.services.ssmincidents.paginators.ListTimelineEventsIterable;
import software.amazon.awssdk.services.ssmincidents.transform.CreateReplicationSetRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.CreateResponsePlanRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.CreateTimelineEventRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.DeleteIncidentRecordRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.DeleteReplicationSetRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.DeleteResponsePlanRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.DeleteTimelineEventRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.GetIncidentRecordRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.GetReplicationSetRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.GetResourcePoliciesRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.GetResponsePlanRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.GetTimelineEventRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.ListIncidentRecordsRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.ListRelatedItemsRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.ListReplicationSetsRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.ListResponsePlansRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.ListTimelineEventsRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.StartIncidentRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.UpdateDeletionProtectionRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.UpdateIncidentRecordRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.UpdateRelatedItemsRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.UpdateReplicationSetRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.UpdateResponsePlanRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.transform.UpdateTimelineEventRequestMarshaller;
import software.amazon.awssdk.services.ssmincidents.waiters.SsmIncidentsWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SsmIncidentsClient}.
 *
 * @see SsmIncidentsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsmIncidentsClient implements SsmIncidentsClient {
    private static final Logger log = Logger.loggerFor(DefaultSsmIncidentsClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSsmIncidentsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * A replication set replicates and encrypts your data to the provided Regions with the provided KMS key.
     * </p>
     *
     * @param createReplicationSetRequest
     * @return Result of the CreateReplicationSet operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.CreateReplicationSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/CreateReplicationSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateReplicationSetResponse createReplicationSet(CreateReplicationSetRequest createReplicationSetRequest)
            throws ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, ValidationException,
            ConflictException, InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateReplicationSetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateReplicationSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createReplicationSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateReplicationSet");

            return clientHandler.execute(new ClientExecutionParams<CreateReplicationSetRequest, CreateReplicationSetResponse>()
                    .withOperationName("CreateReplicationSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createReplicationSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateReplicationSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a response plan that automates the initial response to incidents. A response plan engages contacts,
     * starts chat channel collaboration, and initiates runbooks at the beginning of an incident.
     * </p>
     *
     * @param createResponsePlanRequest
     * @return Result of the CreateResponsePlan operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.CreateResponsePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/CreateResponsePlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateResponsePlanResponse createResponsePlan(CreateResponsePlanRequest createResponsePlanRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResponsePlanResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateResponsePlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResponsePlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResponsePlan");

            return clientHandler.execute(new ClientExecutionParams<CreateResponsePlanRequest, CreateResponsePlanResponse>()
                    .withOperationName("CreateResponsePlan").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createResponsePlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateResponsePlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a custom timeline event on the incident details page of an incident record. Timeline events are
     * automatically created by Incident Manager, marking key moment during an incident. You can create custom timeline
     * events to mark important events that are automatically detected by Incident Manager.
     * </p>
     *
     * @param createTimelineEventRequest
     * @return Result of the CreateTimelineEvent operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.CreateTimelineEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/CreateTimelineEvent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateTimelineEventResponse createTimelineEvent(CreateTimelineEventRequest createTimelineEventRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTimelineEventResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateTimelineEventResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTimelineEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTimelineEvent");

            return clientHandler.execute(new ClientExecutionParams<CreateTimelineEventRequest, CreateTimelineEventResponse>()
                    .withOperationName("CreateTimelineEvent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createTimelineEventRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTimelineEventRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an incident record from Incident Manager.
     * </p>
     *
     * @param deleteIncidentRecordRequest
     * @return Result of the DeleteIncidentRecord operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.DeleteIncidentRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/DeleteIncidentRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteIncidentRecordResponse deleteIncidentRecord(DeleteIncidentRecordRequest deleteIncidentRecordRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteIncidentRecordResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteIncidentRecordResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIncidentRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIncidentRecord");

            return clientHandler.execute(new ClientExecutionParams<DeleteIncidentRecordRequest, DeleteIncidentRecordResponse>()
                    .withOperationName("DeleteIncidentRecord").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteIncidentRecordRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteIncidentRecordRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes all Regions in your replication set. Deleting the replication set deletes all Incident Manager data.
     * </p>
     *
     * @param deleteReplicationSetRequest
     * @return Result of the DeleteReplicationSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.DeleteReplicationSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/DeleteReplicationSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteReplicationSetResponse deleteReplicationSet(DeleteReplicationSetRequest deleteReplicationSetRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteReplicationSetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteReplicationSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteReplicationSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteReplicationSet");

            return clientHandler.execute(new ClientExecutionParams<DeleteReplicationSetRequest, DeleteReplicationSetResponse>()
                    .withOperationName("DeleteReplicationSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteReplicationSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteReplicationSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the resource policy that Resource Access Manager uses to share your Incident Manager resource.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return Result of the DeleteResourcePolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.DeleteResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/DeleteResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResourcePolicyResponse deleteResourcePolicy(DeleteResourcePolicyRequest deleteResourcePolicyRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                    .withOperationName("DeleteResourcePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified response plan. Deleting a response plan stops all linked CloudWatch alarms and EventBridge
     * events from creating an incident with this response plan.
     * </p>
     *
     * @param deleteResponsePlanRequest
     * @return Result of the DeleteResponsePlan operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.DeleteResponsePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/DeleteResponsePlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResponsePlanResponse deleteResponsePlan(DeleteResponsePlanRequest deleteResponsePlanRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResponsePlanResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResponsePlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResponsePlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResponsePlan");

            return clientHandler.execute(new ClientExecutionParams<DeleteResponsePlanRequest, DeleteResponsePlanResponse>()
                    .withOperationName("DeleteResponsePlan").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteResponsePlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResponsePlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a timeline event from an incident.
     * </p>
     *
     * @param deleteTimelineEventRequest
     * @return Result of the DeleteTimelineEvent operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.DeleteTimelineEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/DeleteTimelineEvent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteTimelineEventResponse deleteTimelineEvent(DeleteTimelineEventRequest deleteTimelineEventRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTimelineEventResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteTimelineEventResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTimelineEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTimelineEvent");

            return clientHandler.execute(new ClientExecutionParams<DeleteTimelineEventRequest, DeleteTimelineEventResponse>()
                    .withOperationName("DeleteTimelineEvent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteTimelineEventRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTimelineEventRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the details for the specified incident record.
     * </p>
     *
     * @param getIncidentRecordRequest
     * @return Result of the GetIncidentRecord operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.GetIncidentRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/GetIncidentRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetIncidentRecordResponse getIncidentRecord(GetIncidentRecordRequest getIncidentRecordRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetIncidentRecordResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetIncidentRecordResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIncidentRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIncidentRecord");

            return clientHandler.execute(new ClientExecutionParams<GetIncidentRecordRequest, GetIncidentRecordResponse>()
                    .withOperationName("GetIncidentRecord").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getIncidentRecordRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetIncidentRecordRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieve your Incident Manager replication set.
     * </p>
     *
     * @param getReplicationSetRequest
     * @return Result of the GetReplicationSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.GetReplicationSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/GetReplicationSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReplicationSetResponse getReplicationSet(GetReplicationSetRequest getReplicationSetRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReplicationSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetReplicationSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getReplicationSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReplicationSet");

            return clientHandler.execute(new ClientExecutionParams<GetReplicationSetRequest, GetReplicationSetResponse>()
                    .withOperationName("GetReplicationSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getReplicationSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetReplicationSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the resource policies attached to the specified response plan.
     * </p>
     *
     * @param getResourcePoliciesRequest
     * @return Result of the GetResourcePolicies operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.GetResourcePolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/GetResourcePolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourcePoliciesResponse getResourcePolicies(GetResourcePoliciesRequest getResourcePoliciesRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourcePoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicies");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePoliciesRequest, GetResourcePoliciesResponse>()
                    .withOperationName("GetResourcePolicies").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResourcePoliciesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the resource policies attached to the specified response plan.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourcePolicies(software.amazon.awssdk.services.ssmincidents.model.GetResourcePoliciesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.GetResourcePoliciesIterable responses = client.getResourcePoliciesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmincidents.paginators.GetResourcePoliciesIterable responses = client
     *             .getResourcePoliciesPaginator(request);
     *     for (software.amazon.awssdk.services.ssmincidents.model.GetResourcePoliciesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.GetResourcePoliciesIterable responses = client.getResourcePoliciesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourcePolicies(software.amazon.awssdk.services.ssmincidents.model.GetResourcePoliciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param getResourcePoliciesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.GetResourcePolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/GetResourcePolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourcePoliciesIterable getResourcePoliciesPaginator(GetResourcePoliciesRequest getResourcePoliciesRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        return new GetResourcePoliciesIterable(this, applyPaginatorUserAgent(getResourcePoliciesRequest));
    }

    /**
     * <p>
     * Retrieves the details of the specified response plan.
     * </p>
     *
     * @param getResponsePlanRequest
     * @return Result of the GetResponsePlan operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.GetResponsePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/GetResponsePlan" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetResponsePlanResponse getResponsePlan(GetResponsePlanRequest getResponsePlanRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResponsePlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResponsePlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResponsePlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResponsePlan");

            return clientHandler.execute(new ClientExecutionParams<GetResponsePlanRequest, GetResponsePlanResponse>()
                    .withOperationName("GetResponsePlan").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResponsePlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResponsePlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a timeline event based on its ID and incident record.
     * </p>
     *
     * @param getTimelineEventRequest
     * @return Result of the GetTimelineEvent operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.GetTimelineEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/GetTimelineEvent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetTimelineEventResponse getTimelineEvent(GetTimelineEventRequest getTimelineEventRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTimelineEventResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTimelineEventResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTimelineEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTimelineEvent");

            return clientHandler.execute(new ClientExecutionParams<GetTimelineEventRequest, GetTimelineEventResponse>()
                    .withOperationName("GetTimelineEvent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getTimelineEventRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTimelineEventRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all incident records in your account. Use this command to retrieve the Amazon Resource Name (ARN) of the
     * incident record you want to update.
     * </p>
     *
     * @param listIncidentRecordsRequest
     * @return Result of the ListIncidentRecords operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.ListIncidentRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/ListIncidentRecords"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListIncidentRecordsResponse listIncidentRecords(ListIncidentRecordsRequest listIncidentRecordsRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListIncidentRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListIncidentRecordsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIncidentRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIncidentRecords");

            return clientHandler.execute(new ClientExecutionParams<ListIncidentRecordsRequest, ListIncidentRecordsResponse>()
                    .withOperationName("ListIncidentRecords").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listIncidentRecordsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListIncidentRecordsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all incident records in your account. Use this command to retrieve the Amazon Resource Name (ARN) of the
     * incident record you want to update.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listIncidentRecords(software.amazon.awssdk.services.ssmincidents.model.ListIncidentRecordsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.ListIncidentRecordsIterable responses = client.listIncidentRecordsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmincidents.paginators.ListIncidentRecordsIterable responses = client
     *             .listIncidentRecordsPaginator(request);
     *     for (software.amazon.awssdk.services.ssmincidents.model.ListIncidentRecordsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.ListIncidentRecordsIterable responses = client.listIncidentRecordsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listIncidentRecords(software.amazon.awssdk.services.ssmincidents.model.ListIncidentRecordsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listIncidentRecordsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.ListIncidentRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/ListIncidentRecords"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListIncidentRecordsIterable listIncidentRecordsPaginator(ListIncidentRecordsRequest listIncidentRecordsRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        return new ListIncidentRecordsIterable(this, applyPaginatorUserAgent(listIncidentRecordsRequest));
    }

    /**
     * <p>
     * List all related items for an incident record.
     * </p>
     *
     * @param listRelatedItemsRequest
     * @return Result of the ListRelatedItems operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.ListRelatedItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/ListRelatedItems"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRelatedItemsResponse listRelatedItems(ListRelatedItemsRequest listRelatedItemsRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRelatedItemsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRelatedItemsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRelatedItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRelatedItems");

            return clientHandler.execute(new ClientExecutionParams<ListRelatedItemsRequest, ListRelatedItemsResponse>()
                    .withOperationName("ListRelatedItems").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listRelatedItemsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRelatedItemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all related items for an incident record.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRelatedItems(software.amazon.awssdk.services.ssmincidents.model.ListRelatedItemsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.ListRelatedItemsIterable responses = client.listRelatedItemsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmincidents.paginators.ListRelatedItemsIterable responses = client
     *             .listRelatedItemsPaginator(request);
     *     for (software.amazon.awssdk.services.ssmincidents.model.ListRelatedItemsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.ListRelatedItemsIterable responses = client.listRelatedItemsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRelatedItems(software.amazon.awssdk.services.ssmincidents.model.ListRelatedItemsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRelatedItemsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.ListRelatedItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/ListRelatedItems"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRelatedItemsIterable listRelatedItemsPaginator(ListRelatedItemsRequest listRelatedItemsRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        return new ListRelatedItemsIterable(this, applyPaginatorUserAgent(listRelatedItemsRequest));
    }

    /**
     * <p>
     * Lists details about the replication set configured in your account.
     * </p>
     *
     * @param listReplicationSetsRequest
     * @return Result of the ListReplicationSets operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.ListReplicationSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/ListReplicationSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReplicationSetsResponse listReplicationSets(ListReplicationSetsRequest listReplicationSetsRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReplicationSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListReplicationSetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReplicationSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReplicationSets");

            return clientHandler.execute(new ClientExecutionParams<ListReplicationSetsRequest, ListReplicationSetsResponse>()
                    .withOperationName("ListReplicationSets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listReplicationSetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListReplicationSetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists details about the replication set configured in your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listReplicationSets(software.amazon.awssdk.services.ssmincidents.model.ListReplicationSetsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.ListReplicationSetsIterable responses = client.listReplicationSetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmincidents.paginators.ListReplicationSetsIterable responses = client
     *             .listReplicationSetsPaginator(request);
     *     for (software.amazon.awssdk.services.ssmincidents.model.ListReplicationSetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.ListReplicationSetsIterable responses = client.listReplicationSetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReplicationSets(software.amazon.awssdk.services.ssmincidents.model.ListReplicationSetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listReplicationSetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.ListReplicationSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/ListReplicationSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReplicationSetsIterable listReplicationSetsPaginator(ListReplicationSetsRequest listReplicationSetsRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        return new ListReplicationSetsIterable(this, applyPaginatorUserAgent(listReplicationSetsRequest));
    }

    /**
     * <p>
     * Lists all response plans in your account.
     * </p>
     *
     * @param listResponsePlansRequest
     * @return Result of the ListResponsePlans operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.ListResponsePlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/ListResponsePlans"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResponsePlansResponse listResponsePlans(ListResponsePlansRequest listResponsePlansRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResponsePlansResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResponsePlansResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResponsePlansRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResponsePlans");

            return clientHandler.execute(new ClientExecutionParams<ListResponsePlansRequest, ListResponsePlansResponse>()
                    .withOperationName("ListResponsePlans").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listResponsePlansRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResponsePlansRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all response plans in your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResponsePlans(software.amazon.awssdk.services.ssmincidents.model.ListResponsePlansRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.ListResponsePlansIterable responses = client.listResponsePlansPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmincidents.paginators.ListResponsePlansIterable responses = client
     *             .listResponsePlansPaginator(request);
     *     for (software.amazon.awssdk.services.ssmincidents.model.ListResponsePlansResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.ListResponsePlansIterable responses = client.listResponsePlansPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResponsePlans(software.amazon.awssdk.services.ssmincidents.model.ListResponsePlansRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResponsePlansRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.ListResponsePlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/ListResponsePlans"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResponsePlansIterable listResponsePlansPaginator(ListResponsePlansRequest listResponsePlansRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        return new ListResponsePlansIterable(this, applyPaginatorUserAgent(listResponsePlansRequest));
    }

    /**
     * <p>
     * Lists the tags that are attached to the specified response plan.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists timeline events for the specified incident record.
     * </p>
     *
     * @param listTimelineEventsRequest
     * @return Result of the ListTimelineEvents operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.ListTimelineEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/ListTimelineEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTimelineEventsResponse listTimelineEvents(ListTimelineEventsRequest listTimelineEventsRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTimelineEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTimelineEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTimelineEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTimelineEvents");

            return clientHandler.execute(new ClientExecutionParams<ListTimelineEventsRequest, ListTimelineEventsResponse>()
                    .withOperationName("ListTimelineEvents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTimelineEventsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTimelineEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists timeline events for the specified incident record.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTimelineEvents(software.amazon.awssdk.services.ssmincidents.model.ListTimelineEventsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.ListTimelineEventsIterable responses = client.listTimelineEventsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmincidents.paginators.ListTimelineEventsIterable responses = client
     *             .listTimelineEventsPaginator(request);
     *     for (software.amazon.awssdk.services.ssmincidents.model.ListTimelineEventsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmincidents.paginators.ListTimelineEventsIterable responses = client.listTimelineEventsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTimelineEvents(software.amazon.awssdk.services.ssmincidents.model.ListTimelineEventsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTimelineEventsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.ListTimelineEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/ListTimelineEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTimelineEventsIterable listTimelineEventsPaginator(ListTimelineEventsRequest listTimelineEventsRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        return new ListTimelineEventsIterable(this, applyPaginatorUserAgent(listTimelineEventsRequest));
    }

    /**
     * <p>
     * Adds a resource policy to the specified response plan. The resource policy is used to share the response plan
     * using Resource Access Manager (RAM). For more information about cross-account sharing, see <a href=
     * "https://docs.aws.amazon.com/incident-manager/latest/userguide/incident-manager-cross-account-cross-region.html"
     * >Cross-Region and cross-account incident management</a>.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return Result of the PutResourcePolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.PutResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/PutResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutResourcePolicyResponse putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                    .withOperationName("PutResourcePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Used to start an incident from CloudWatch alarms, EventBridge events, or manually.
     * </p>
     *
     * @param startIncidentRequest
     * @return Result of the StartIncident operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.StartIncident
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/StartIncident" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartIncidentResponse startIncident(StartIncidentRequest startIncidentRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException, InternalServerException,
            AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartIncidentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartIncidentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startIncidentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartIncident");

            return clientHandler.execute(new ClientExecutionParams<StartIncidentRequest, StartIncidentResponse>()
                    .withOperationName("StartIncident").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startIncidentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartIncidentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a tag to a response plan.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ServiceQuotaExceededException,
            ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException, InternalServerException,
            AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update deletion protection to either allow or deny deletion of the final Region in a replication set.
     * </p>
     *
     * @param updateDeletionProtectionRequest
     * @return Result of the UpdateDeletionProtection operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.UpdateDeletionProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/UpdateDeletionProtection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDeletionProtectionResponse updateDeletionProtection(
            UpdateDeletionProtectionRequest updateDeletionProtectionRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDeletionProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDeletionProtectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDeletionProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDeletionProtection");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateDeletionProtectionRequest, UpdateDeletionProtectionResponse>()
                            .withOperationName("UpdateDeletionProtection").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateDeletionProtectionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateDeletionProtectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update the details of an incident record. You can use this operation to update an incident record from the
     * defined chat channel. For more information about using actions in chat channels, see <a
     * href="https://docs.aws.amazon.com/incident-manager/latest/userguide/chat.html#chat-interact">Interacting through
     * chat</a>.
     * </p>
     *
     * @param updateIncidentRecordRequest
     * @return Result of the UpdateIncidentRecord operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.UpdateIncidentRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/UpdateIncidentRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateIncidentRecordResponse updateIncidentRecord(UpdateIncidentRecordRequest updateIncidentRecordRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateIncidentRecordResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateIncidentRecordResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateIncidentRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIncidentRecord");

            return clientHandler.execute(new ClientExecutionParams<UpdateIncidentRecordRequest, UpdateIncidentRecordResponse>()
                    .withOperationName("UpdateIncidentRecord").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateIncidentRecordRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateIncidentRecordRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Add or remove related items from the related items tab of an incident record.
     * </p>
     *
     * @param updateRelatedItemsRequest
     * @return Result of the UpdateRelatedItems operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.UpdateRelatedItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/UpdateRelatedItems"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateRelatedItemsResponse updateRelatedItems(UpdateRelatedItemsRequest updateRelatedItemsRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRelatedItemsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRelatedItemsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRelatedItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRelatedItems");

            return clientHandler.execute(new ClientExecutionParams<UpdateRelatedItemsRequest, UpdateRelatedItemsResponse>()
                    .withOperationName("UpdateRelatedItems").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateRelatedItemsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRelatedItemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Add or delete Regions from your replication set.
     * </p>
     *
     * @param updateReplicationSetRequest
     * @return Result of the UpdateReplicationSet operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.UpdateReplicationSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/UpdateReplicationSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateReplicationSetResponse updateReplicationSet(UpdateReplicationSetRequest updateReplicationSetRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateReplicationSetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateReplicationSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateReplicationSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateReplicationSet");

            return clientHandler.execute(new ClientExecutionParams<UpdateReplicationSetRequest, UpdateReplicationSetResponse>()
                    .withOperationName("UpdateReplicationSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateReplicationSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateReplicationSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified response plan.
     * </p>
     *
     * @param updateResponsePlanRequest
     * @return Result of the UpdateResponsePlan operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.UpdateResponsePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/UpdateResponsePlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateResponsePlanResponse updateResponsePlan(UpdateResponsePlanRequest updateResponsePlanRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResponsePlanResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateResponsePlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResponsePlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResponsePlan");

            return clientHandler.execute(new ClientExecutionParams<UpdateResponsePlanRequest, UpdateResponsePlanResponse>()
                    .withOperationName("UpdateResponsePlan").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateResponsePlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateResponsePlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a timeline event. You can update events of type <code>Custom Event</code>.
     * </p>
     *
     * @param updateTimelineEventRequest
     * @return Result of the UpdateTimelineEvent operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmIncidentsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmIncidentsClient.UpdateTimelineEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-incidents-2018-05-10/UpdateTimelineEvent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateTimelineEventResponse updateTimelineEvent(UpdateTimelineEventRequest updateTimelineEventRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, SsmIncidentsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTimelineEventResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateTimelineEventResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTimelineEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Incidents");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTimelineEvent");

            return clientHandler.execute(new ClientExecutionParams<UpdateTimelineEventRequest, UpdateTimelineEventResponse>()
                    .withOperationName("UpdateTimelineEvent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateTimelineEventRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTimelineEventRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsmIncidentsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends SsmIncidentsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public SsmIncidentsWaiter waiter() {
        return SsmIncidentsWaiter.builder().client(this).build();
    }
}
