/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmincidents.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about the Systems Manager automation document that will be used as a runbook during an incident.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SsmAutomation implements SdkPojo, Serializable, ToCopyableBuilder<SsmAutomation.Builder, SsmAutomation> {
    private static final SdkField<String> DOCUMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("documentName").getter(getter(SsmAutomation::documentName)).setter(setter(Builder::documentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("documentName").build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("documentVersion").getter(getter(SsmAutomation::documentVersion))
            .setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("documentVersion").build()).build();

    private static final SdkField<Map<String, DynamicSsmParameterValue>> DYNAMIC_PARAMETERS_FIELD = SdkField
            .<Map<String, DynamicSsmParameterValue>> builder(MarshallingType.MAP)
            .memberName("dynamicParameters")
            .getter(getter(SsmAutomation::dynamicParameters))
            .setter(setter(Builder::dynamicParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dynamicParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<DynamicSsmParameterValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(DynamicSsmParameterValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, List<String>>> PARAMETERS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("parameters")
            .getter(getter(SsmAutomation::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(SsmAutomation::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<String> TARGET_ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("targetAccount").getter(getter(SsmAutomation::targetAccountAsString))
            .setter(setter(Builder::targetAccount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetAccount").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOCUMENT_NAME_FIELD,
            DOCUMENT_VERSION_FIELD, DYNAMIC_PARAMETERS_FIELD, PARAMETERS_FIELD, ROLE_ARN_FIELD, TARGET_ACCOUNT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String documentName;

    private final String documentVersion;

    private final Map<String, DynamicSsmParameterValue> dynamicParameters;

    private final Map<String, List<String>> parameters;

    private final String roleArn;

    private final String targetAccount;

    private SsmAutomation(BuilderImpl builder) {
        this.documentName = builder.documentName;
        this.documentVersion = builder.documentVersion;
        this.dynamicParameters = builder.dynamicParameters;
        this.parameters = builder.parameters;
        this.roleArn = builder.roleArn;
        this.targetAccount = builder.targetAccount;
    }

    /**
     * <p>
     * The automation document's name.
     * </p>
     * 
     * @return The automation document's name.
     */
    public final String documentName() {
        return documentName;
    }

    /**
     * <p>
     * The automation document's version to use when running.
     * </p>
     * 
     * @return The automation document's version to use when running.
     */
    public final String documentVersion() {
        return documentVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the DynamicParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDynamicParameters() {
        return dynamicParameters != null && !(dynamicParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The key-value pair to resolve dynamic parameter values when processing a Systems Manager Automation runbook.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDynamicParameters} method.
     * </p>
     * 
     * @return The key-value pair to resolve dynamic parameter values when processing a Systems Manager Automation
     *         runbook.
     */
    public final Map<String, DynamicSsmParameterValue> dynamicParameters() {
        return dynamicParameters;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The key-value pair parameters to use when running the automation document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return The key-value pair parameters to use when running the automation document.
     */
    public final Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the role that the automation document will assume when running commands.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the role that the automation document will assume when running
     *         commands.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The account that the automation document will be run in. This can be in either the management account or an
     * application account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetAccount}
     * will return {@link SsmTargetAccount#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #targetAccountAsString}.
     * </p>
     * 
     * @return The account that the automation document will be run in. This can be in either the management account or
     *         an application account.
     * @see SsmTargetAccount
     */
    public final SsmTargetAccount targetAccount() {
        return SsmTargetAccount.fromValue(targetAccount);
    }

    /**
     * <p>
     * The account that the automation document will be run in. This can be in either the management account or an
     * application account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetAccount}
     * will return {@link SsmTargetAccount#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #targetAccountAsString}.
     * </p>
     * 
     * @return The account that the automation document will be run in. This can be in either the management account or
     *         an application account.
     * @see SsmTargetAccount
     */
    public final String targetAccountAsString() {
        return targetAccount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(documentName());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasDynamicParameters() ? dynamicParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(targetAccountAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SsmAutomation)) {
            return false;
        }
        SsmAutomation other = (SsmAutomation) obj;
        return Objects.equals(documentName(), other.documentName()) && Objects.equals(documentVersion(), other.documentVersion())
                && hasDynamicParameters() == other.hasDynamicParameters()
                && Objects.equals(dynamicParameters(), other.dynamicParameters()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(targetAccountAsString(), other.targetAccountAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SsmAutomation").add("DocumentName", documentName()).add("DocumentVersion", documentVersion())
                .add("DynamicParameters", hasDynamicParameters() ? dynamicParameters() : null)
                .add("Parameters", hasParameters() ? parameters() : null).add("RoleArn", roleArn())
                .add("TargetAccount", targetAccountAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "documentName":
            return Optional.ofNullable(clazz.cast(documentName()));
        case "documentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "dynamicParameters":
            return Optional.ofNullable(clazz.cast(dynamicParameters()));
        case "parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "targetAccount":
            return Optional.ofNullable(clazz.cast(targetAccountAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SsmAutomation, T> g) {
        return obj -> g.apply((SsmAutomation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SsmAutomation> {
        /**
         * <p>
         * The automation document's name.
         * </p>
         * 
         * @param documentName
         *        The automation document's name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentName(String documentName);

        /**
         * <p>
         * The automation document's version to use when running.
         * </p>
         * 
         * @param documentVersion
         *        The automation document's version to use when running.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The key-value pair to resolve dynamic parameter values when processing a Systems Manager Automation runbook.
         * </p>
         * 
         * @param dynamicParameters
         *        The key-value pair to resolve dynamic parameter values when processing a Systems Manager Automation
         *        runbook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dynamicParameters(Map<String, DynamicSsmParameterValue> dynamicParameters);

        /**
         * <p>
         * The key-value pair parameters to use when running the automation document.
         * </p>
         * 
         * @param parameters
         *        The key-value pair parameters to use when running the automation document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the role that the automation document will assume when running commands.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the role that the automation document will assume when running
         *        commands.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The account that the automation document will be run in. This can be in either the management account or an
         * application account.
         * </p>
         * 
         * @param targetAccount
         *        The account that the automation document will be run in. This can be in either the management account
         *        or an application account.
         * @see SsmTargetAccount
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SsmTargetAccount
         */
        Builder targetAccount(String targetAccount);

        /**
         * <p>
         * The account that the automation document will be run in. This can be in either the management account or an
         * application account.
         * </p>
         * 
         * @param targetAccount
         *        The account that the automation document will be run in. This can be in either the management account
         *        or an application account.
         * @see SsmTargetAccount
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SsmTargetAccount
         */
        Builder targetAccount(SsmTargetAccount targetAccount);
    }

    static final class BuilderImpl implements Builder {
        private String documentName;

        private String documentVersion;

        private Map<String, DynamicSsmParameterValue> dynamicParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, List<String>> parameters = DefaultSdkAutoConstructMap.getInstance();

        private String roleArn;

        private String targetAccount;

        private BuilderImpl() {
        }

        private BuilderImpl(SsmAutomation model) {
            documentName(model.documentName);
            documentVersion(model.documentVersion);
            dynamicParameters(model.dynamicParameters);
            parameters(model.parameters);
            roleArn(model.roleArn);
            targetAccount(model.targetAccount);
        }

        public final String getDocumentName() {
            return documentName;
        }

        public final void setDocumentName(String documentName) {
            this.documentName = documentName;
        }

        @Override
        public final Builder documentName(String documentName) {
            this.documentName = documentName;
            return this;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final Map<String, DynamicSsmParameterValue.Builder> getDynamicParameters() {
            Map<String, DynamicSsmParameterValue.Builder> result = DynamicSsmParametersCopier
                    .copyToBuilder(this.dynamicParameters);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setDynamicParameters(Map<String, DynamicSsmParameterValue.BuilderImpl> dynamicParameters) {
            this.dynamicParameters = DynamicSsmParametersCopier.copyFromBuilder(dynamicParameters);
        }

        @Override
        public final Builder dynamicParameters(Map<String, DynamicSsmParameterValue> dynamicParameters) {
            this.dynamicParameters = DynamicSsmParametersCopier.copy(dynamicParameters);
            return this;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            if (parameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return parameters;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = SsmParametersCopier.copy(parameters);
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = SsmParametersCopier.copy(parameters);
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getTargetAccount() {
            return targetAccount;
        }

        public final void setTargetAccount(String targetAccount) {
            this.targetAccount = targetAccount;
        }

        @Override
        public final Builder targetAccount(String targetAccount) {
            this.targetAccount = targetAccount;
            return this;
        }

        @Override
        public final Builder targetAccount(SsmTargetAccount targetAccount) {
            this.targetAccount(targetAccount == null ? null : targetAccount.toString());
            return this;
        }

        @Override
        public SsmAutomation build() {
            return new SsmAutomation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
