/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmsap;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ssmsap.model.ConflictException;
import software.amazon.awssdk.services.ssmsap.model.DeleteResourcePermissionRequest;
import software.amazon.awssdk.services.ssmsap.model.DeleteResourcePermissionResponse;
import software.amazon.awssdk.services.ssmsap.model.DeregisterApplicationRequest;
import software.amazon.awssdk.services.ssmsap.model.DeregisterApplicationResponse;
import software.amazon.awssdk.services.ssmsap.model.GetApplicationRequest;
import software.amazon.awssdk.services.ssmsap.model.GetApplicationResponse;
import software.amazon.awssdk.services.ssmsap.model.GetComponentRequest;
import software.amazon.awssdk.services.ssmsap.model.GetComponentResponse;
import software.amazon.awssdk.services.ssmsap.model.GetDatabaseRequest;
import software.amazon.awssdk.services.ssmsap.model.GetDatabaseResponse;
import software.amazon.awssdk.services.ssmsap.model.GetOperationRequest;
import software.amazon.awssdk.services.ssmsap.model.GetOperationResponse;
import software.amazon.awssdk.services.ssmsap.model.GetResourcePermissionRequest;
import software.amazon.awssdk.services.ssmsap.model.GetResourcePermissionResponse;
import software.amazon.awssdk.services.ssmsap.model.InternalServerException;
import software.amazon.awssdk.services.ssmsap.model.ListApplicationsRequest;
import software.amazon.awssdk.services.ssmsap.model.ListApplicationsResponse;
import software.amazon.awssdk.services.ssmsap.model.ListComponentsRequest;
import software.amazon.awssdk.services.ssmsap.model.ListComponentsResponse;
import software.amazon.awssdk.services.ssmsap.model.ListDatabasesRequest;
import software.amazon.awssdk.services.ssmsap.model.ListDatabasesResponse;
import software.amazon.awssdk.services.ssmsap.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ssmsap.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ssmsap.model.PutResourcePermissionRequest;
import software.amazon.awssdk.services.ssmsap.model.PutResourcePermissionResponse;
import software.amazon.awssdk.services.ssmsap.model.RegisterApplicationRequest;
import software.amazon.awssdk.services.ssmsap.model.RegisterApplicationResponse;
import software.amazon.awssdk.services.ssmsap.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ssmsap.model.SsmSapException;
import software.amazon.awssdk.services.ssmsap.model.SsmSapRequest;
import software.amazon.awssdk.services.ssmsap.model.TagResourceRequest;
import software.amazon.awssdk.services.ssmsap.model.TagResourceResponse;
import software.amazon.awssdk.services.ssmsap.model.UntagResourceRequest;
import software.amazon.awssdk.services.ssmsap.model.UntagResourceResponse;
import software.amazon.awssdk.services.ssmsap.model.UpdateApplicationSettingsRequest;
import software.amazon.awssdk.services.ssmsap.model.UpdateApplicationSettingsResponse;
import software.amazon.awssdk.services.ssmsap.model.ValidationException;
import software.amazon.awssdk.services.ssmsap.paginators.ListApplicationsIterable;
import software.amazon.awssdk.services.ssmsap.paginators.ListComponentsIterable;
import software.amazon.awssdk.services.ssmsap.paginators.ListDatabasesIterable;
import software.amazon.awssdk.services.ssmsap.transform.DeleteResourcePermissionRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.DeregisterApplicationRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.GetComponentRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.GetDatabaseRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.GetOperationRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.GetResourcePermissionRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.ListComponentsRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.ListDatabasesRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.PutResourcePermissionRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.RegisterApplicationRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.UpdateApplicationSettingsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SsmSapClient}.
 *
 * @see SsmSapClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsmSapClient implements SsmSapClient {
    private static final Logger log = Logger.loggerFor(DefaultSsmSapClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSsmSapClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p/>
     *
     * @param deleteResourcePermissionRequest
     * @return Result of the DeleteResourcePermission operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.DeleteResourcePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/DeleteResourcePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResourcePermissionResponse deleteResourcePermission(
            DeleteResourcePermissionRequest deleteResourcePermissionRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourcePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourcePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePermission");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteResourcePermissionRequest, DeleteResourcePermissionResponse>()
                            .withOperationName("DeleteResourcePermission").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteResourcePermissionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteResourcePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param deregisterApplicationRequest
     * @return Result of the DeregisterApplication operation returned by the service.
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.DeregisterApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/DeregisterApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeregisterApplicationResponse deregisterApplication(DeregisterApplicationRequest deregisterApplicationRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterApplication");

            return clientHandler.execute(new ClientExecutionParams<DeregisterApplicationRequest, DeregisterApplicationResponse>()
                    .withOperationName("DeregisterApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deregisterApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeregisterApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param getApplicationRequest
     * @return Result of the GetApplication operation returned by the service.
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.GetApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/GetApplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetApplicationResponse getApplication(GetApplicationRequest getApplicationRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplication");

            return clientHandler.execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                    .withOperationName("GetApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param getComponentRequest
     * @return Result of the GetComponent operation returned by the service.
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.GetComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/GetComponent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetComponentResponse getComponent(GetComponentRequest getComponentRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetComponentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetComponentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComponent");

            return clientHandler.execute(new ClientExecutionParams<GetComponentRequest, GetComponentResponse>()
                    .withOperationName("GetComponent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getComponentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetComponentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param getDatabaseRequest
     * @return Result of the GetDatabase operation returned by the service.
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.GetDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/GetDatabase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDatabaseResponse getDatabase(GetDatabaseRequest getDatabaseRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDatabaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDatabase");

            return clientHandler.execute(new ClientExecutionParams<GetDatabaseRequest, GetDatabaseResponse>()
                    .withOperationName("GetDatabase").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDatabaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDatabaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param getOperationRequest
     * @return Result of the GetOperation operation returned by the service.
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.GetOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/GetOperation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetOperationResponse getOperation(GetOperationRequest getOperationRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOperationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetOperationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOperation");

            return clientHandler.execute(new ClientExecutionParams<GetOperationRequest, GetOperationResponse>()
                    .withOperationName("GetOperation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getOperationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetOperationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param getResourcePermissionRequest
     * @return Result of the GetResourcePermission operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.GetResourcePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/GetResourcePermission" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetResourcePermissionResponse getResourcePermission(GetResourcePermissionRequest getResourcePermissionRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourcePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePermission");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePermissionRequest, GetResourcePermissionResponse>()
                    .withOperationName("GetResourcePermission").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResourcePermissionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param listApplicationsRequest
     * @return Result of the ListApplications operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/ListApplications" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListApplicationsResponse listApplications(ListApplicationsRequest listApplicationsRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListApplicationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");

            return clientHandler.execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                    .withOperationName("ListApplications").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listApplicationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplications(software.amazon.awssdk.services.ssmsap.model.ListApplicationsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmsap.paginators.ListApplicationsIterable responses = client.listApplicationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmsap.paginators.ListApplicationsIterable responses = client
     *             .listApplicationsPaginator(request);
     *     for (software.amazon.awssdk.services.ssmsap.model.ListApplicationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmsap.paginators.ListApplicationsIterable responses = client.listApplicationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplications(software.amazon.awssdk.services.ssmsap.model.ListApplicationsRequest)} operation.</b>
     * </p>
     *
     * @param listApplicationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/ListApplications" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListApplicationsIterable listApplicationsPaginator(ListApplicationsRequest listApplicationsRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmSapException {
        return new ListApplicationsIterable(this, applyPaginatorUserAgent(listApplicationsRequest));
    }

    /**
     * <p/>
     *
     * @param listComponentsRequest
     * @return Result of the ListComponents operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/ListComponents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListComponentsResponse listComponents(ListComponentsRequest listComponentsRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListComponentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListComponentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComponents");

            return clientHandler.execute(new ClientExecutionParams<ListComponentsRequest, ListComponentsResponse>()
                    .withOperationName("ListComponents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listComponentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListComponentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     * <br/>
     * <p>
     * This is a variant of {@link #listComponents(software.amazon.awssdk.services.ssmsap.model.ListComponentsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmsap.paginators.ListComponentsIterable responses = client.listComponentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmsap.paginators.ListComponentsIterable responses = client.listComponentsPaginator(request);
     *     for (software.amazon.awssdk.services.ssmsap.model.ListComponentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmsap.paginators.ListComponentsIterable responses = client.listComponentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listComponents(software.amazon.awssdk.services.ssmsap.model.ListComponentsRequest)} operation.</b>
     * </p>
     *
     * @param listComponentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/ListComponents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListComponentsIterable listComponentsPaginator(ListComponentsRequest listComponentsRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmSapException {
        return new ListComponentsIterable(this, applyPaginatorUserAgent(listComponentsRequest));
    }

    /**
     * <p/>
     *
     * @param listDatabasesRequest
     * @return Result of the ListDatabases operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/ListDatabases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDatabasesResponse listDatabases(ListDatabasesRequest listDatabasesRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDatabasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatabasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatabases");

            return clientHandler.execute(new ClientExecutionParams<ListDatabasesRequest, ListDatabasesResponse>()
                    .withOperationName("ListDatabases").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDatabasesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDatabasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     * <br/>
     * <p>
     * This is a variant of {@link #listDatabases(software.amazon.awssdk.services.ssmsap.model.ListDatabasesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmsap.paginators.ListDatabasesIterable responses = client.listDatabasesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmsap.paginators.ListDatabasesIterable responses = client.listDatabasesPaginator(request);
     *     for (software.amazon.awssdk.services.ssmsap.model.ListDatabasesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmsap.paginators.ListDatabasesIterable responses = client.listDatabasesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatabases(software.amazon.awssdk.services.ssmsap.model.ListDatabasesRequest)} operation.</b>
     * </p>
     *
     * @param listDatabasesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/ListDatabases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDatabasesIterable listDatabasesPaginator(ListDatabasesRequest listDatabasesRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmSapException {
        return new ListDatabasesIterable(this, applyPaginatorUserAgent(listDatabasesRequest));
    }

    /**
     * <p/>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, ConflictException, AwsServiceException, SdkClientException,
            SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param putResourcePermissionRequest
     * @return Result of the PutResourcePermission operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.PutResourcePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/PutResourcePermission" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutResourcePermissionResponse putResourcePermission(PutResourcePermissionRequest putResourcePermissionRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResourcePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutResourcePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePermission");

            return clientHandler.execute(new ClientExecutionParams<PutResourcePermissionRequest, PutResourcePermissionResponse>()
                    .withOperationName("PutResourcePermission").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putResourcePermissionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutResourcePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param registerApplicationRequest
     * @return Result of the RegisterApplication operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.RegisterApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/RegisterApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RegisterApplicationResponse registerApplication(RegisterApplicationRequest registerApplicationRequest)
            throws ValidationException, ConflictException, InternalServerException, AwsServiceException, SdkClientException,
            SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RegisterApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterApplication");

            return clientHandler.execute(new ClientExecutionParams<RegisterApplicationRequest, RegisterApplicationResponse>()
                    .withOperationName("RegisterApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(registerApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RegisterApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, ConflictException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, ConflictException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p/>
     *
     * @param updateApplicationSettingsRequest
     * @return Result of the UpdateApplicationSettings operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.UpdateApplicationSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssmsap-2018-05-10/UpdateApplicationSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateApplicationSettingsResponse updateApplicationSettings(
            UpdateApplicationSettingsRequest updateApplicationSettingsRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateApplicationSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateApplicationSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SsmSap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplicationSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationSettingsRequest, UpdateApplicationSettingsResponse>()
                            .withOperationName("UpdateApplicationSettings").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateApplicationSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateApplicationSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsmSapException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends SsmSapRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
