/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides detailed information about a medical transcription job.
 * </p>
 * <p>
 * To view the status of the specified medical transcription job, check the <code>TranscriptionJobStatus</code> field.
 * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location specified
 * in <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code> provides details
 * on why your transcription job failed.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MedicalTranscriptionJob implements SdkPojo, Serializable,
        ToCopyableBuilder<MedicalTranscriptionJob.Builder, MedicalTranscriptionJob> {
    private static final SdkField<String> MEDICAL_TRANSCRIPTION_JOB_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("MedicalTranscriptionJobName")
            .getter(getter(MedicalTranscriptionJob::medicalTranscriptionJobName))
            .setter(setter(Builder::medicalTranscriptionJobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MedicalTranscriptionJobName")
                    .build()).build();

    private static final SdkField<String> TRANSCRIPTION_JOB_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TranscriptionJobStatus").getter(getter(MedicalTranscriptionJob::transcriptionJobStatusAsString))
            .setter(setter(Builder::transcriptionJobStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscriptionJobStatus").build())
            .build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(MedicalTranscriptionJob::languageCodeAsString))
            .setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<Integer> MEDIA_SAMPLE_RATE_HERTZ_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MediaSampleRateHertz").getter(getter(MedicalTranscriptionJob::mediaSampleRateHertz))
            .setter(setter(Builder::mediaSampleRateHertz))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaSampleRateHertz").build())
            .build();

    private static final SdkField<String> MEDIA_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MediaFormat").getter(getter(MedicalTranscriptionJob::mediaFormatAsString))
            .setter(setter(Builder::mediaFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaFormat").build()).build();

    private static final SdkField<Media> MEDIA_FIELD = SdkField.<Media> builder(MarshallingType.SDK_POJO).memberName("Media")
            .getter(getter(MedicalTranscriptionJob::media)).setter(setter(Builder::media)).constructor(Media::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Media").build()).build();

    private static final SdkField<MedicalTranscript> TRANSCRIPT_FIELD = SdkField
            .<MedicalTranscript> builder(MarshallingType.SDK_POJO).memberName("Transcript")
            .getter(getter(MedicalTranscriptionJob::transcript)).setter(setter(Builder::transcript))
            .constructor(MedicalTranscript::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Transcript").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(MedicalTranscriptionJob::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(MedicalTranscriptionJob::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> COMPLETION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletionTime").getter(getter(MedicalTranscriptionJob::completionTime))
            .setter(setter(Builder::completionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionTime").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(MedicalTranscriptionJob::failureReason))
            .setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final SdkField<MedicalTranscriptionSetting> SETTINGS_FIELD = SdkField
            .<MedicalTranscriptionSetting> builder(MarshallingType.SDK_POJO).memberName("Settings")
            .getter(getter(MedicalTranscriptionJob::settings)).setter(setter(Builder::settings))
            .constructor(MedicalTranscriptionSetting::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Settings").build()).build();

    private static final SdkField<String> CONTENT_IDENTIFICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContentIdentificationType").getter(getter(MedicalTranscriptionJob::contentIdentificationTypeAsString))
            .setter(setter(Builder::contentIdentificationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentIdentificationType").build())
            .build();

    private static final SdkField<String> SPECIALTY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Specialty").getter(getter(MedicalTranscriptionJob::specialtyAsString))
            .setter(setter(Builder::specialty))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Specialty").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(MedicalTranscriptionJob::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(MedicalTranscriptionJob::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            MEDICAL_TRANSCRIPTION_JOB_NAME_FIELD, TRANSCRIPTION_JOB_STATUS_FIELD, LANGUAGE_CODE_FIELD,
            MEDIA_SAMPLE_RATE_HERTZ_FIELD, MEDIA_FORMAT_FIELD, MEDIA_FIELD, TRANSCRIPT_FIELD, START_TIME_FIELD,
            CREATION_TIME_FIELD, COMPLETION_TIME_FIELD, FAILURE_REASON_FIELD, SETTINGS_FIELD, CONTENT_IDENTIFICATION_TYPE_FIELD,
            SPECIALTY_FIELD, TYPE_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String medicalTranscriptionJobName;

    private final String transcriptionJobStatus;

    private final String languageCode;

    private final Integer mediaSampleRateHertz;

    private final String mediaFormat;

    private final Media media;

    private final MedicalTranscript transcript;

    private final Instant startTime;

    private final Instant creationTime;

    private final Instant completionTime;

    private final String failureReason;

    private final MedicalTranscriptionSetting settings;

    private final String contentIdentificationType;

    private final String specialty;

    private final String type;

    private final List<Tag> tags;

    private MedicalTranscriptionJob(BuilderImpl builder) {
        this.medicalTranscriptionJobName = builder.medicalTranscriptionJobName;
        this.transcriptionJobStatus = builder.transcriptionJobStatus;
        this.languageCode = builder.languageCode;
        this.mediaSampleRateHertz = builder.mediaSampleRateHertz;
        this.mediaFormat = builder.mediaFormat;
        this.media = builder.media;
        this.transcript = builder.transcript;
        this.startTime = builder.startTime;
        this.creationTime = builder.creationTime;
        this.completionTime = builder.completionTime;
        this.failureReason = builder.failureReason;
        this.settings = builder.settings;
        this.contentIdentificationType = builder.contentIdentificationType;
        this.specialty = builder.specialty;
        this.type = builder.type;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the medical transcription job. Job names are case sensitive and must be unique within an Amazon Web
     * Services account.
     * </p>
     * 
     * @return The name of the medical transcription job. Job names are case sensitive and must be unique within an
     *         Amazon Web Services account.
     */
    public final String medicalTranscriptionJobName() {
        return medicalTranscriptionJobName;
    }

    /**
     * <p>
     * Provides the status of the specified medical transcription job.
     * </p>
     * <p>
     * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     * specified in <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code>
     * provides details on why your transcription job failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptionJobStatus} will return {@link TranscriptionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptionJobStatusAsString}.
     * </p>
     * 
     * @return Provides the status of the specified medical transcription job.</p>
     *         <p>
     *         If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     *         specified in <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>,
     *         <code>FailureReason</code> provides details on why your transcription job failed.
     * @see TranscriptionJobStatus
     */
    public final TranscriptionJobStatus transcriptionJobStatus() {
        return TranscriptionJobStatus.fromValue(transcriptionJobStatus);
    }

    /**
     * <p>
     * Provides the status of the specified medical transcription job.
     * </p>
     * <p>
     * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     * specified in <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code>
     * provides details on why your transcription job failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptionJobStatus} will return {@link TranscriptionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptionJobStatusAsString}.
     * </p>
     * 
     * @return Provides the status of the specified medical transcription job.</p>
     *         <p>
     *         If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     *         specified in <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>,
     *         <code>FailureReason</code> provides details on why your transcription job failed.
     * @see TranscriptionJobStatus
     */
    public final String transcriptionJobStatusAsString() {
        return transcriptionJobStatus;
    }

    /**
     * <p>
     * The language code used to create your medical transcription job. US English (<code>en-US</code>) is the only
     * supported language for medical transcriptions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code used to create your medical transcription job. US English (<code>en-US</code>) is the
     *         only supported language for medical transcriptions.
     * @see LanguageCode
     */
    public final LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * The language code used to create your medical transcription job. US English (<code>en-US</code>) is the only
     * supported language for medical transcriptions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code used to create your medical transcription job. US English (<code>en-US</code>) is the
     *         only supported language for medical transcriptions.
     * @see LanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * <p>
     * The sample rate, in Hertz, of the audio track in your input media file.
     * </p>
     * 
     * @return The sample rate, in Hertz, of the audio track in your input media file.
     */
    public final Integer mediaSampleRateHertz() {
        return mediaSampleRateHertz;
    }

    /**
     * <p>
     * The format of the input media file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input media file.
     * @see MediaFormat
     */
    public final MediaFormat mediaFormat() {
        return MediaFormat.fromValue(mediaFormat);
    }

    /**
     * <p>
     * The format of the input media file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input media file.
     * @see MediaFormat
     */
    public final String mediaFormatAsString() {
        return mediaFormat;
    }

    /**
     * Returns the value of the Media property for this object.
     * 
     * @return The value of the Media property for this object.
     */
    public final Media media() {
        return media;
    }

    /**
     * <p>
     * Provides you with the Amazon S3 URI you can use to access your transcript.
     * </p>
     * 
     * @return Provides you with the Amazon S3 URI you can use to access your transcript.
     */
    public final MedicalTranscript transcript() {
        return transcript;
    }

    /**
     * <p>
     * The date and time the specified medical transcription job began processing.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing at 12:32 PM
     * UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified medical transcription job began processing.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing at
     *         12:32 PM UTC-7 on May 4, 2022.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time the specified medical transcription job request was made.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing at 12:32 PM
     * UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified medical transcription job request was made.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing at
     *         12:32 PM UTC-7 on May 4, 2022.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The date and time the specified medical transcription job finished processing.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing at 12:33 PM
     * UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified medical transcription job finished processing.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing at
     *         12:33 PM UTC-7 on May 4, 2022.
     */
    public final Instant completionTime() {
        return completionTime;
    }

    /**
     * <p>
     * If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains information
     * about why the transcription job request failed.
     * </p>
     * <p>
     * The <code>FailureReason</code> field contains one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Unsupported media format</code>.
     * </p>
     * <p>
     * The media format specified in <code>MediaFormat</code> isn't valid. Refer to <b>MediaFormat</b> for a list of
     * supported formats.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>The media format provided does not match the detected media format</code>.
     * </p>
     * <p>
     * The media format specified in <code>MediaFormat</code> doesn't match the format of the input file. Check the
     * media format of your media file and correct the specified value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid sample rate for audio file</code>.
     * </p>
     * <p>
     * The sample rate specified in <code>MediaSampleRateHertz</code> isn't valid. The sample rate must be between
     * 16,000 and 48,000 Hertz.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>The sample rate provided does not match the detected sample rate</code>.
     * </p>
     * <p>
     * The sample rate specified in <code>MediaSampleRateHertz</code> doesn't match the sample rate detected in your
     * input media file. Check the sample rate of your media file and correct the specified value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid file size: file size too large</code>.
     * </p>
     * <p>
     * The size of your media file is larger than what Amazon Transcribe can process. For more information, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid number of channels: number of channels too large</code>.
     * </p>
     * <p>
     * Your audio contains more channels than Amazon Transcribe is able to process. For more information, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and quotas</a>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
     *         information about why the transcription job request failed.</p>
     *         <p>
     *         The <code>FailureReason</code> field contains one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Unsupported media format</code>.
     *         </p>
     *         <p>
     *         The media format specified in <code>MediaFormat</code> isn't valid. Refer to <b>MediaFormat</b> for a
     *         list of supported formats.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>The media format provided does not match the detected media format</code>.
     *         </p>
     *         <p>
     *         The media format specified in <code>MediaFormat</code> doesn't match the format of the input file. Check
     *         the media format of your media file and correct the specified value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid sample rate for audio file</code>.
     *         </p>
     *         <p>
     *         The sample rate specified in <code>MediaSampleRateHertz</code> isn't valid. The sample rate must be
     *         between 16,000 and 48,000 Hertz.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>The sample rate provided does not match the detected sample rate</code>.
     *         </p>
     *         <p>
     *         The sample rate specified in <code>MediaSampleRateHertz</code> doesn't match the sample rate detected in
     *         your input media file. Check the sample rate of your media file and correct the specified value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid file size: file size too large</code>.
     *         </p>
     *         <p>
     *         The size of your media file is larger than what Amazon Transcribe can process. For more information,
     *         refer to <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and
     *         quotas</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid number of channels: number of channels too large</code>.
     *         </p>
     *         <p>
     *         Your audio contains more channels than Amazon Transcribe is able to process. For more information, refer
     *         to <a href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines
     *         and quotas</a>.
     *         </p>
     *         </li>
     */
    public final String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * Specify additional optional settings in your request, including channel identification, alternative
     * transcriptions, and speaker labeling; allows you to apply custom vocabularies to your medical transcription job.
     * </p>
     * 
     * @return Specify additional optional settings in your request, including channel identification, alternative
     *         transcriptions, and speaker labeling; allows you to apply custom vocabularies to your medical
     *         transcription job.
     */
    public final MedicalTranscriptionSetting settings() {
        return settings;
    }

    /**
     * <p>
     * Labels all personal health information (PHI) identified in your transcript. For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/phi-id.html">Identifying personal health information (PHI)
     * in a transcription</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #contentIdentificationType} will return {@link MedicalContentIdentificationType#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #contentIdentificationTypeAsString}.
     * </p>
     * 
     * @return Labels all personal health information (PHI) identified in your transcript. For more information, see <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/phi-id.html">Identifying personal health
     *         information (PHI) in a transcription</a>.
     * @see MedicalContentIdentificationType
     */
    public final MedicalContentIdentificationType contentIdentificationType() {
        return MedicalContentIdentificationType.fromValue(contentIdentificationType);
    }

    /**
     * <p>
     * Labels all personal health information (PHI) identified in your transcript. For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/phi-id.html">Identifying personal health information (PHI)
     * in a transcription</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #contentIdentificationType} will return {@link MedicalContentIdentificationType#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #contentIdentificationTypeAsString}.
     * </p>
     * 
     * @return Labels all personal health information (PHI) identified in your transcript. For more information, see <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/phi-id.html">Identifying personal health
     *         information (PHI) in a transcription</a>.
     * @see MedicalContentIdentificationType
     */
    public final String contentIdentificationTypeAsString() {
        return contentIdentificationType;
    }

    /**
     * <p>
     * Describes the medical specialty represented in your media.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #specialty} will
     * return {@link Specialty#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #specialtyAsString}.
     * </p>
     * 
     * @return Describes the medical specialty represented in your media.
     * @see Specialty
     */
    public final Specialty specialty() {
        return Specialty.fromValue(specialty);
    }

    /**
     * <p>
     * Describes the medical specialty represented in your media.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #specialty} will
     * return {@link Specialty#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #specialtyAsString}.
     * </p>
     * 
     * @return Describes the medical specialty represented in your media.
     * @see Specialty
     */
    public final String specialtyAsString() {
        return specialty;
    }

    /**
     * <p>
     * Indicates whether the input media is a dictation or a conversation, as specified in the
     * <code>StartMedicalTranscriptionJob</code> request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link Type#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Indicates whether the input media is a dictation or a conversation, as specified in the
     *         <code>StartMedicalTranscriptionJob</code> request.
     * @see Type
     */
    public final Type type() {
        return Type.fromValue(type);
    }

    /**
     * <p>
     * Indicates whether the input media is a dictation or a conversation, as specified in the
     * <code>StartMedicalTranscriptionJob</code> request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link Type#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Indicates whether the input media is a dictation or a conversation, as specified in the
     *         <code>StartMedicalTranscriptionJob</code> request.
     * @see Type
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags, each in the form of a key:value pair, assigned to the specified medical transcription job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags, each in the form of a key:value pair, assigned to the specified medical transcription job.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(medicalTranscriptionJobName());
        hashCode = 31 * hashCode + Objects.hashCode(transcriptionJobStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mediaSampleRateHertz());
        hashCode = 31 * hashCode + Objects.hashCode(mediaFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(media());
        hashCode = 31 * hashCode + Objects.hashCode(transcript());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(completionTime());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(settings());
        hashCode = 31 * hashCode + Objects.hashCode(contentIdentificationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(specialtyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MedicalTranscriptionJob)) {
            return false;
        }
        MedicalTranscriptionJob other = (MedicalTranscriptionJob) obj;
        return Objects.equals(medicalTranscriptionJobName(), other.medicalTranscriptionJobName())
                && Objects.equals(transcriptionJobStatusAsString(), other.transcriptionJobStatusAsString())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && Objects.equals(mediaSampleRateHertz(), other.mediaSampleRateHertz())
                && Objects.equals(mediaFormatAsString(), other.mediaFormatAsString()) && Objects.equals(media(), other.media())
                && Objects.equals(transcript(), other.transcript()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(completionTime(), other.completionTime())
                && Objects.equals(failureReason(), other.failureReason()) && Objects.equals(settings(), other.settings())
                && Objects.equals(contentIdentificationTypeAsString(), other.contentIdentificationTypeAsString())
                && Objects.equals(specialtyAsString(), other.specialtyAsString())
                && Objects.equals(typeAsString(), other.typeAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MedicalTranscriptionJob").add("MedicalTranscriptionJobName", medicalTranscriptionJobName())
                .add("TranscriptionJobStatus", transcriptionJobStatusAsString()).add("LanguageCode", languageCodeAsString())
                .add("MediaSampleRateHertz", mediaSampleRateHertz()).add("MediaFormat", mediaFormatAsString())
                .add("Media", media()).add("Transcript", transcript()).add("StartTime", startTime())
                .add("CreationTime", creationTime()).add("CompletionTime", completionTime())
                .add("FailureReason", failureReason()).add("Settings", settings())
                .add("ContentIdentificationType", contentIdentificationTypeAsString()).add("Specialty", specialtyAsString())
                .add("Type", typeAsString()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MedicalTranscriptionJobName":
            return Optional.ofNullable(clazz.cast(medicalTranscriptionJobName()));
        case "TranscriptionJobStatus":
            return Optional.ofNullable(clazz.cast(transcriptionJobStatusAsString()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "MediaSampleRateHertz":
            return Optional.ofNullable(clazz.cast(mediaSampleRateHertz()));
        case "MediaFormat":
            return Optional.ofNullable(clazz.cast(mediaFormatAsString()));
        case "Media":
            return Optional.ofNullable(clazz.cast(media()));
        case "Transcript":
            return Optional.ofNullable(clazz.cast(transcript()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "CompletionTime":
            return Optional.ofNullable(clazz.cast(completionTime()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "Settings":
            return Optional.ofNullable(clazz.cast(settings()));
        case "ContentIdentificationType":
            return Optional.ofNullable(clazz.cast(contentIdentificationTypeAsString()));
        case "Specialty":
            return Optional.ofNullable(clazz.cast(specialtyAsString()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MedicalTranscriptionJob, T> g) {
        return obj -> g.apply((MedicalTranscriptionJob) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MedicalTranscriptionJob> {
        /**
         * <p>
         * The name of the medical transcription job. Job names are case sensitive and must be unique within an Amazon
         * Web Services account.
         * </p>
         * 
         * @param medicalTranscriptionJobName
         *        The name of the medical transcription job. Job names are case sensitive and must be unique within an
         *        Amazon Web Services account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder medicalTranscriptionJobName(String medicalTranscriptionJobName);

        /**
         * <p>
         * Provides the status of the specified medical transcription job.
         * </p>
         * <p>
         * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
         * specified in <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code>
         * provides details on why your transcription job failed.
         * </p>
         * 
         * @param transcriptionJobStatus
         *        Provides the status of the specified medical transcription job.</p>
         *        <p>
         *        If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the
         *        location specified in <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>,
         *        <code>FailureReason</code> provides details on why your transcription job failed.
         * @see TranscriptionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptionJobStatus
         */
        Builder transcriptionJobStatus(String transcriptionJobStatus);

        /**
         * <p>
         * Provides the status of the specified medical transcription job.
         * </p>
         * <p>
         * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
         * specified in <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code>
         * provides details on why your transcription job failed.
         * </p>
         * 
         * @param transcriptionJobStatus
         *        Provides the status of the specified medical transcription job.</p>
         *        <p>
         *        If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the
         *        location specified in <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>,
         *        <code>FailureReason</code> provides details on why your transcription job failed.
         * @see TranscriptionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptionJobStatus
         */
        Builder transcriptionJobStatus(TranscriptionJobStatus transcriptionJobStatus);

        /**
         * <p>
         * The language code used to create your medical transcription job. US English (<code>en-US</code>) is the only
         * supported language for medical transcriptions.
         * </p>
         * 
         * @param languageCode
         *        The language code used to create your medical transcription job. US English (<code>en-US</code>) is
         *        the only supported language for medical transcriptions.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * The language code used to create your medical transcription job. US English (<code>en-US</code>) is the only
         * supported language for medical transcriptions.
         * </p>
         * 
         * @param languageCode
         *        The language code used to create your medical transcription job. US English (<code>en-US</code>) is
         *        the only supported language for medical transcriptions.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);

        /**
         * <p>
         * The sample rate, in Hertz, of the audio track in your input media file.
         * </p>
         * 
         * @param mediaSampleRateHertz
         *        The sample rate, in Hertz, of the audio track in your input media file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaSampleRateHertz(Integer mediaSampleRateHertz);

        /**
         * <p>
         * The format of the input media file.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input media file.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(String mediaFormat);

        /**
         * <p>
         * The format of the input media file.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input media file.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(MediaFormat mediaFormat);

        /**
         * Sets the value of the Media property for this object.
         *
         * @param media
         *        The new value for the Media property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder media(Media media);

        /**
         * Sets the value of the Media property for this object.
         *
         * This is a convenience method that creates an instance of the {@link Media.Builder} avoiding the need to
         * create one manually via {@link Media#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Media.Builder#build()} is called immediately and its result is
         * passed to {@link #media(Media)}.
         * 
         * @param media
         *        a consumer that will call methods on {@link Media.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #media(Media)
         */
        default Builder media(Consumer<Media.Builder> media) {
            return media(Media.builder().applyMutation(media).build());
        }

        /**
         * <p>
         * Provides you with the Amazon S3 URI you can use to access your transcript.
         * </p>
         * 
         * @param transcript
         *        Provides you with the Amazon S3 URI you can use to access your transcript.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transcript(MedicalTranscript transcript);

        /**
         * <p>
         * Provides you with the Amazon S3 URI you can use to access your transcript.
         * </p>
         * This is a convenience method that creates an instance of the {@link MedicalTranscript.Builder} avoiding the
         * need to create one manually via {@link MedicalTranscript#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MedicalTranscript.Builder#build()} is called immediately and its
         * result is passed to {@link #transcript(MedicalTranscript)}.
         * 
         * @param transcript
         *        a consumer that will call methods on {@link MedicalTranscript.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transcript(MedicalTranscript)
         */
        default Builder transcript(Consumer<MedicalTranscript.Builder> transcript) {
            return transcript(MedicalTranscript.builder().applyMutation(transcript).build());
        }

        /**
         * <p>
         * The date and time the specified medical transcription job began processing.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing at 12:32
         * PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param startTime
         *        The date and time the specified medical transcription job began processing.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing
         *        at 12:32 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time the specified medical transcription job request was made.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing at 12:32
         * PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param creationTime
         *        The date and time the specified medical transcription job request was made.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing
         *        at 12:32 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The date and time the specified medical transcription job finished processing.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing at 12:33
         * PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param completionTime
         *        The date and time the specified medical transcription job finished processing.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing
         *        at 12:33 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionTime(Instant completionTime);

        /**
         * <p>
         * If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
         * information about why the transcription job request failed.
         * </p>
         * <p>
         * The <code>FailureReason</code> field contains one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Unsupported media format</code>.
         * </p>
         * <p>
         * The media format specified in <code>MediaFormat</code> isn't valid. Refer to <b>MediaFormat</b> for a list of
         * supported formats.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>The media format provided does not match the detected media format</code>.
         * </p>
         * <p>
         * The media format specified in <code>MediaFormat</code> doesn't match the format of the input file. Check the
         * media format of your media file and correct the specified value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid sample rate for audio file</code>.
         * </p>
         * <p>
         * The sample rate specified in <code>MediaSampleRateHertz</code> isn't valid. The sample rate must be between
         * 16,000 and 48,000 Hertz.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>The sample rate provided does not match the detected sample rate</code>.
         * </p>
         * <p>
         * The sample rate specified in <code>MediaSampleRateHertz</code> doesn't match the sample rate detected in your
         * input media file. Check the sample rate of your media file and correct the specified value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid file size: file size too large</code>.
         * </p>
         * <p>
         * The size of your media file is larger than what Amazon Transcribe can process. For more information, refer to
         * <a href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and
         * quotas</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid number of channels: number of channels too large</code>.
         * </p>
         * <p>
         * Your audio contains more channels than Amazon Transcribe is able to process. For more information, refer to
         * <a href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and
         * quotas</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param failureReason
         *        If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
         *        information about why the transcription job request failed.</p>
         *        <p>
         *        The <code>FailureReason</code> field contains one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Unsupported media format</code>.
         *        </p>
         *        <p>
         *        The media format specified in <code>MediaFormat</code> isn't valid. Refer to <b>MediaFormat</b> for a
         *        list of supported formats.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>The media format provided does not match the detected media format</code>.
         *        </p>
         *        <p>
         *        The media format specified in <code>MediaFormat</code> doesn't match the format of the input file.
         *        Check the media format of your media file and correct the specified value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid sample rate for audio file</code>.
         *        </p>
         *        <p>
         *        The sample rate specified in <code>MediaSampleRateHertz</code> isn't valid. The sample rate must be
         *        between 16,000 and 48,000 Hertz.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>The sample rate provided does not match the detected sample rate</code>.
         *        </p>
         *        <p>
         *        The sample rate specified in <code>MediaSampleRateHertz</code> doesn't match the sample rate detected
         *        in your input media file. Check the sample rate of your media file and correct the specified value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid file size: file size too large</code>.
         *        </p>
         *        <p>
         *        The size of your media file is larger than what Amazon Transcribe can process. For more information,
         *        refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and
         *        quotas</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid number of channels: number of channels too large</code>.
         *        </p>
         *        <p>
         *        Your audio contains more channels than Amazon Transcribe is able to process. For more information,
         *        refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and
         *        quotas</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * Specify additional optional settings in your request, including channel identification, alternative
         * transcriptions, and speaker labeling; allows you to apply custom vocabularies to your medical transcription
         * job.
         * </p>
         * 
         * @param settings
         *        Specify additional optional settings in your request, including channel identification, alternative
         *        transcriptions, and speaker labeling; allows you to apply custom vocabularies to your medical
         *        transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settings(MedicalTranscriptionSetting settings);

        /**
         * <p>
         * Specify additional optional settings in your request, including channel identification, alternative
         * transcriptions, and speaker labeling; allows you to apply custom vocabularies to your medical transcription
         * job.
         * </p>
         * This is a convenience method that creates an instance of the {@link MedicalTranscriptionSetting.Builder}
         * avoiding the need to create one manually via {@link MedicalTranscriptionSetting#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MedicalTranscriptionSetting.Builder#build()} is called
         * immediately and its result is passed to {@link #settings(MedicalTranscriptionSetting)}.
         * 
         * @param settings
         *        a consumer that will call methods on {@link MedicalTranscriptionSetting.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #settings(MedicalTranscriptionSetting)
         */
        default Builder settings(Consumer<MedicalTranscriptionSetting.Builder> settings) {
            return settings(MedicalTranscriptionSetting.builder().applyMutation(settings).build());
        }

        /**
         * <p>
         * Labels all personal health information (PHI) identified in your transcript. For more information, see <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/phi-id.html">Identifying personal health information
         * (PHI) in a transcription</a>.
         * </p>
         * 
         * @param contentIdentificationType
         *        Labels all personal health information (PHI) identified in your transcript. For more information, see
         *        <a href="https://docs.aws.amazon.com/transcribe/latest/dg/phi-id.html">Identifying personal health
         *        information (PHI) in a transcription</a>.
         * @see MedicalContentIdentificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MedicalContentIdentificationType
         */
        Builder contentIdentificationType(String contentIdentificationType);

        /**
         * <p>
         * Labels all personal health information (PHI) identified in your transcript. For more information, see <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/phi-id.html">Identifying personal health information
         * (PHI) in a transcription</a>.
         * </p>
         * 
         * @param contentIdentificationType
         *        Labels all personal health information (PHI) identified in your transcript. For more information, see
         *        <a href="https://docs.aws.amazon.com/transcribe/latest/dg/phi-id.html">Identifying personal health
         *        information (PHI) in a transcription</a>.
         * @see MedicalContentIdentificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MedicalContentIdentificationType
         */
        Builder contentIdentificationType(MedicalContentIdentificationType contentIdentificationType);

        /**
         * <p>
         * Describes the medical specialty represented in your media.
         * </p>
         * 
         * @param specialty
         *        Describes the medical specialty represented in your media.
         * @see Specialty
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Specialty
         */
        Builder specialty(String specialty);

        /**
         * <p>
         * Describes the medical specialty represented in your media.
         * </p>
         * 
         * @param specialty
         *        Describes the medical specialty represented in your media.
         * @see Specialty
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Specialty
         */
        Builder specialty(Specialty specialty);

        /**
         * <p>
         * Indicates whether the input media is a dictation or a conversation, as specified in the
         * <code>StartMedicalTranscriptionJob</code> request.
         * </p>
         * 
         * @param type
         *        Indicates whether the input media is a dictation or a conversation, as specified in the
         *        <code>StartMedicalTranscriptionJob</code> request.
         * @see Type
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Type
         */
        Builder type(String type);

        /**
         * <p>
         * Indicates whether the input media is a dictation or a conversation, as specified in the
         * <code>StartMedicalTranscriptionJob</code> request.
         * </p>
         * 
         * @param type
         *        Indicates whether the input media is a dictation or a conversation, as specified in the
         *        <code>StartMedicalTranscriptionJob</code> request.
         * @see Type
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Type
         */
        Builder type(Type type);

        /**
         * <p>
         * The tags, each in the form of a key:value pair, assigned to the specified medical transcription job.
         * </p>
         * 
         * @param tags
         *        The tags, each in the form of a key:value pair, assigned to the specified medical transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags, each in the form of a key:value pair, assigned to the specified medical transcription job.
         * </p>
         * 
         * @param tags
         *        The tags, each in the form of a key:value pair, assigned to the specified medical transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags, each in the form of a key:value pair, assigned to the specified medical transcription job.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.transcribe.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.transcribe.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.transcribe.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.transcribe.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String medicalTranscriptionJobName;

        private String transcriptionJobStatus;

        private String languageCode;

        private Integer mediaSampleRateHertz;

        private String mediaFormat;

        private Media media;

        private MedicalTranscript transcript;

        private Instant startTime;

        private Instant creationTime;

        private Instant completionTime;

        private String failureReason;

        private MedicalTranscriptionSetting settings;

        private String contentIdentificationType;

        private String specialty;

        private String type;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MedicalTranscriptionJob model) {
            medicalTranscriptionJobName(model.medicalTranscriptionJobName);
            transcriptionJobStatus(model.transcriptionJobStatus);
            languageCode(model.languageCode);
            mediaSampleRateHertz(model.mediaSampleRateHertz);
            mediaFormat(model.mediaFormat);
            media(model.media);
            transcript(model.transcript);
            startTime(model.startTime);
            creationTime(model.creationTime);
            completionTime(model.completionTime);
            failureReason(model.failureReason);
            settings(model.settings);
            contentIdentificationType(model.contentIdentificationType);
            specialty(model.specialty);
            type(model.type);
            tags(model.tags);
        }

        public final String getMedicalTranscriptionJobName() {
            return medicalTranscriptionJobName;
        }

        public final void setMedicalTranscriptionJobName(String medicalTranscriptionJobName) {
            this.medicalTranscriptionJobName = medicalTranscriptionJobName;
        }

        @Override
        public final Builder medicalTranscriptionJobName(String medicalTranscriptionJobName) {
            this.medicalTranscriptionJobName = medicalTranscriptionJobName;
            return this;
        }

        public final String getTranscriptionJobStatus() {
            return transcriptionJobStatus;
        }

        public final void setTranscriptionJobStatus(String transcriptionJobStatus) {
            this.transcriptionJobStatus = transcriptionJobStatus;
        }

        @Override
        public final Builder transcriptionJobStatus(String transcriptionJobStatus) {
            this.transcriptionJobStatus = transcriptionJobStatus;
            return this;
        }

        @Override
        public final Builder transcriptionJobStatus(TranscriptionJobStatus transcriptionJobStatus) {
            this.transcriptionJobStatus(transcriptionJobStatus == null ? null : transcriptionJobStatus.toString());
            return this;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final Integer getMediaSampleRateHertz() {
            return mediaSampleRateHertz;
        }

        public final void setMediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
        }

        @Override
        public final Builder mediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
            return this;
        }

        public final String getMediaFormat() {
            return mediaFormat;
        }

        public final void setMediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
        }

        @Override
        public final Builder mediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
            return this;
        }

        @Override
        public final Builder mediaFormat(MediaFormat mediaFormat) {
            this.mediaFormat(mediaFormat == null ? null : mediaFormat.toString());
            return this;
        }

        public final Media.Builder getMedia() {
            return media != null ? media.toBuilder() : null;
        }

        public final void setMedia(Media.BuilderImpl media) {
            this.media = media != null ? media.build() : null;
        }

        @Override
        public final Builder media(Media media) {
            this.media = media;
            return this;
        }

        public final MedicalTranscript.Builder getTranscript() {
            return transcript != null ? transcript.toBuilder() : null;
        }

        public final void setTranscript(MedicalTranscript.BuilderImpl transcript) {
            this.transcript = transcript != null ? transcript.build() : null;
        }

        @Override
        public final Builder transcript(MedicalTranscript transcript) {
            this.transcript = transcript;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getCompletionTime() {
            return completionTime;
        }

        public final void setCompletionTime(Instant completionTime) {
            this.completionTime = completionTime;
        }

        @Override
        public final Builder completionTime(Instant completionTime) {
            this.completionTime = completionTime;
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final MedicalTranscriptionSetting.Builder getSettings() {
            return settings != null ? settings.toBuilder() : null;
        }

        public final void setSettings(MedicalTranscriptionSetting.BuilderImpl settings) {
            this.settings = settings != null ? settings.build() : null;
        }

        @Override
        @Transient
        public final Builder settings(MedicalTranscriptionSetting settings) {
            this.settings = settings;
            return this;
        }

        public final String getContentIdentificationType() {
            return contentIdentificationType;
        }

        public final void setContentIdentificationType(String contentIdentificationType) {
            this.contentIdentificationType = contentIdentificationType;
        }

        @Override
        public final Builder contentIdentificationType(String contentIdentificationType) {
            this.contentIdentificationType = contentIdentificationType;
            return this;
        }

        @Override
        public final Builder contentIdentificationType(MedicalContentIdentificationType contentIdentificationType) {
            this.contentIdentificationType(contentIdentificationType == null ? null : contentIdentificationType.toString());
            return this;
        }

        public final String getSpecialty() {
            return specialty;
        }

        public final void setSpecialty(String specialty) {
            this.specialty = specialty;
        }

        @Override
        public final Builder specialty(String specialty) {
            this.specialty = specialty;
            return this;
        }

        @Override
        public final Builder specialty(Specialty specialty) {
            this.specialty(specialty == null ? null : specialty.toString());
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(Type type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public MedicalTranscriptionJob build() {
            return new MedicalTranscriptionJob(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
