/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workspaces.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a WorkSpace.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkspaceProperties implements SdkPojo, Serializable,
        ToCopyableBuilder<WorkspaceProperties.Builder, WorkspaceProperties> {
    private static final SdkField<String> RUNNING_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RunningMode").getter(getter(WorkspaceProperties::runningModeAsString))
            .setter(setter(Builder::runningMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RunningMode").build()).build();

    private static final SdkField<Integer> RUNNING_MODE_AUTO_STOP_TIMEOUT_IN_MINUTES_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("RunningModeAutoStopTimeoutInMinutes")
            .getter(getter(WorkspaceProperties::runningModeAutoStopTimeoutInMinutes))
            .setter(setter(Builder::runningModeAutoStopTimeoutInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("RunningModeAutoStopTimeoutInMinutes").build()).build();

    private static final SdkField<Integer> ROOT_VOLUME_SIZE_GIB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RootVolumeSizeGib").getter(getter(WorkspaceProperties::rootVolumeSizeGib))
            .setter(setter(Builder::rootVolumeSizeGib))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RootVolumeSizeGib").build()).build();

    private static final SdkField<Integer> USER_VOLUME_SIZE_GIB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("UserVolumeSizeGib").getter(getter(WorkspaceProperties::userVolumeSizeGib))
            .setter(setter(Builder::userVolumeSizeGib))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserVolumeSizeGib").build()).build();

    private static final SdkField<String> COMPUTE_TYPE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComputeTypeName").getter(getter(WorkspaceProperties::computeTypeNameAsString))
            .setter(setter(Builder::computeTypeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComputeTypeName").build()).build();

    private static final SdkField<List<String>> PROTOCOLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Protocols")
            .getter(getter(WorkspaceProperties::protocolsAsStrings))
            .setter(setter(Builder::protocolsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocols").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> OPERATING_SYSTEM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperatingSystemName").getter(getter(WorkspaceProperties::operatingSystemNameAsString))
            .setter(setter(Builder::operatingSystemName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingSystemName").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RUNNING_MODE_FIELD,
            RUNNING_MODE_AUTO_STOP_TIMEOUT_IN_MINUTES_FIELD, ROOT_VOLUME_SIZE_GIB_FIELD, USER_VOLUME_SIZE_GIB_FIELD,
            COMPUTE_TYPE_NAME_FIELD, PROTOCOLS_FIELD, OPERATING_SYSTEM_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String runningMode;

    private final Integer runningModeAutoStopTimeoutInMinutes;

    private final Integer rootVolumeSizeGib;

    private final Integer userVolumeSizeGib;

    private final String computeTypeName;

    private final List<String> protocols;

    private final String operatingSystemName;

    private WorkspaceProperties(BuilderImpl builder) {
        this.runningMode = builder.runningMode;
        this.runningModeAutoStopTimeoutInMinutes = builder.runningModeAutoStopTimeoutInMinutes;
        this.rootVolumeSizeGib = builder.rootVolumeSizeGib;
        this.userVolumeSizeGib = builder.userVolumeSizeGib;
        this.computeTypeName = builder.computeTypeName;
        this.protocols = builder.protocols;
        this.operatingSystemName = builder.operatingSystemName;
    }

    /**
     * <p>
     * The running mode. For more information, see <a
     * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/running-mode.html">Manage the WorkSpace Running
     * Mode</a>.
     * </p>
     * <note>
     * <p>
     * The <code>MANUAL</code> value is only supported by Amazon WorkSpaces Core. Contact your account team to be
     * allow-listed to use this value. For more information, see <a href="http://aws.amazon.com/workspaces/core/">Amazon
     * WorkSpaces Core</a>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #runningMode} will
     * return {@link RunningMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #runningModeAsString}.
     * </p>
     * 
     * @return The running mode. For more information, see <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/adminguide/running-mode.html">Manage the WorkSpace
     *         Running Mode</a>.</p> <note>
     *         <p>
     *         The <code>MANUAL</code> value is only supported by Amazon WorkSpaces Core. Contact your account team to
     *         be allow-listed to use this value. For more information, see <a
     *         href="http://aws.amazon.com/workspaces/core/">Amazon WorkSpaces Core</a>.
     *         </p>
     * @see RunningMode
     */
    public final RunningMode runningMode() {
        return RunningMode.fromValue(runningMode);
    }

    /**
     * <p>
     * The running mode. For more information, see <a
     * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/running-mode.html">Manage the WorkSpace Running
     * Mode</a>.
     * </p>
     * <note>
     * <p>
     * The <code>MANUAL</code> value is only supported by Amazon WorkSpaces Core. Contact your account team to be
     * allow-listed to use this value. For more information, see <a href="http://aws.amazon.com/workspaces/core/">Amazon
     * WorkSpaces Core</a>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #runningMode} will
     * return {@link RunningMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #runningModeAsString}.
     * </p>
     * 
     * @return The running mode. For more information, see <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/adminguide/running-mode.html">Manage the WorkSpace
     *         Running Mode</a>.</p> <note>
     *         <p>
     *         The <code>MANUAL</code> value is only supported by Amazon WorkSpaces Core. Contact your account team to
     *         be allow-listed to use this value. For more information, see <a
     *         href="http://aws.amazon.com/workspaces/core/">Amazon WorkSpaces Core</a>.
     *         </p>
     * @see RunningMode
     */
    public final String runningModeAsString() {
        return runningMode;
    }

    /**
     * <p>
     * The time after a user logs off when WorkSpaces are automatically stopped. Configured in 60-minute intervals.
     * </p>
     * 
     * @return The time after a user logs off when WorkSpaces are automatically stopped. Configured in 60-minute
     *         intervals.
     */
    public final Integer runningModeAutoStopTimeoutInMinutes() {
        return runningModeAutoStopTimeoutInMinutes;
    }

    /**
     * <p>
     * The size of the root volume. For important information about how to modify the size of the root and user volumes,
     * see <a href="https://docs.aws.amazon.com/workspaces/latest/adminguide/modify-workspaces.html">Modify a
     * WorkSpace</a>.
     * </p>
     * 
     * @return The size of the root volume. For important information about how to modify the size of the root and user
     *         volumes, see <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/adminguide/modify-workspaces.html">Modify a
     *         WorkSpace</a>.
     */
    public final Integer rootVolumeSizeGib() {
        return rootVolumeSizeGib;
    }

    /**
     * <p>
     * The size of the user storage. For important information about how to modify the size of the root and user
     * volumes, see <a href="https://docs.aws.amazon.com/workspaces/latest/adminguide/modify-workspaces.html">Modify a
     * WorkSpace</a>.
     * </p>
     * 
     * @return The size of the user storage. For important information about how to modify the size of the root and user
     *         volumes, see <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/adminguide/modify-workspaces.html">Modify a
     *         WorkSpace</a>.
     */
    public final Integer userVolumeSizeGib() {
        return userVolumeSizeGib;
    }

    /**
     * <p>
     * The compute type. For more information, see <a
     * href="http://aws.amazon.com/workspaces/details/#Amazon_WorkSpaces_Bundles">Amazon WorkSpaces Bundles</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #computeTypeName}
     * will return {@link Compute#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #computeTypeNameAsString}.
     * </p>
     * 
     * @return The compute type. For more information, see <a
     *         href="http://aws.amazon.com/workspaces/details/#Amazon_WorkSpaces_Bundles">Amazon WorkSpaces Bundles</a>.
     * @see Compute
     */
    public final Compute computeTypeName() {
        return Compute.fromValue(computeTypeName);
    }

    /**
     * <p>
     * The compute type. For more information, see <a
     * href="http://aws.amazon.com/workspaces/details/#Amazon_WorkSpaces_Bundles">Amazon WorkSpaces Bundles</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #computeTypeName}
     * will return {@link Compute#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #computeTypeNameAsString}.
     * </p>
     * 
     * @return The compute type. For more information, see <a
     *         href="http://aws.amazon.com/workspaces/details/#Amazon_WorkSpaces_Bundles">Amazon WorkSpaces Bundles</a>.
     * @see Compute
     */
    public final String computeTypeNameAsString() {
        return computeTypeName;
    }

    /**
     * <p>
     * The protocol. For more information, see <a
     * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html"> Protocols for
     * Amazon WorkSpaces</a>.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * Only available for WorkSpaces created with PCoIP bundles.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or <code>WSP</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
     * Graphics.g4dn, and GraphicsPro.g4dn).
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProtocols} method.
     * </p>
     * 
     * @return The protocol. For more information, see <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html">
     *         Protocols for Amazon WorkSpaces</a>.</p> <note>
     *         <ul>
     *         <li>
     *         <p>
     *         Only available for WorkSpaces created with PCoIP bundles.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or
     *         <code>WSP</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
     *         Graphics.g4dn, and GraphicsPro.g4dn).
     *         </p>
     *         </li>
     *         </ul>
     */
    public final List<Protocol> protocols() {
        return ProtocolListCopier.copyStringToEnum(protocols);
    }

    /**
     * For responses, this returns true if the service returned a value for the Protocols property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProtocols() {
        return protocols != null && !(protocols instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The protocol. For more information, see <a
     * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html"> Protocols for
     * Amazon WorkSpaces</a>.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * Only available for WorkSpaces created with PCoIP bundles.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or <code>WSP</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
     * Graphics.g4dn, and GraphicsPro.g4dn).
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProtocols} method.
     * </p>
     * 
     * @return The protocol. For more information, see <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html">
     *         Protocols for Amazon WorkSpaces</a>.</p> <note>
     *         <ul>
     *         <li>
     *         <p>
     *         Only available for WorkSpaces created with PCoIP bundles.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or
     *         <code>WSP</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
     *         Graphics.g4dn, and GraphicsPro.g4dn).
     *         </p>
     *         </li>
     *         </ul>
     */
    public final List<String> protocolsAsStrings() {
        return protocols;
    }

    /**
     * <p>
     * The name of the operating system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #operatingSystemName} will return {@link OperatingSystemName#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #operatingSystemNameAsString}.
     * </p>
     * 
     * @return The name of the operating system.
     * @see OperatingSystemName
     */
    public final OperatingSystemName operatingSystemName() {
        return OperatingSystemName.fromValue(operatingSystemName);
    }

    /**
     * <p>
     * The name of the operating system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #operatingSystemName} will return {@link OperatingSystemName#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #operatingSystemNameAsString}.
     * </p>
     * 
     * @return The name of the operating system.
     * @see OperatingSystemName
     */
    public final String operatingSystemNameAsString() {
        return operatingSystemName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(runningModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(runningModeAutoStopTimeoutInMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(rootVolumeSizeGib());
        hashCode = 31 * hashCode + Objects.hashCode(userVolumeSizeGib());
        hashCode = 31 * hashCode + Objects.hashCode(computeTypeNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasProtocols() ? protocolsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystemNameAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkspaceProperties)) {
            return false;
        }
        WorkspaceProperties other = (WorkspaceProperties) obj;
        return Objects.equals(runningModeAsString(), other.runningModeAsString())
                && Objects.equals(runningModeAutoStopTimeoutInMinutes(), other.runningModeAutoStopTimeoutInMinutes())
                && Objects.equals(rootVolumeSizeGib(), other.rootVolumeSizeGib())
                && Objects.equals(userVolumeSizeGib(), other.userVolumeSizeGib())
                && Objects.equals(computeTypeNameAsString(), other.computeTypeNameAsString())
                && hasProtocols() == other.hasProtocols() && Objects.equals(protocolsAsStrings(), other.protocolsAsStrings())
                && Objects.equals(operatingSystemNameAsString(), other.operatingSystemNameAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WorkspaceProperties").add("RunningMode", runningModeAsString())
                .add("RunningModeAutoStopTimeoutInMinutes", runningModeAutoStopTimeoutInMinutes())
                .add("RootVolumeSizeGib", rootVolumeSizeGib()).add("UserVolumeSizeGib", userVolumeSizeGib())
                .add("ComputeTypeName", computeTypeNameAsString()).add("Protocols", hasProtocols() ? protocolsAsStrings() : null)
                .add("OperatingSystemName", operatingSystemNameAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RunningMode":
            return Optional.ofNullable(clazz.cast(runningModeAsString()));
        case "RunningModeAutoStopTimeoutInMinutes":
            return Optional.ofNullable(clazz.cast(runningModeAutoStopTimeoutInMinutes()));
        case "RootVolumeSizeGib":
            return Optional.ofNullable(clazz.cast(rootVolumeSizeGib()));
        case "UserVolumeSizeGib":
            return Optional.ofNullable(clazz.cast(userVolumeSizeGib()));
        case "ComputeTypeName":
            return Optional.ofNullable(clazz.cast(computeTypeNameAsString()));
        case "Protocols":
            return Optional.ofNullable(clazz.cast(protocolsAsStrings()));
        case "OperatingSystemName":
            return Optional.ofNullable(clazz.cast(operatingSystemNameAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WorkspaceProperties, T> g) {
        return obj -> g.apply((WorkspaceProperties) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkspaceProperties> {
        /**
         * <p>
         * The running mode. For more information, see <a
         * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/running-mode.html">Manage the WorkSpace
         * Running Mode</a>.
         * </p>
         * <note>
         * <p>
         * The <code>MANUAL</code> value is only supported by Amazon WorkSpaces Core. Contact your account team to be
         * allow-listed to use this value. For more information, see <a
         * href="http://aws.amazon.com/workspaces/core/">Amazon WorkSpaces Core</a>.
         * </p>
         * </note>
         * 
         * @param runningMode
         *        The running mode. For more information, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/running-mode.html">Manage the WorkSpace
         *        Running Mode</a>.</p> <note>
         *        <p>
         *        The <code>MANUAL</code> value is only supported by Amazon WorkSpaces Core. Contact your account team
         *        to be allow-listed to use this value. For more information, see <a
         *        href="http://aws.amazon.com/workspaces/core/">Amazon WorkSpaces Core</a>.
         *        </p>
         * @see RunningMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RunningMode
         */
        Builder runningMode(String runningMode);

        /**
         * <p>
         * The running mode. For more information, see <a
         * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/running-mode.html">Manage the WorkSpace
         * Running Mode</a>.
         * </p>
         * <note>
         * <p>
         * The <code>MANUAL</code> value is only supported by Amazon WorkSpaces Core. Contact your account team to be
         * allow-listed to use this value. For more information, see <a
         * href="http://aws.amazon.com/workspaces/core/">Amazon WorkSpaces Core</a>.
         * </p>
         * </note>
         * 
         * @param runningMode
         *        The running mode. For more information, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/running-mode.html">Manage the WorkSpace
         *        Running Mode</a>.</p> <note>
         *        <p>
         *        The <code>MANUAL</code> value is only supported by Amazon WorkSpaces Core. Contact your account team
         *        to be allow-listed to use this value. For more information, see <a
         *        href="http://aws.amazon.com/workspaces/core/">Amazon WorkSpaces Core</a>.
         *        </p>
         * @see RunningMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RunningMode
         */
        Builder runningMode(RunningMode runningMode);

        /**
         * <p>
         * The time after a user logs off when WorkSpaces are automatically stopped. Configured in 60-minute intervals.
         * </p>
         * 
         * @param runningModeAutoStopTimeoutInMinutes
         *        The time after a user logs off when WorkSpaces are automatically stopped. Configured in 60-minute
         *        intervals.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runningModeAutoStopTimeoutInMinutes(Integer runningModeAutoStopTimeoutInMinutes);

        /**
         * <p>
         * The size of the root volume. For important information about how to modify the size of the root and user
         * volumes, see <a href="https://docs.aws.amazon.com/workspaces/latest/adminguide/modify-workspaces.html">Modify
         * a WorkSpace</a>.
         * </p>
         * 
         * @param rootVolumeSizeGib
         *        The size of the root volume. For important information about how to modify the size of the root and
         *        user volumes, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/modify-workspaces.html">Modify a
         *        WorkSpace</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootVolumeSizeGib(Integer rootVolumeSizeGib);

        /**
         * <p>
         * The size of the user storage. For important information about how to modify the size of the root and user
         * volumes, see <a href="https://docs.aws.amazon.com/workspaces/latest/adminguide/modify-workspaces.html">Modify
         * a WorkSpace</a>.
         * </p>
         * 
         * @param userVolumeSizeGib
         *        The size of the user storage. For important information about how to modify the size of the root and
         *        user volumes, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/modify-workspaces.html">Modify a
         *        WorkSpace</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userVolumeSizeGib(Integer userVolumeSizeGib);

        /**
         * <p>
         * The compute type. For more information, see <a
         * href="http://aws.amazon.com/workspaces/details/#Amazon_WorkSpaces_Bundles">Amazon WorkSpaces Bundles</a>.
         * </p>
         * 
         * @param computeTypeName
         *        The compute type. For more information, see <a
         *        href="http://aws.amazon.com/workspaces/details/#Amazon_WorkSpaces_Bundles">Amazon WorkSpaces
         *        Bundles</a>.
         * @see Compute
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Compute
         */
        Builder computeTypeName(String computeTypeName);

        /**
         * <p>
         * The compute type. For more information, see <a
         * href="http://aws.amazon.com/workspaces/details/#Amazon_WorkSpaces_Bundles">Amazon WorkSpaces Bundles</a>.
         * </p>
         * 
         * @param computeTypeName
         *        The compute type. For more information, see <a
         *        href="http://aws.amazon.com/workspaces/details/#Amazon_WorkSpaces_Bundles">Amazon WorkSpaces
         *        Bundles</a>.
         * @see Compute
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Compute
         */
        Builder computeTypeName(Compute computeTypeName);

        /**
         * <p>
         * The protocol. For more information, see <a
         * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html"> Protocols
         * for Amazon WorkSpaces</a>.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * Only available for WorkSpaces created with PCoIP bundles.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or <code>WSP</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
         * Graphics.g4dn, and GraphicsPro.g4dn).
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param protocols
         *        The protocol. For more information, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html">
         *        Protocols for Amazon WorkSpaces</a>.</p> <note>
         *        <ul>
         *        <li>
         *        <p>
         *        Only available for WorkSpaces created with PCoIP bundles.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or
         *        <code>WSP</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
         *        Graphics.g4dn, and GraphicsPro.g4dn).
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(Collection<String> protocols);

        /**
         * <p>
         * The protocol. For more information, see <a
         * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html"> Protocols
         * for Amazon WorkSpaces</a>.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * Only available for WorkSpaces created with PCoIP bundles.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or <code>WSP</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
         * Graphics.g4dn, and GraphicsPro.g4dn).
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param protocols
         *        The protocol. For more information, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html">
         *        Protocols for Amazon WorkSpaces</a>.</p> <note>
         *        <ul>
         *        <li>
         *        <p>
         *        Only available for WorkSpaces created with PCoIP bundles.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or
         *        <code>WSP</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
         *        Graphics.g4dn, and GraphicsPro.g4dn).
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(String... protocols);

        /**
         * <p>
         * The protocol. For more information, see <a
         * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html"> Protocols
         * for Amazon WorkSpaces</a>.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * Only available for WorkSpaces created with PCoIP bundles.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or <code>WSP</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
         * Graphics.g4dn, and GraphicsPro.g4dn).
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param protocols
         *        The protocol. For more information, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html">
         *        Protocols for Amazon WorkSpaces</a>.</p> <note>
         *        <ul>
         *        <li>
         *        <p>
         *        Only available for WorkSpaces created with PCoIP bundles.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or
         *        <code>WSP</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
         *        Graphics.g4dn, and GraphicsPro.g4dn).
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Collection<Protocol> protocols);

        /**
         * <p>
         * The protocol. For more information, see <a
         * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html"> Protocols
         * for Amazon WorkSpaces</a>.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * Only available for WorkSpaces created with PCoIP bundles.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or <code>WSP</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
         * Graphics.g4dn, and GraphicsPro.g4dn).
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param protocols
         *        The protocol. For more information, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-protocols.html">
         *        Protocols for Amazon WorkSpaces</a>.</p> <note>
         *        <ul>
         *        <li>
         *        <p>
         *        Only available for WorkSpaces created with PCoIP bundles.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>Protocols</code> property is case sensitive. Ensure you use <code>PCOIP</code> or
         *        <code>WSP</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Unavailable for Windows 7 WorkSpaces and WorkSpaces using GPU-based bundles (Graphics, GraphicsPro,
         *        Graphics.g4dn, and GraphicsPro.g4dn).
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Protocol... protocols);

        /**
         * <p>
         * The name of the operating system.
         * </p>
         * 
         * @param operatingSystemName
         *        The name of the operating system.
         * @see OperatingSystemName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystemName
         */
        Builder operatingSystemName(String operatingSystemName);

        /**
         * <p>
         * The name of the operating system.
         * </p>
         * 
         * @param operatingSystemName
         *        The name of the operating system.
         * @see OperatingSystemName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystemName
         */
        Builder operatingSystemName(OperatingSystemName operatingSystemName);
    }

    static final class BuilderImpl implements Builder {
        private String runningMode;

        private Integer runningModeAutoStopTimeoutInMinutes;

        private Integer rootVolumeSizeGib;

        private Integer userVolumeSizeGib;

        private String computeTypeName;

        private List<String> protocols = DefaultSdkAutoConstructList.getInstance();

        private String operatingSystemName;

        private BuilderImpl() {
        }

        private BuilderImpl(WorkspaceProperties model) {
            runningMode(model.runningMode);
            runningModeAutoStopTimeoutInMinutes(model.runningModeAutoStopTimeoutInMinutes);
            rootVolumeSizeGib(model.rootVolumeSizeGib);
            userVolumeSizeGib(model.userVolumeSizeGib);
            computeTypeName(model.computeTypeName);
            protocolsWithStrings(model.protocols);
            operatingSystemName(model.operatingSystemName);
        }

        public final String getRunningMode() {
            return runningMode;
        }

        public final void setRunningMode(String runningMode) {
            this.runningMode = runningMode;
        }

        @Override
        public final Builder runningMode(String runningMode) {
            this.runningMode = runningMode;
            return this;
        }

        @Override
        public final Builder runningMode(RunningMode runningMode) {
            this.runningMode(runningMode == null ? null : runningMode.toString());
            return this;
        }

        public final Integer getRunningModeAutoStopTimeoutInMinutes() {
            return runningModeAutoStopTimeoutInMinutes;
        }

        public final void setRunningModeAutoStopTimeoutInMinutes(Integer runningModeAutoStopTimeoutInMinutes) {
            this.runningModeAutoStopTimeoutInMinutes = runningModeAutoStopTimeoutInMinutes;
        }

        @Override
        public final Builder runningModeAutoStopTimeoutInMinutes(Integer runningModeAutoStopTimeoutInMinutes) {
            this.runningModeAutoStopTimeoutInMinutes = runningModeAutoStopTimeoutInMinutes;
            return this;
        }

        public final Integer getRootVolumeSizeGib() {
            return rootVolumeSizeGib;
        }

        public final void setRootVolumeSizeGib(Integer rootVolumeSizeGib) {
            this.rootVolumeSizeGib = rootVolumeSizeGib;
        }

        @Override
        public final Builder rootVolumeSizeGib(Integer rootVolumeSizeGib) {
            this.rootVolumeSizeGib = rootVolumeSizeGib;
            return this;
        }

        public final Integer getUserVolumeSizeGib() {
            return userVolumeSizeGib;
        }

        public final void setUserVolumeSizeGib(Integer userVolumeSizeGib) {
            this.userVolumeSizeGib = userVolumeSizeGib;
        }

        @Override
        public final Builder userVolumeSizeGib(Integer userVolumeSizeGib) {
            this.userVolumeSizeGib = userVolumeSizeGib;
            return this;
        }

        public final String getComputeTypeName() {
            return computeTypeName;
        }

        public final void setComputeTypeName(String computeTypeName) {
            this.computeTypeName = computeTypeName;
        }

        @Override
        public final Builder computeTypeName(String computeTypeName) {
            this.computeTypeName = computeTypeName;
            return this;
        }

        @Override
        public final Builder computeTypeName(Compute computeTypeName) {
            this.computeTypeName(computeTypeName == null ? null : computeTypeName.toString());
            return this;
        }

        public final Collection<String> getProtocols() {
            if (protocols instanceof SdkAutoConstructList) {
                return null;
            }
            return protocols;
        }

        public final void setProtocols(Collection<String> protocols) {
            this.protocols = ProtocolListCopier.copy(protocols);
        }

        @Override
        public final Builder protocolsWithStrings(Collection<String> protocols) {
            this.protocols = ProtocolListCopier.copy(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocolsWithStrings(String... protocols) {
            protocolsWithStrings(Arrays.asList(protocols));
            return this;
        }

        @Override
        public final Builder protocols(Collection<Protocol> protocols) {
            this.protocols = ProtocolListCopier.copyEnumToString(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocols(Protocol... protocols) {
            protocols(Arrays.asList(protocols));
            return this;
        }

        public final String getOperatingSystemName() {
            return operatingSystemName;
        }

        public final void setOperatingSystemName(String operatingSystemName) {
            this.operatingSystemName = operatingSystemName;
        }

        @Override
        public final Builder operatingSystemName(String operatingSystemName) {
            this.operatingSystemName = operatingSystemName;
            return this;
        }

        @Override
        public final Builder operatingSystemName(OperatingSystemName operatingSystemName) {
            this.operatingSystemName(operatingSystemName == null ? null : operatingSystemName.toString());
            return this;
        }

        @Override
        public WorkspaceProperties build() {
            return new WorkspaceProperties(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
