/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workspaces.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a WorkSpace.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Workspace implements SdkPojo, Serializable, ToCopyableBuilder<Workspace.Builder, Workspace> {
    private static final SdkField<String> WORKSPACE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkspaceId").getter(getter(Workspace::workspaceId)).setter(setter(Builder::workspaceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkspaceId").build()).build();

    private static final SdkField<String> DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DirectoryId").getter(getter(Workspace::directoryId)).setter(setter(Builder::directoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryId").build()).build();

    private static final SdkField<String> USER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserName").getter(getter(Workspace::userName)).setter(setter(Builder::userName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserName").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddress").getter(getter(Workspace::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddress").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(Workspace::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> BUNDLE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BundleId").getter(getter(Workspace::bundleId)).setter(setter(Builder::bundleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BundleId").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubnetId").getter(getter(Workspace::subnetId)).setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId").build()).build();

    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorMessage").getter(getter(Workspace::errorMessage)).setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorMessage").build()).build();

    private static final SdkField<String> ERROR_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorCode").getter(getter(Workspace::errorCode)).setter(setter(Builder::errorCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorCode").build()).build();

    private static final SdkField<String> COMPUTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComputerName").getter(getter(Workspace::computerName)).setter(setter(Builder::computerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComputerName").build()).build();

    private static final SdkField<String> VOLUME_ENCRYPTION_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VolumeEncryptionKey").getter(getter(Workspace::volumeEncryptionKey))
            .setter(setter(Builder::volumeEncryptionKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeEncryptionKey").build())
            .build();

    private static final SdkField<Boolean> USER_VOLUME_ENCRYPTION_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("UserVolumeEncryptionEnabled")
            .getter(getter(Workspace::userVolumeEncryptionEnabled))
            .setter(setter(Builder::userVolumeEncryptionEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserVolumeEncryptionEnabled")
                    .build()).build();

    private static final SdkField<Boolean> ROOT_VOLUME_ENCRYPTION_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RootVolumeEncryptionEnabled")
            .getter(getter(Workspace::rootVolumeEncryptionEnabled))
            .setter(setter(Builder::rootVolumeEncryptionEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RootVolumeEncryptionEnabled")
                    .build()).build();

    private static final SdkField<WorkspaceProperties> WORKSPACE_PROPERTIES_FIELD = SdkField
            .<WorkspaceProperties> builder(MarshallingType.SDK_POJO).memberName("WorkspaceProperties")
            .getter(getter(Workspace::workspaceProperties)).setter(setter(Builder::workspaceProperties))
            .constructor(WorkspaceProperties::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkspaceProperties").build())
            .build();

    private static final SdkField<List<ModificationState>> MODIFICATION_STATES_FIELD = SdkField
            .<List<ModificationState>> builder(MarshallingType.LIST)
            .memberName("ModificationStates")
            .getter(getter(Workspace::modificationStates))
            .setter(setter(Builder::modificationStates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModificationStates").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ModificationState> builder(MarshallingType.SDK_POJO)
                                            .constructor(ModificationState::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<RelatedWorkspaceProperties>> RELATED_WORKSPACES_FIELD = SdkField
            .<List<RelatedWorkspaceProperties>> builder(MarshallingType.LIST)
            .memberName("RelatedWorkspaces")
            .getter(getter(Workspace::relatedWorkspaces))
            .setter(setter(Builder::relatedWorkspaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedWorkspaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RelatedWorkspaceProperties> builder(MarshallingType.SDK_POJO)
                                            .constructor(RelatedWorkspaceProperties::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DataReplicationSettings> DATA_REPLICATION_SETTINGS_FIELD = SdkField
            .<DataReplicationSettings> builder(MarshallingType.SDK_POJO).memberName("DataReplicationSettings")
            .getter(getter(Workspace::dataReplicationSettings)).setter(setter(Builder::dataReplicationSettings))
            .constructor(DataReplicationSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataReplicationSettings").build())
            .build();

    private static final SdkField<List<StandbyWorkspacesProperties>> STANDBY_WORKSPACES_PROPERTIES_FIELD = SdkField
            .<List<StandbyWorkspacesProperties>> builder(MarshallingType.LIST)
            .memberName("StandbyWorkspacesProperties")
            .getter(getter(Workspace::standbyWorkspacesProperties))
            .setter(setter(Builder::standbyWorkspacesProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandbyWorkspacesProperties")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<StandbyWorkspacesProperties> builder(MarshallingType.SDK_POJO)
                                            .constructor(StandbyWorkspacesProperties::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(WORKSPACE_ID_FIELD,
            DIRECTORY_ID_FIELD, USER_NAME_FIELD, IP_ADDRESS_FIELD, STATE_FIELD, BUNDLE_ID_FIELD, SUBNET_ID_FIELD,
            ERROR_MESSAGE_FIELD, ERROR_CODE_FIELD, COMPUTER_NAME_FIELD, VOLUME_ENCRYPTION_KEY_FIELD,
            USER_VOLUME_ENCRYPTION_ENABLED_FIELD, ROOT_VOLUME_ENCRYPTION_ENABLED_FIELD, WORKSPACE_PROPERTIES_FIELD,
            MODIFICATION_STATES_FIELD, RELATED_WORKSPACES_FIELD, DATA_REPLICATION_SETTINGS_FIELD,
            STANDBY_WORKSPACES_PROPERTIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String workspaceId;

    private final String directoryId;

    private final String userName;

    private final String ipAddress;

    private final String state;

    private final String bundleId;

    private final String subnetId;

    private final String errorMessage;

    private final String errorCode;

    private final String computerName;

    private final String volumeEncryptionKey;

    private final Boolean userVolumeEncryptionEnabled;

    private final Boolean rootVolumeEncryptionEnabled;

    private final WorkspaceProperties workspaceProperties;

    private final List<ModificationState> modificationStates;

    private final List<RelatedWorkspaceProperties> relatedWorkspaces;

    private final DataReplicationSettings dataReplicationSettings;

    private final List<StandbyWorkspacesProperties> standbyWorkspacesProperties;

    private Workspace(BuilderImpl builder) {
        this.workspaceId = builder.workspaceId;
        this.directoryId = builder.directoryId;
        this.userName = builder.userName;
        this.ipAddress = builder.ipAddress;
        this.state = builder.state;
        this.bundleId = builder.bundleId;
        this.subnetId = builder.subnetId;
        this.errorMessage = builder.errorMessage;
        this.errorCode = builder.errorCode;
        this.computerName = builder.computerName;
        this.volumeEncryptionKey = builder.volumeEncryptionKey;
        this.userVolumeEncryptionEnabled = builder.userVolumeEncryptionEnabled;
        this.rootVolumeEncryptionEnabled = builder.rootVolumeEncryptionEnabled;
        this.workspaceProperties = builder.workspaceProperties;
        this.modificationStates = builder.modificationStates;
        this.relatedWorkspaces = builder.relatedWorkspaces;
        this.dataReplicationSettings = builder.dataReplicationSettings;
        this.standbyWorkspacesProperties = builder.standbyWorkspacesProperties;
    }

    /**
     * <p>
     * The identifier of the WorkSpace.
     * </p>
     * 
     * @return The identifier of the WorkSpace.
     */
    public final String workspaceId() {
        return workspaceId;
    }

    /**
     * <p>
     * The identifier of the Directory Service directory for the WorkSpace.
     * </p>
     * 
     * @return The identifier of the Directory Service directory for the WorkSpace.
     */
    public final String directoryId() {
        return directoryId;
    }

    /**
     * <p>
     * The user for the WorkSpace.
     * </p>
     * 
     * @return The user for the WorkSpace.
     */
    public final String userName() {
        return userName;
    }

    /**
     * <p>
     * The IP address of the WorkSpace.
     * </p>
     * 
     * @return The IP address of the WorkSpace.
     */
    public final String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * The operational state of the WorkSpace.
     * </p>
     * <note>
     * <p>
     * After a WorkSpace is terminated, the <code>TERMINATED</code> state is returned only briefly before the WorkSpace
     * directory metadata is cleaned up, so this state is rarely returned. To confirm that a WorkSpace is terminated,
     * check for the WorkSpace ID by using <a
     * href="https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html"> DescribeWorkSpaces</a>. If
     * the WorkSpace ID isn't returned, then the WorkSpace has been successfully terminated.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link WorkspaceState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The operational state of the WorkSpace.</p> <note>
     *         <p>
     *         After a WorkSpace is terminated, the <code>TERMINATED</code> state is returned only briefly before the
     *         WorkSpace directory metadata is cleaned up, so this state is rarely returned. To confirm that a WorkSpace
     *         is terminated, check for the WorkSpace ID by using <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html">
     *         DescribeWorkSpaces</a>. If the WorkSpace ID isn't returned, then the WorkSpace has been successfully
     *         terminated.
     *         </p>
     * @see WorkspaceState
     */
    public final WorkspaceState state() {
        return WorkspaceState.fromValue(state);
    }

    /**
     * <p>
     * The operational state of the WorkSpace.
     * </p>
     * <note>
     * <p>
     * After a WorkSpace is terminated, the <code>TERMINATED</code> state is returned only briefly before the WorkSpace
     * directory metadata is cleaned up, so this state is rarely returned. To confirm that a WorkSpace is terminated,
     * check for the WorkSpace ID by using <a
     * href="https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html"> DescribeWorkSpaces</a>. If
     * the WorkSpace ID isn't returned, then the WorkSpace has been successfully terminated.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link WorkspaceState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The operational state of the WorkSpace.</p> <note>
     *         <p>
     *         After a WorkSpace is terminated, the <code>TERMINATED</code> state is returned only briefly before the
     *         WorkSpace directory metadata is cleaned up, so this state is rarely returned. To confirm that a WorkSpace
     *         is terminated, check for the WorkSpace ID by using <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html">
     *         DescribeWorkSpaces</a>. If the WorkSpace ID isn't returned, then the WorkSpace has been successfully
     *         terminated.
     *         </p>
     * @see WorkspaceState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The identifier of the bundle used to create the WorkSpace.
     * </p>
     * 
     * @return The identifier of the bundle used to create the WorkSpace.
     */
    public final String bundleId() {
        return bundleId;
    }

    /**
     * <p>
     * The identifier of the subnet for the WorkSpace.
     * </p>
     * 
     * @return The identifier of the subnet for the WorkSpace.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The text of the error message that is returned if the WorkSpace cannot be created.
     * </p>
     * 
     * @return The text of the error message that is returned if the WorkSpace cannot be created.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    /**
     * <p>
     * The error code that is returned if the WorkSpace cannot be created.
     * </p>
     * 
     * @return The error code that is returned if the WorkSpace cannot be created.
     */
    public final String errorCode() {
        return errorCode;
    }

    /**
     * <p>
     * The name of the WorkSpace, as seen by the operating system. The format of this name varies. For more information,
     * see <a href="https://docs.aws.amazon.com/workspaces/latest/adminguide/launch-workspaces-tutorials.html"> Launch a
     * WorkSpace</a>.
     * </p>
     * 
     * @return The name of the WorkSpace, as seen by the operating system. The format of this name varies. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/adminguide/launch-workspaces-tutorials.html"> Launch
     *         a WorkSpace</a>.
     */
    public final String computerName() {
        return computerName;
    }

    /**
     * <p>
     * The ARN of the symmetric KMS key used to encrypt data stored on your WorkSpace. Amazon WorkSpaces does not
     * support asymmetric KMS keys.
     * </p>
     * 
     * @return The ARN of the symmetric KMS key used to encrypt data stored on your WorkSpace. Amazon WorkSpaces does
     *         not support asymmetric KMS keys.
     */
    public final String volumeEncryptionKey() {
        return volumeEncryptionKey;
    }

    /**
     * <p>
     * Indicates whether the data stored on the user volume is encrypted.
     * </p>
     * 
     * @return Indicates whether the data stored on the user volume is encrypted.
     */
    public final Boolean userVolumeEncryptionEnabled() {
        return userVolumeEncryptionEnabled;
    }

    /**
     * <p>
     * Indicates whether the data stored on the root volume is encrypted.
     * </p>
     * 
     * @return Indicates whether the data stored on the root volume is encrypted.
     */
    public final Boolean rootVolumeEncryptionEnabled() {
        return rootVolumeEncryptionEnabled;
    }

    /**
     * <p>
     * The properties of the WorkSpace.
     * </p>
     * 
     * @return The properties of the WorkSpace.
     */
    public final WorkspaceProperties workspaceProperties() {
        return workspaceProperties;
    }

    /**
     * For responses, this returns true if the service returned a value for the ModificationStates property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasModificationStates() {
        return modificationStates != null && !(modificationStates instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The modification states of the WorkSpace.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasModificationStates} method.
     * </p>
     * 
     * @return The modification states of the WorkSpace.
     */
    public final List<ModificationState> modificationStates() {
        return modificationStates;
    }

    /**
     * For responses, this returns true if the service returned a value for the RelatedWorkspaces property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRelatedWorkspaces() {
        return relatedWorkspaces != null && !(relatedWorkspaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The standby WorkSpace or primary WorkSpace related to the specified WorkSpace.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelatedWorkspaces} method.
     * </p>
     * 
     * @return The standby WorkSpace or primary WorkSpace related to the specified WorkSpace.
     */
    public final List<RelatedWorkspaceProperties> relatedWorkspaces() {
        return relatedWorkspaces;
    }

    /**
     * <p>
     * Indicates the settings of the data replication.
     * </p>
     * 
     * @return Indicates the settings of the data replication.
     */
    public final DataReplicationSettings dataReplicationSettings() {
        return dataReplicationSettings;
    }

    /**
     * For responses, this returns true if the service returned a value for the StandbyWorkspacesProperties property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasStandbyWorkspacesProperties() {
        return standbyWorkspacesProperties != null && !(standbyWorkspacesProperties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The properties of the standby WorkSpace
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStandbyWorkspacesProperties} method.
     * </p>
     * 
     * @return The properties of the standby WorkSpace
     */
    public final List<StandbyWorkspacesProperties> standbyWorkspacesProperties() {
        return standbyWorkspacesProperties;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(workspaceId());
        hashCode = 31 * hashCode + Objects.hashCode(directoryId());
        hashCode = 31 * hashCode + Objects.hashCode(userName());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(bundleId());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(errorCode());
        hashCode = 31 * hashCode + Objects.hashCode(computerName());
        hashCode = 31 * hashCode + Objects.hashCode(volumeEncryptionKey());
        hashCode = 31 * hashCode + Objects.hashCode(userVolumeEncryptionEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(rootVolumeEncryptionEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(workspaceProperties());
        hashCode = 31 * hashCode + Objects.hashCode(hasModificationStates() ? modificationStates() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRelatedWorkspaces() ? relatedWorkspaces() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dataReplicationSettings());
        hashCode = 31 * hashCode + Objects.hashCode(hasStandbyWorkspacesProperties() ? standbyWorkspacesProperties() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Workspace)) {
            return false;
        }
        Workspace other = (Workspace) obj;
        return Objects.equals(workspaceId(), other.workspaceId()) && Objects.equals(directoryId(), other.directoryId())
                && Objects.equals(userName(), other.userName()) && Objects.equals(ipAddress(), other.ipAddress())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(bundleId(), other.bundleId())
                && Objects.equals(subnetId(), other.subnetId()) && Objects.equals(errorMessage(), other.errorMessage())
                && Objects.equals(errorCode(), other.errorCode()) && Objects.equals(computerName(), other.computerName())
                && Objects.equals(volumeEncryptionKey(), other.volumeEncryptionKey())
                && Objects.equals(userVolumeEncryptionEnabled(), other.userVolumeEncryptionEnabled())
                && Objects.equals(rootVolumeEncryptionEnabled(), other.rootVolumeEncryptionEnabled())
                && Objects.equals(workspaceProperties(), other.workspaceProperties())
                && hasModificationStates() == other.hasModificationStates()
                && Objects.equals(modificationStates(), other.modificationStates())
                && hasRelatedWorkspaces() == other.hasRelatedWorkspaces()
                && Objects.equals(relatedWorkspaces(), other.relatedWorkspaces())
                && Objects.equals(dataReplicationSettings(), other.dataReplicationSettings())
                && hasStandbyWorkspacesProperties() == other.hasStandbyWorkspacesProperties()
                && Objects.equals(standbyWorkspacesProperties(), other.standbyWorkspacesProperties());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Workspace").add("WorkspaceId", workspaceId()).add("DirectoryId", directoryId())
                .add("UserName", userName()).add("IpAddress", ipAddress()).add("State", stateAsString())
                .add("BundleId", bundleId()).add("SubnetId", subnetId()).add("ErrorMessage", errorMessage())
                .add("ErrorCode", errorCode()).add("ComputerName", computerName())
                .add("VolumeEncryptionKey", volumeEncryptionKey())
                .add("UserVolumeEncryptionEnabled", userVolumeEncryptionEnabled())
                .add("RootVolumeEncryptionEnabled", rootVolumeEncryptionEnabled())
                .add("WorkspaceProperties", workspaceProperties())
                .add("ModificationStates", hasModificationStates() ? modificationStates() : null)
                .add("RelatedWorkspaces", hasRelatedWorkspaces() ? relatedWorkspaces() : null)
                .add("DataReplicationSettings", dataReplicationSettings())
                .add("StandbyWorkspacesProperties", hasStandbyWorkspacesProperties() ? standbyWorkspacesProperties() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WorkspaceId":
            return Optional.ofNullable(clazz.cast(workspaceId()));
        case "DirectoryId":
            return Optional.ofNullable(clazz.cast(directoryId()));
        case "UserName":
            return Optional.ofNullable(clazz.cast(userName()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "BundleId":
            return Optional.ofNullable(clazz.cast(bundleId()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "ErrorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        case "ErrorCode":
            return Optional.ofNullable(clazz.cast(errorCode()));
        case "ComputerName":
            return Optional.ofNullable(clazz.cast(computerName()));
        case "VolumeEncryptionKey":
            return Optional.ofNullable(clazz.cast(volumeEncryptionKey()));
        case "UserVolumeEncryptionEnabled":
            return Optional.ofNullable(clazz.cast(userVolumeEncryptionEnabled()));
        case "RootVolumeEncryptionEnabled":
            return Optional.ofNullable(clazz.cast(rootVolumeEncryptionEnabled()));
        case "WorkspaceProperties":
            return Optional.ofNullable(clazz.cast(workspaceProperties()));
        case "ModificationStates":
            return Optional.ofNullable(clazz.cast(modificationStates()));
        case "RelatedWorkspaces":
            return Optional.ofNullable(clazz.cast(relatedWorkspaces()));
        case "DataReplicationSettings":
            return Optional.ofNullable(clazz.cast(dataReplicationSettings()));
        case "StandbyWorkspacesProperties":
            return Optional.ofNullable(clazz.cast(standbyWorkspacesProperties()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Workspace, T> g) {
        return obj -> g.apply((Workspace) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Workspace> {
        /**
         * <p>
         * The identifier of the WorkSpace.
         * </p>
         * 
         * @param workspaceId
         *        The identifier of the WorkSpace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workspaceId(String workspaceId);

        /**
         * <p>
         * The identifier of the Directory Service directory for the WorkSpace.
         * </p>
         * 
         * @param directoryId
         *        The identifier of the Directory Service directory for the WorkSpace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryId(String directoryId);

        /**
         * <p>
         * The user for the WorkSpace.
         * </p>
         * 
         * @param userName
         *        The user for the WorkSpace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userName(String userName);

        /**
         * <p>
         * The IP address of the WorkSpace.
         * </p>
         * 
         * @param ipAddress
         *        The IP address of the WorkSpace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * The operational state of the WorkSpace.
         * </p>
         * <note>
         * <p>
         * After a WorkSpace is terminated, the <code>TERMINATED</code> state is returned only briefly before the
         * WorkSpace directory metadata is cleaned up, so this state is rarely returned. To confirm that a WorkSpace is
         * terminated, check for the WorkSpace ID by using <a
         * href="https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html"> DescribeWorkSpaces</a>.
         * If the WorkSpace ID isn't returned, then the WorkSpace has been successfully terminated.
         * </p>
         * </note>
         * 
         * @param state
         *        The operational state of the WorkSpace.</p> <note>
         *        <p>
         *        After a WorkSpace is terminated, the <code>TERMINATED</code> state is returned only briefly before the
         *        WorkSpace directory metadata is cleaned up, so this state is rarely returned. To confirm that a
         *        WorkSpace is terminated, check for the WorkSpace ID by using <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html">
         *        DescribeWorkSpaces</a>. If the WorkSpace ID isn't returned, then the WorkSpace has been successfully
         *        terminated.
         *        </p>
         * @see WorkspaceState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceState
         */
        Builder state(String state);

        /**
         * <p>
         * The operational state of the WorkSpace.
         * </p>
         * <note>
         * <p>
         * After a WorkSpace is terminated, the <code>TERMINATED</code> state is returned only briefly before the
         * WorkSpace directory metadata is cleaned up, so this state is rarely returned. To confirm that a WorkSpace is
         * terminated, check for the WorkSpace ID by using <a
         * href="https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html"> DescribeWorkSpaces</a>.
         * If the WorkSpace ID isn't returned, then the WorkSpace has been successfully terminated.
         * </p>
         * </note>
         * 
         * @param state
         *        The operational state of the WorkSpace.</p> <note>
         *        <p>
         *        After a WorkSpace is terminated, the <code>TERMINATED</code> state is returned only briefly before the
         *        WorkSpace directory metadata is cleaned up, so this state is rarely returned. To confirm that a
         *        WorkSpace is terminated, check for the WorkSpace ID by using <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html">
         *        DescribeWorkSpaces</a>. If the WorkSpace ID isn't returned, then the WorkSpace has been successfully
         *        terminated.
         *        </p>
         * @see WorkspaceState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceState
         */
        Builder state(WorkspaceState state);

        /**
         * <p>
         * The identifier of the bundle used to create the WorkSpace.
         * </p>
         * 
         * @param bundleId
         *        The identifier of the bundle used to create the WorkSpace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bundleId(String bundleId);

        /**
         * <p>
         * The identifier of the subnet for the WorkSpace.
         * </p>
         * 
         * @param subnetId
         *        The identifier of the subnet for the WorkSpace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The text of the error message that is returned if the WorkSpace cannot be created.
         * </p>
         * 
         * @param errorMessage
         *        The text of the error message that is returned if the WorkSpace cannot be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);

        /**
         * <p>
         * The error code that is returned if the WorkSpace cannot be created.
         * </p>
         * 
         * @param errorCode
         *        The error code that is returned if the WorkSpace cannot be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorCode(String errorCode);

        /**
         * <p>
         * The name of the WorkSpace, as seen by the operating system. The format of this name varies. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/launch-workspaces-tutorials.html"> Launch a
         * WorkSpace</a>.
         * </p>
         * 
         * @param computerName
         *        The name of the WorkSpace, as seen by the operating system. The format of this name varies. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/launch-workspaces-tutorials.html">
         *        Launch a WorkSpace</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computerName(String computerName);

        /**
         * <p>
         * The ARN of the symmetric KMS key used to encrypt data stored on your WorkSpace. Amazon WorkSpaces does not
         * support asymmetric KMS keys.
         * </p>
         * 
         * @param volumeEncryptionKey
         *        The ARN of the symmetric KMS key used to encrypt data stored on your WorkSpace. Amazon WorkSpaces does
         *        not support asymmetric KMS keys.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeEncryptionKey(String volumeEncryptionKey);

        /**
         * <p>
         * Indicates whether the data stored on the user volume is encrypted.
         * </p>
         * 
         * @param userVolumeEncryptionEnabled
         *        Indicates whether the data stored on the user volume is encrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userVolumeEncryptionEnabled(Boolean userVolumeEncryptionEnabled);

        /**
         * <p>
         * Indicates whether the data stored on the root volume is encrypted.
         * </p>
         * 
         * @param rootVolumeEncryptionEnabled
         *        Indicates whether the data stored on the root volume is encrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootVolumeEncryptionEnabled(Boolean rootVolumeEncryptionEnabled);

        /**
         * <p>
         * The properties of the WorkSpace.
         * </p>
         * 
         * @param workspaceProperties
         *        The properties of the WorkSpace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workspaceProperties(WorkspaceProperties workspaceProperties);

        /**
         * <p>
         * The properties of the WorkSpace.
         * </p>
         * This is a convenience method that creates an instance of the {@link WorkspaceProperties.Builder} avoiding the
         * need to create one manually via {@link WorkspaceProperties#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WorkspaceProperties.Builder#build()} is called immediately and
         * its result is passed to {@link #workspaceProperties(WorkspaceProperties)}.
         * 
         * @param workspaceProperties
         *        a consumer that will call methods on {@link WorkspaceProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #workspaceProperties(WorkspaceProperties)
         */
        default Builder workspaceProperties(Consumer<WorkspaceProperties.Builder> workspaceProperties) {
            return workspaceProperties(WorkspaceProperties.builder().applyMutation(workspaceProperties).build());
        }

        /**
         * <p>
         * The modification states of the WorkSpace.
         * </p>
         * 
         * @param modificationStates
         *        The modification states of the WorkSpace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modificationStates(Collection<ModificationState> modificationStates);

        /**
         * <p>
         * The modification states of the WorkSpace.
         * </p>
         * 
         * @param modificationStates
         *        The modification states of the WorkSpace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modificationStates(ModificationState... modificationStates);

        /**
         * <p>
         * The modification states of the WorkSpace.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.workspaces.model.ModificationState.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.workspaces.model.ModificationState#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.workspaces.model.ModificationState.Builder#build()} is called
         * immediately and its result is passed to {@link #modificationStates(List<ModificationState>)}.
         * 
         * @param modificationStates
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.workspaces.model.ModificationState.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modificationStates(java.util.Collection<ModificationState>)
         */
        Builder modificationStates(Consumer<ModificationState.Builder>... modificationStates);

        /**
         * <p>
         * The standby WorkSpace or primary WorkSpace related to the specified WorkSpace.
         * </p>
         * 
         * @param relatedWorkspaces
         *        The standby WorkSpace or primary WorkSpace related to the specified WorkSpace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedWorkspaces(Collection<RelatedWorkspaceProperties> relatedWorkspaces);

        /**
         * <p>
         * The standby WorkSpace or primary WorkSpace related to the specified WorkSpace.
         * </p>
         * 
         * @param relatedWorkspaces
         *        The standby WorkSpace or primary WorkSpace related to the specified WorkSpace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedWorkspaces(RelatedWorkspaceProperties... relatedWorkspaces);

        /**
         * <p>
         * The standby WorkSpace or primary WorkSpace related to the specified WorkSpace.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.workspaces.model.RelatedWorkspaceProperties.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.workspaces.model.RelatedWorkspaceProperties#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.workspaces.model.RelatedWorkspaceProperties.Builder#build()} is called
         * immediately and its result is passed to {@link #relatedWorkspaces(List<RelatedWorkspaceProperties>)}.
         * 
         * @param relatedWorkspaces
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.workspaces.model.RelatedWorkspaceProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relatedWorkspaces(java.util.Collection<RelatedWorkspaceProperties>)
         */
        Builder relatedWorkspaces(Consumer<RelatedWorkspaceProperties.Builder>... relatedWorkspaces);

        /**
         * <p>
         * Indicates the settings of the data replication.
         * </p>
         * 
         * @param dataReplicationSettings
         *        Indicates the settings of the data replication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataReplicationSettings(DataReplicationSettings dataReplicationSettings);

        /**
         * <p>
         * Indicates the settings of the data replication.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataReplicationSettings.Builder} avoiding
         * the need to create one manually via {@link DataReplicationSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataReplicationSettings.Builder#build()} is called immediately
         * and its result is passed to {@link #dataReplicationSettings(DataReplicationSettings)}.
         * 
         * @param dataReplicationSettings
         *        a consumer that will call methods on {@link DataReplicationSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataReplicationSettings(DataReplicationSettings)
         */
        default Builder dataReplicationSettings(Consumer<DataReplicationSettings.Builder> dataReplicationSettings) {
            return dataReplicationSettings(DataReplicationSettings.builder().applyMutation(dataReplicationSettings).build());
        }

        /**
         * <p>
         * The properties of the standby WorkSpace
         * </p>
         * 
         * @param standbyWorkspacesProperties
         *        The properties of the standby WorkSpace
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standbyWorkspacesProperties(Collection<StandbyWorkspacesProperties> standbyWorkspacesProperties);

        /**
         * <p>
         * The properties of the standby WorkSpace
         * </p>
         * 
         * @param standbyWorkspacesProperties
         *        The properties of the standby WorkSpace
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standbyWorkspacesProperties(StandbyWorkspacesProperties... standbyWorkspacesProperties);

        /**
         * <p>
         * The properties of the standby WorkSpace
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.workspaces.model.StandbyWorkspacesProperties.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.workspaces.model.StandbyWorkspacesProperties#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.workspaces.model.StandbyWorkspacesProperties.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #standbyWorkspacesProperties(List<StandbyWorkspacesProperties>)}.
         * 
         * @param standbyWorkspacesProperties
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.workspaces.model.StandbyWorkspacesProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #standbyWorkspacesProperties(java.util.Collection<StandbyWorkspacesProperties>)
         */
        Builder standbyWorkspacesProperties(Consumer<StandbyWorkspacesProperties.Builder>... standbyWorkspacesProperties);
    }

    static final class BuilderImpl implements Builder {
        private String workspaceId;

        private String directoryId;

        private String userName;

        private String ipAddress;

        private String state;

        private String bundleId;

        private String subnetId;

        private String errorMessage;

        private String errorCode;

        private String computerName;

        private String volumeEncryptionKey;

        private Boolean userVolumeEncryptionEnabled;

        private Boolean rootVolumeEncryptionEnabled;

        private WorkspaceProperties workspaceProperties;

        private List<ModificationState> modificationStates = DefaultSdkAutoConstructList.getInstance();

        private List<RelatedWorkspaceProperties> relatedWorkspaces = DefaultSdkAutoConstructList.getInstance();

        private DataReplicationSettings dataReplicationSettings;

        private List<StandbyWorkspacesProperties> standbyWorkspacesProperties = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Workspace model) {
            workspaceId(model.workspaceId);
            directoryId(model.directoryId);
            userName(model.userName);
            ipAddress(model.ipAddress);
            state(model.state);
            bundleId(model.bundleId);
            subnetId(model.subnetId);
            errorMessage(model.errorMessage);
            errorCode(model.errorCode);
            computerName(model.computerName);
            volumeEncryptionKey(model.volumeEncryptionKey);
            userVolumeEncryptionEnabled(model.userVolumeEncryptionEnabled);
            rootVolumeEncryptionEnabled(model.rootVolumeEncryptionEnabled);
            workspaceProperties(model.workspaceProperties);
            modificationStates(model.modificationStates);
            relatedWorkspaces(model.relatedWorkspaces);
            dataReplicationSettings(model.dataReplicationSettings);
            standbyWorkspacesProperties(model.standbyWorkspacesProperties);
        }

        public final String getWorkspaceId() {
            return workspaceId;
        }

        public final void setWorkspaceId(String workspaceId) {
            this.workspaceId = workspaceId;
        }

        @Override
        public final Builder workspaceId(String workspaceId) {
            this.workspaceId = workspaceId;
            return this;
        }

        public final String getDirectoryId() {
            return directoryId;
        }

        public final void setDirectoryId(String directoryId) {
            this.directoryId = directoryId;
        }

        @Override
        public final Builder directoryId(String directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public final String getUserName() {
            return userName;
        }

        public final void setUserName(String userName) {
            this.userName = userName;
        }

        @Override
        public final Builder userName(String userName) {
            this.userName = userName;
            return this;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(WorkspaceState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getBundleId() {
            return bundleId;
        }

        public final void setBundleId(String bundleId) {
            this.bundleId = bundleId;
        }

        @Override
        public final Builder bundleId(String bundleId) {
            this.bundleId = bundleId;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public final String getErrorCode() {
            return errorCode;
        }

        public final void setErrorCode(String errorCode) {
            this.errorCode = errorCode;
        }

        @Override
        public final Builder errorCode(String errorCode) {
            this.errorCode = errorCode;
            return this;
        }

        public final String getComputerName() {
            return computerName;
        }

        public final void setComputerName(String computerName) {
            this.computerName = computerName;
        }

        @Override
        public final Builder computerName(String computerName) {
            this.computerName = computerName;
            return this;
        }

        public final String getVolumeEncryptionKey() {
            return volumeEncryptionKey;
        }

        public final void setVolumeEncryptionKey(String volumeEncryptionKey) {
            this.volumeEncryptionKey = volumeEncryptionKey;
        }

        @Override
        public final Builder volumeEncryptionKey(String volumeEncryptionKey) {
            this.volumeEncryptionKey = volumeEncryptionKey;
            return this;
        }

        public final Boolean getUserVolumeEncryptionEnabled() {
            return userVolumeEncryptionEnabled;
        }

        public final void setUserVolumeEncryptionEnabled(Boolean userVolumeEncryptionEnabled) {
            this.userVolumeEncryptionEnabled = userVolumeEncryptionEnabled;
        }

        @Override
        public final Builder userVolumeEncryptionEnabled(Boolean userVolumeEncryptionEnabled) {
            this.userVolumeEncryptionEnabled = userVolumeEncryptionEnabled;
            return this;
        }

        public final Boolean getRootVolumeEncryptionEnabled() {
            return rootVolumeEncryptionEnabled;
        }

        public final void setRootVolumeEncryptionEnabled(Boolean rootVolumeEncryptionEnabled) {
            this.rootVolumeEncryptionEnabled = rootVolumeEncryptionEnabled;
        }

        @Override
        public final Builder rootVolumeEncryptionEnabled(Boolean rootVolumeEncryptionEnabled) {
            this.rootVolumeEncryptionEnabled = rootVolumeEncryptionEnabled;
            return this;
        }

        public final WorkspaceProperties.Builder getWorkspaceProperties() {
            return workspaceProperties != null ? workspaceProperties.toBuilder() : null;
        }

        public final void setWorkspaceProperties(WorkspaceProperties.BuilderImpl workspaceProperties) {
            this.workspaceProperties = workspaceProperties != null ? workspaceProperties.build() : null;
        }

        @Override
        public final Builder workspaceProperties(WorkspaceProperties workspaceProperties) {
            this.workspaceProperties = workspaceProperties;
            return this;
        }

        public final List<ModificationState.Builder> getModificationStates() {
            List<ModificationState.Builder> result = ModificationStateListCopier.copyToBuilder(this.modificationStates);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setModificationStates(Collection<ModificationState.BuilderImpl> modificationStates) {
            this.modificationStates = ModificationStateListCopier.copyFromBuilder(modificationStates);
        }

        @Override
        public final Builder modificationStates(Collection<ModificationState> modificationStates) {
            this.modificationStates = ModificationStateListCopier.copy(modificationStates);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modificationStates(ModificationState... modificationStates) {
            modificationStates(Arrays.asList(modificationStates));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modificationStates(Consumer<ModificationState.Builder>... modificationStates) {
            modificationStates(Stream.of(modificationStates).map(c -> ModificationState.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<RelatedWorkspaceProperties.Builder> getRelatedWorkspaces() {
            List<RelatedWorkspaceProperties.Builder> result = RelatedWorkspacesCopier.copyToBuilder(this.relatedWorkspaces);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRelatedWorkspaces(Collection<RelatedWorkspaceProperties.BuilderImpl> relatedWorkspaces) {
            this.relatedWorkspaces = RelatedWorkspacesCopier.copyFromBuilder(relatedWorkspaces);
        }

        @Override
        public final Builder relatedWorkspaces(Collection<RelatedWorkspaceProperties> relatedWorkspaces) {
            this.relatedWorkspaces = RelatedWorkspacesCopier.copy(relatedWorkspaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedWorkspaces(RelatedWorkspaceProperties... relatedWorkspaces) {
            relatedWorkspaces(Arrays.asList(relatedWorkspaces));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedWorkspaces(Consumer<RelatedWorkspaceProperties.Builder>... relatedWorkspaces) {
            relatedWorkspaces(Stream.of(relatedWorkspaces)
                    .map(c -> RelatedWorkspaceProperties.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final DataReplicationSettings.Builder getDataReplicationSettings() {
            return dataReplicationSettings != null ? dataReplicationSettings.toBuilder() : null;
        }

        public final void setDataReplicationSettings(DataReplicationSettings.BuilderImpl dataReplicationSettings) {
            this.dataReplicationSettings = dataReplicationSettings != null ? dataReplicationSettings.build() : null;
        }

        @Override
        public final Builder dataReplicationSettings(DataReplicationSettings dataReplicationSettings) {
            this.dataReplicationSettings = dataReplicationSettings;
            return this;
        }

        public final List<StandbyWorkspacesProperties.Builder> getStandbyWorkspacesProperties() {
            List<StandbyWorkspacesProperties.Builder> result = StandbyWorkspacesPropertiesListCopier
                    .copyToBuilder(this.standbyWorkspacesProperties);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setStandbyWorkspacesProperties(
                Collection<StandbyWorkspacesProperties.BuilderImpl> standbyWorkspacesProperties) {
            this.standbyWorkspacesProperties = StandbyWorkspacesPropertiesListCopier.copyFromBuilder(standbyWorkspacesProperties);
        }

        @Override
        public final Builder standbyWorkspacesProperties(Collection<StandbyWorkspacesProperties> standbyWorkspacesProperties) {
            this.standbyWorkspacesProperties = StandbyWorkspacesPropertiesListCopier.copy(standbyWorkspacesProperties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder standbyWorkspacesProperties(StandbyWorkspacesProperties... standbyWorkspacesProperties) {
            standbyWorkspacesProperties(Arrays.asList(standbyWorkspacesProperties));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder standbyWorkspacesProperties(
                Consumer<StandbyWorkspacesProperties.Builder>... standbyWorkspacesProperties) {
            standbyWorkspacesProperties(Stream.of(standbyWorkspacesProperties)
                    .map(c -> StandbyWorkspacesProperties.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Workspace build() {
            return new Workspace(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
