/*
 * Decompiled with CFR 0.152.
 */
package tech.gtt.common.ipdb.service;

import com.github.davidmoten.geo.GeoHash;
import com.google.common.base.Strings;
import com.maxmind.db.CHMCache;
import com.maxmind.db.NodeCache;
import com.maxmind.geoip2.DatabaseReader;
import com.maxmind.geoip2.exception.AddressNotFoundException;
import com.maxmind.geoip2.model.AsnResponse;
import com.maxmind.geoip2.model.CityResponse;
import com.maxmind.geoip2.record.City;
import com.maxmind.geoip2.record.Country;
import com.maxmind.geoip2.record.Location;
import com.maxmind.geoip2.record.Postal;
import com.maxmind.geoip2.record.Subdivision;
import jakarta.annotation.PostConstruct;
import java.io.File;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.util.StopWatch;
import tech.gtt.common.config.config.ConfigTools3;
import tech.gtt.common.ipdb.api.geoip.EGeoIPDBType;
import tech.gtt.common.ipdb.httptool.Validator;
import tech.gtt.common.ipdb.model.IPASNInfo;
import tech.gtt.common.ipdb.model.IPInfo;
import tech.gtt.common.ipdb.service.InputIPBO;
import tech.gtt.common.ipdb.service.OutputIPBO;

@Component
public class IpTool {
    private static final Logger logger = LoggerFactory.getLogger(IpTool.class);
    private static ConcurrentHashMap<EGeoIPDBType, DatabaseReader> dbReaderMap = new ConcurrentHashMap();

    @PostConstruct
    public void init() {
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        IpTool.loadFromDefaultPath();
        stopWatch.stop();
        logger.info("init load ip db success, cost:{}ms", (Object)stopWatch.getTotalTimeMillis());
    }

    @Scheduled(fixedRate=300000L)
    public void reload() {
        if (ConfigTools3.getBoolean("gtt.ip.db.reload.switch", false)) {
            StopWatch stopWatch = new StopWatch();
            stopWatch.start();
            IpTool.loadFromDefaultPath();
            stopWatch.stop();
            logger.info("reload ip db success, cost:{}ms", (Object)stopWatch.getTotalTimeMillis());
        }
    }

    public static String getDefaultPath() {
        String osName = System.getProperty("os.name").toLowerCase();
        if (osName.contains("window")) {
            return "c:/gtt/ipdb";
        }
        return "/gtt/ipdb";
    }

    public static String getOrDefaultDBPath(EGeoIPDBType types, String dbPath) {
        if (Strings.isNullOrEmpty((String)dbPath)) {
            dbPath = String.format("%s/%s", IpTool.getDefaultPath(), types.getName());
            return dbPath;
        }
        return String.format("%s/%s", dbPath, types.getName());
    }

    public static void loadDB(EGeoIPDBType types, String dbName) {
        try {
            DatabaseReader reader = dbReaderMap.get((Object)types);
            if (reader != null) {
                reader.close();
            }
        }
        catch (Exception e) {
            logger.info("Reader close exeption:{}", (Throwable)e);
        }
        try {
            dbName = IpTool.getOrDefaultDBPath(types, dbName);
            File dbFile = new File(dbName);
            if (!dbFile.exists()) {
                logger.info("DB[{}] not exist", (Object)dbFile);
            } else {
                DatabaseReader reader = new DatabaseReader.Builder(dbFile).withCache((NodeCache)new CHMCache()).build();
                dbReaderMap.put(types, reader);
                logger.info("DB[{}] is loaded", (Object)dbFile);
            }
        }
        catch (Exception e) {
            logger.error("DB[{}] load failed! ex[{}]", (Object)dbName, (Object)e);
        }
    }

    public static void loadFromDefaultPath() {
        for (EGeoIPDBType types : EGeoIPDBType.values()) {
            String ipPath = ConfigTools3.getString("gtt.ip.db.path", "/gtt/ipdb");
            IpTool.loadDB(types, ipPath);
        }
    }

    public static IPInfo getCityInfo(String ip) {
        try {
            Postal postal;
            Location location;
            Country country;
            DatabaseReader reader = dbReaderMap.get((Object)EGeoIPDBType.GEOLITE2_CITY);
            if (reader == null) {
                return null;
            }
            InetAddress ipAddress = InetAddress.getByName(ip);
            CityResponse response = reader.city(ipAddress);
            if (response == null) {
                return null;
            }
            IPInfo ipInfo = new IPInfo();
            ipInfo.setIp(ip);
            City city = response.getCity();
            if (city != null) {
                ipInfo.setCity(city.getName());
            }
            if ((country = response.getCountry()) != null) {
                ipInfo.setCountry(Strings.nullToEmpty((String)country.getIsoCode()).toLowerCase());
            }
            if ((location = response.getLocation()) != null) {
                ipInfo.setLatitude(location.getLatitude());
                ipInfo.setLongitude(location.getLongitude());
                ipInfo.setLocation(String.format("%s,%s", location.getLatitude(), location.getLongitude()));
                String geoHash = GeoHash.encodeHash((double)location.getLatitude(), (double)location.getLongitude());
                ipInfo.setGeoHash(geoHash);
            }
            if ((postal = response.getPostal()) != null) {
                ipInfo.setPostCode(postal.getCode());
            }
            return ipInfo;
        }
        catch (Exception e) {
            if (e instanceof AddressNotFoundException) {
                logger.error("IP[{}] Not Found", (Object)ip, (Object)e.getMessage());
            } else {
                logger.error("IP[{}] message:[{}]", (Object)ip, (Object)e.getMessage());
            }
            return null;
        }
    }

    public static String getCountry(String ip) {
        IPInfo ipInfo = IpTool.getCityInfo(ip);
        if (ipInfo == null) {
            return "";
        }
        return Strings.nullToEmpty((String)ipInfo.getCountry());
    }

    public static IPASNInfo getASNInfo(String ip) {
        try {
            DatabaseReader reader = dbReaderMap.get((Object)EGeoIPDBType.GEOLITE2_ASN);
            if (reader == null) {
                return null;
            }
            InetAddress ipAddress = InetAddress.getByName(ip);
            AsnResponse response = reader.asn(ipAddress);
            if (response == null) {
                return null;
            }
            IPASNInfo asnInfo = new IPASNInfo();
            asnInfo.setASN(Math.toIntExact(response.getAutonomousSystemNumber()));
            asnInfo.setASO(response.getAutonomousSystemOrganization());
            return asnInfo;
        }
        catch (Exception e) {
            if (e instanceof AddressNotFoundException) {
                logger.error("IP[{}] Not Found", (Object)ip, (Object)e.getMessage());
            } else {
                logger.error("IP[{}] message:[{}]", (Object)ip, (Object)e.getMessage());
            }
            return null;
        }
    }

    public static OutputIPBO getCity(String ip) {
        if (Validator.isNullOrEmpty(ip)) {
            return null;
        }
        DatabaseReader reader = dbReaderMap.get((Object)EGeoIPDBType.GEOLITE2_CITY);
        if (Validator.isNullOrEmpty(reader)) {
            return null;
        }
        try {
            InetAddress ipAddress = InetAddress.getByName(ip);
            CityResponse response = reader.city(ipAddress);
            if (Validator.isNullOrEmpty(response)) {
                return null;
            }
            OutputIPBO outputIPBO = new OutputIPBO();
            outputIPBO.setIp(ip);
            City city = response.getCity();
            outputIPBO.setCityId(0L);
            outputIPBO.setCityName("");
            if (Validator.isNotNullOrEmpty(city)) {
                outputIPBO.setCityId(Validator.isNotNullOrEmpty(city.getGeoNameId()) ? city.getGeoNameId() : 0L);
                outputIPBO.setCityName(Validator.isNotNullOrEmpty(city.getNames().get("en")) ? (String)city.getNames().get("en") : "");
            }
            Country country = response.getCountry();
            outputIPBO.setCountryId(0L);
            outputIPBO.setCountryName("");
            if (Validator.isNotNullOrEmpty(country)) {
                outputIPBO.setCountryName(Validator.isNotNullOrEmpty(country.getNames().get("en")) ? (String)country.getNames().get("en") : "");
                outputIPBO.setCountryId(Validator.isNotNullOrEmpty(country.getGeoNameId()) ? country.getGeoNameId() : 0L);
                outputIPBO.setCountryCode(country.getIsoCode());
            }
            List subdivisions = response.getSubdivisions();
            outputIPBO.setProvinceId(0L);
            outputIPBO.setProvinceName("");
            if (Validator.isNotNullOrEmpty(subdivisions)) {
                Subdivision subdivision = (Subdivision)subdivisions.get(0);
                outputIPBO.setProvinceId(Validator.isNotNullOrEmpty(subdivision.getGeoNameId()) ? subdivision.getGeoNameId() : 0L);
                outputIPBO.setProvinceName(Validator.isNotNullOrEmpty(subdivision.getNames().get("en")) ? (String)subdivision.getNames().get("en") : "");
                outputIPBO.setProvinceCode(subdivision.getIsoCode());
            }
            return outputIPBO;
        }
        catch (Exception e) {
            try {
                if (e instanceof AddressNotFoundException) {
                    logger.error("ip not found. ip:{}, error:{}", (Object)ip, (Object)e.getMessage());
                } else {
                    logger.error("unknown error. error:{}", (Object)e.getMessage());
                }
            }
            catch (Exception e2) {
                logger.error("ip db is error. error:{}", (Object)e2.getMessage());
            }
        }
        return null;
    }

    public static List<OutputIPBO> getCityList(List<InputIPBO> ipList) {
        ArrayList<OutputIPBO> result = new ArrayList<OutputIPBO>();
        if (Validator.isNullOrEmpty(ipList)) {
            return result;
        }
        try {
            DatabaseReader reader = dbReaderMap.get((Object)EGeoIPDBType.GEOLITE2_CITY);
            if (Validator.isNullOrEmpty(reader)) {
                return result;
            }
            for (InputIPBO inputIPBO : ipList) {
                String ip = inputIPBO.getIp();
                try {
                    List subdivisions;
                    Country country;
                    InetAddress ipAddress = InetAddress.getByName(ip);
                    CityResponse response = reader.city(ipAddress);
                    if (Validator.isNullOrEmpty(response)) continue;
                    OutputIPBO outputIPBO = new OutputIPBO();
                    outputIPBO.setIp(ip);
                    outputIPBO.setId(inputIPBO.getId());
                    City city = response.getCity();
                    if (Validator.isNotNullOrEmpty(city)) {
                        outputIPBO.setCityId(city.getGeoNameId());
                        outputIPBO.setCityName((String)city.getNames().get("en"));
                    }
                    if (Validator.isNotNullOrEmpty(country = response.getCountry())) {
                        outputIPBO.setCountryName((String)country.getNames().get("en"));
                        outputIPBO.setCountryId(country.getGeoNameId());
                    }
                    if (Validator.isNotNullOrEmpty(subdivisions = response.getSubdivisions())) {
                        Subdivision subdivision = (Subdivision)subdivisions.get(0);
                        outputIPBO.setProvinceId(subdivision.getGeoNameId());
                        outputIPBO.setProvinceName((String)subdivision.getNames().get("en"));
                    }
                    result.add(outputIPBO);
                }
                catch (Exception e) {
                    if (e instanceof AddressNotFoundException) {
                        logger.error("ip not found. ip:{}, error:{}", (Object)ip, (Object)e.getMessage());
                        continue;
                    }
                    logger.error("unknown error. error:{}", (Object)e.getMessage());
                }
            }
        }
        catch (Exception e) {
            logger.error("ip db is error. error:{}", (Object)e.getMessage());
        }
        return result;
    }
}

