package tv.danmaku.ijk.media.drm.wrapper;

import android.media.MediaCrypto;
import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.util.Log;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import okhttp3.Call;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.logging.HttpLoggingInterceptor;
import tv.danmaku.ijk.media.common.drm.DrmConstant;
import tv.danmaku.ijk.media.common.drm.DrmInitInfo;
import tv.danmaku.ijk.media.common.drm.DrmInitInfoParser;
import tv.danmaku.ijk.media.common.drm.DrmManager;
import tv.danmaku.ijk.media.common.drm.OnDrmErrorListener;
import tv.danmaku.ijk.media.drm.BuildConfig;
import tv.danmaku.ijk.media.drm.DefaultDrmSessionManager;
import tv.danmaku.ijk.media.drm.DrmInitData;
import tv.danmaku.ijk.media.drm.FrameworkMediaDrm;
import tv.danmaku.ijk.media.drm.HttpMediaDrmCallback;
import tv.danmaku.ijk.media.drm.MediaDrmCallbackException;
import tv.danmaku.ijk.media.drm.upstream.DefaultHttpDataSource;
import tv.danmaku.ijk.media.drm.upstream.HttpDataSource;
import tv.danmaku.ijk.media.drm.upstream.OkHttpDataSource;
import tv.danmaku.ijk.media.drm.util.C;

import static tv.danmaku.ijk.media.common.drm.DrmConstant.ACQUIRE_SESSION_FLAG_INIT;
import static tv.danmaku.ijk.media.common.drm.DrmConstant.ACQUIRE_SESSION_FLAG_SEEK;
import static tv.danmaku.ijk.media.common.drm.DrmConstant.AUDIO_INDEX;
import static tv.danmaku.ijk.media.common.drm.DrmConstant.VIDEO_INDEX;
import static tv.danmaku.ijk.media.drm.util.Assertions.checkNotNull;

@RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
public class DefaultDrmManager implements DrmManager {
    private static final String TAG = "DefaultDrmManager";

    public static final int HTTP_LOG_LEVEL_NONE = 0;
    public static final int HTTP_LOG_LEVEL_BASIC = 1;
    public static final int HTTP_LOG_LEVEL_HEADERS = 2;
    public static final int HTTP_LOG_LEVEL_BODY = 3;

    private static final int MSG_DO_SOME_WORK = 101;

    private static final int MEDIA_DRM_ERROR_PREFIX = 10000;
    private static final int HTTP_DRM_ERROR_PREFIX = 20000;
    private static final int UNKNOWN_DRM_ERROR_PREFIX = 90000;

    private final String drmLicenseUrl;
    private final Map<String, String> httpRequestHeaders;
    private final UUID uuid;
    private final boolean multiSession;
    private final boolean useOkhttp;
    private final int httpLogLevel;
    private OnDrmErrorListener errorListener;
    private byte[] offlineLicenseKeySetId = null;
    private byte[] offlineLicensePssh = null;

    private HandlerThread handlerThread = null;
    private Handler workHandler = null;
    private DefaultDrmSessionManager drmSessionManager = null;

    private final DrmSessionHolder[] drmSessionHolders = new DrmSessionHolder[2];

    private Exception doSomeWorkError = null;
    private boolean doSomeWorkLoopQuit = false;

    private final Object getCryptoLock = new Object();
    private volatile boolean isCryptoLock;

    public DefaultDrmManager(String drmLicenceUrl, Map<String, String> httpRequestHeaders, UUID uuid, boolean multiSession) {
        this(drmLicenceUrl, httpRequestHeaders, uuid, multiSession, false);
    }

    public DefaultDrmManager(String drmLicenceUrl, Map<String, String> httpRequestHeaders, UUID uuid, boolean multiSession, boolean useOkhttp) {
        this(drmLicenceUrl, httpRequestHeaders, uuid, multiSession, useOkhttp, HTTP_LOG_LEVEL_BASIC);
    }

    public DefaultDrmManager(String drmLicenceUrl, Map<String, String> httpRequestHeaders, UUID uuid, boolean multiSession, boolean useOkhttp, int level) {
        this.drmLicenseUrl = drmLicenceUrl;
        this.httpRequestHeaders = new HashMap<>();
        if (httpRequestHeaders != null) {
            this.httpRequestHeaders.putAll(httpRequestHeaders);
        }
        this.uuid = uuid;
        this.multiSession = multiSession;
        this.useOkhttp = useOkhttp;
        this.httpLogLevel = level;
        Log.i(TAG, "external-ijkplayer version is " + BuildConfig.VERSION_CODE + "_" + BuildConfig.GIT_HASH);
    }

    @Override
    public synchronized void prepare() {
        if (handlerThread == null) {
            handlerThread = new HandlerThread(DefaultDrmManager.class.getSimpleName());
            handlerThread.start();
            workHandler = new Handler(handlerThread.getLooper()) {
                @Override
                public void handleMessage(@NonNull Message msg) {
                    if (msg.what == MSG_DO_SOME_WORK) {
                        if (doSomeWorkError != null) {
                            doSomeWorkError.printStackTrace();
                            return;
                        }
                        try {
                            doSomeWork();
                        } catch (Exception e) {
                            Log.e(TAG, "doSomeWork error : " + e);
                            doSomeWorkError = e;
                            doSomeWorkError.printStackTrace();
                            doSomeWorkLoopQuit = true;
                            workHandler.removeMessages(MSG_DO_SOME_WORK);
                            notifyError(e);
                        }
                    }
                }
            };
            boolean needPlaceholderSession = false;
            if (drmSessionManager == null) {
                drmSessionManager = createManager(drmLicenseUrl, httpRequestHeaders, uuid, multiSession, offlineLicenseKeySetId, useOkhttp, httpLogLevel);
                drmSessionHolders[AUDIO_INDEX] = new DrmSessionHolder(drmSessionManager, workHandler.getLooper(), "audio/mp4", multiSession);
                drmSessionHolders[VIDEO_INDEX] = new DrmSessionHolder(drmSessionManager, workHandler.getLooper(), "video/mp4", multiSession);
                // {@link MediaCrypto#setMediaDrmSession(byte[])} min api level is 24
                needPlaceholderSession = Build.VERSION.SDK_INT > Build.VERSION_CODES.M;
            }
            workHandler.post(new Runnable() {
                @Override
                public void run() {
                    drmSessionManager.prepare();
                    Log.i("IJKMEDIA", "[DRMSTARTUP] prepare drm session manager");
                }
            });
            if (needPlaceholderSession && offlineLicenseKeySetId == null) {
                drmSessionHolders[AUDIO_INDEX].feedDrmInitData(null);
                drmSessionHolders[VIDEO_INDEX].feedDrmInitData(null);
                workHandler.sendEmptyMessage(MSG_DO_SOME_WORK);
            }
        }
    }

    @Override
    public synchronized void release() {
        if (drmSessionManager != null) {
            if (isCryptoLock) {
                Log.w(TAG, "force unlock get crypto method");
                synchronized (getCryptoLock) {
                    getCryptoLock.notifyAll();
                }
            }
            checkNotNull(workHandler).post(new Runnable() {
                @Override
                public void run() {
                    drmSessionManager.release();
                    drmSessionManager = null;
                    doSomeWorkLoopQuit = true;
                    handlerThread.quitSafely();
                    Log.i("IJKMEDIA", "[DRMSTARTUP] release drm session manager");
                }
            });
        }
    }

    @Override
    public synchronized DrmConstant.DrmSessionState acquireSession(final DrmInitInfo drmInitInfo, final int flag) {
        if (drmSessionManager == null) {
            Log.e(TAG, "acquireSession error: drm session manager is released");
            return DrmConstant.DrmSessionState.STATE_UNKNOWN;
        }
        @NonNull final String schemeType = drmInitInfo.schemeType;
        final UUID uuid = drmInitInfo.uuid;
        @Nullable final String sampleMimeType = drmInitInfo.sampleMimeType;
        final byte[] psshData = drmInitInfo.psshData;

        final boolean isSeek = (flag & ACQUIRE_SESSION_FLAG_SEEK) != 0;
        final boolean isInit = (flag & ACQUIRE_SESSION_FLAG_INIT) != 0;

        DrmSessionHolder targetHolder = drmSessionHolders[drmInitInfo.index];

        final DrmInitData drmInitData;
        if (uuid != null && psshData != null) {
            List<DrmInitData.SchemeData> schemeDataList = new ArrayList<>();
            schemeDataList.add(new DrmInitData.SchemeData(uuid, sampleMimeType, psshData));
            drmInitData = new DrmInitData(schemeType, schemeDataList);
        } else {
            drmInitData = null;
            Log.w(TAG, "acquireSession a null drmInitData");
        }

        DrmConstant.DrmSessionState result = isSeek ? targetHolder.seekDrmInitData(drmInitData) : targetHolder.feedDrmInitData(drmInitData);

        if (offlineLicensePssh != null && psshData != null && !Arrays.equals(offlineLicensePssh, psshData)) {
            Log.w(TAG, "drm key rotation and clear the old offline license key set");
            drmSessionManager.clearOfflineLicenseKeySetId();
            offlineLicensePssh = null;
        }

        if (isInit || isSeek) {
            //由于是音频视频同时触发，removeMessages会清除掉音频或视频的任务
//            workHandler.removeMessages(MSG_DO_SOME_WORK);
            workHandler.sendEmptyMessage(MSG_DO_SOME_WORK);
        }
        return result;
    }

    @Override
    public MediaCrypto getMediaCrypto(final int type) {
        if (drmSessionManager == null) {
            Log.e(TAG, "getMediaCrypto error: drm session manager is released");
            return null;
        }
        if (type != AUDIO_INDEX && type != VIDEO_INDEX) {
            return null;
        }
        final DrmSessionHolder targetHolder = drmSessionHolders[type];

        MediaCrypto[] result = new MediaCrypto[1];
        result[0] = targetHolder.getCurrentMediaCrypto();

        if (result[0] == null) {
            Log.w(TAG, "waiting for creating MediaCrypto");
            final Object lock = getCryptoLock;
            checkNotNull(workHandler).post(new Runnable() {
                @Override
                public void run() {
                    try {
                        result[0] = targetHolder.getCurrentMediaCrypto();
                    } finally {
                        synchronized (lock) {
                            lock.notifyAll();
                        }
                    }
                }
            });
            synchronized (lock) {
                try {
                    isCryptoLock = true;
                    lock.wait();
                    isCryptoLock = false;
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            Log.w(TAG, "MediaCrypto is created");
        }
        return result[0];
    }

    @Override
    public DrmConstant.DrmSessionState getDrmSessionState(int type, int flag) {
        if (drmSessionManager == null) {
            Log.e(TAG, "getDrmSessionState error: drm session manager is released");
            return DrmConstant.DrmSessionState.STATE_UNKNOWN;
        }
        if (type != AUDIO_INDEX && type != VIDEO_INDEX) {
            return DrmConstant.DrmSessionState.STATE_UNKNOWN;
        }

        final DrmSessionHolder targetHolder = drmSessionHolders[type];
        return targetHolder.getSeekDrmInitDataState();
    }

    @Override
    public void setOnDrmErrorListener(OnDrmErrorListener listener) {
        errorListener = listener;
    }

    public void setOfflineLicenseKeySetId(byte[] offlineLicenseKeySetId) {
        this.offlineLicenseKeySetId = offlineLicenseKeySetId;
    }

    public void setOfflineLicenseKeySetId(byte[] offlineLicenseKeySetId, String offlineLicenseDrmInitInfo) {
        this.offlineLicenseKeySetId = offlineLicenseKeySetId;
        this.offlineLicensePssh = null;
        if (offlineLicenseKeySetId != null && offlineLicenseDrmInitInfo != null) {
            List<DrmInitInfo> offlineDrmInitInfos = DrmInitInfoParser.parse(offlineLicenseDrmInitInfo);
            if (!offlineDrmInitInfos.isEmpty()) {
                DrmInitInfo drmInitInfo = offlineDrmInitInfos.get(0);
                offlineLicensePssh = drmInitInfo.psshData;
            }
        }
    }

    private void doSomeWork() throws Exception {
        final DrmSessionHolder audioHolder = drmSessionHolders[AUDIO_INDEX];
        final DrmSessionHolder videoHolder = drmSessionHolders[VIDEO_INDEX];
        boolean audioFirstSessionLoaded = true;
        boolean videoFirstSessionLoaded = true;
        if (audioHolder.isEnabled()) {
            audioHolder.doSomeWork();
            audioFirstSessionLoaded = audioHolder.isFinished();
        }
        if (videoHolder.isEnabled()) {
            videoHolder.doSomeWork();
            videoFirstSessionLoaded = videoHolder.isFinished();
        }
        if (!multiSession && audioFirstSessionLoaded && videoFirstSessionLoaded) {
            doSomeWorkLoopQuit = true;
            Log.i(TAG, "quit doSomeWork looper when first session loaded for none-multi session");
        }
        if (!doSomeWorkLoopQuit) {
            if (!workHandler.hasMessages(MSG_DO_SOME_WORK)) {
                workHandler.sendEmptyMessageDelayed(MSG_DO_SOME_WORK, 10);
            }
        }
    }

    private void notifyError(Exception e) {
        if (errorListener == null) {
            return;
        }
        if (e instanceof MediaDrmCallbackException) {
            errorListener.onDrmError(MEDIA_DRM_ERROR_PREFIX, e);
        } else if (e instanceof HttpDataSource.InvalidResponseCodeException) {
            errorListener.onDrmError(HTTP_DRM_ERROR_PREFIX + ((HttpDataSource.InvalidResponseCodeException) e).responseCode, e);
        } else {
            errorListener.onDrmError(UNKNOWN_DRM_ERROR_PREFIX, e);
        }
    }

    private static OkHttpDataSource.Factory createOkHttpDataSourceFactory(int level) {
        OkHttpClient.Builder builder = new OkHttpClient.Builder();
        HttpLoggingInterceptor loggingInterceptor = new HttpLoggingInterceptor();
        if (level == HTTP_LOG_LEVEL_BASIC) {
            loggingInterceptor.setLevel(HttpLoggingInterceptor.Level.BASIC);
        } else if (level == HTTP_LOG_LEVEL_HEADERS) {
            loggingInterceptor.setLevel(HttpLoggingInterceptor.Level.HEADERS);
        } else if (level == HTTP_LOG_LEVEL_BODY) {
            loggingInterceptor.setLevel(HttpLoggingInterceptor.Level.BODY);
        } else {
            loggingInterceptor.setLevel(HttpLoggingInterceptor.Level.NONE);
        }
        builder.addInterceptor(loggingInterceptor);
        OkHttpClient client = builder.build();
        return new OkHttpDataSource.Factory(new Call.Factory() {
            @NonNull
            @Override
            public Call newCall(@NonNull Request request) {
                return client.newCall(request);
            }
        });
    }

    private static DefaultDrmSessionManager createManager(
            String licenseUrl, Map<String, String> httpRequestHeaders, UUID uuid, boolean multiSession,
            byte[] offlineLicenseKeySetId, boolean useOkhttp, int level) {
        HttpDataSource.Factory dataSourceFactory;
        if (useOkhttp) {
            dataSourceFactory = createOkHttpDataSourceFactory(level).setUserAgent("ijk-drm-http");
        } else {
            dataSourceFactory = new DefaultHttpDataSource.Factory().setUserAgent("ijk-drm-http");
        }
        HttpMediaDrmCallback httpDrmCallback =
                new HttpMediaDrmCallback(licenseUrl, dataSourceFactory);
        for (Map.Entry<String, String> entry : httpRequestHeaders.entrySet()) {
            httpDrmCallback.setKeyRequestProperty(entry.getKey(), entry.getValue());
        }
        DefaultDrmSessionManager.Builder builder = new DefaultDrmSessionManager.Builder();
        if (multiSession) {
            builder.setSessionKeepaliveMs(C.TIME_UNSET);
        }
        DefaultDrmSessionManager drmSessionManager = builder
                .setUuidAndExoMediaDrmProvider(uuid, FrameworkMediaDrm.DEFAULT_PROVIDER)
                .setMultiSession(multiSession)
                .setUseDrmSessionsForClearContent(C.TRACK_TYPE_AUDIO, C.TRACK_TYPE_VIDEO)
                .build(httpDrmCallback);
        drmSessionManager.setMode(DefaultDrmSessionManager.MODE_PLAYBACK, offlineLicenseKeySetId);
        return drmSessionManager;
    }
}
