/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package tv.danmaku.ijk.media.drm;

import android.media.MediaDrm;
import android.os.ConditionVariable;
import android.os.Handler;
import android.os.HandlerThread;
import android.util.Log;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import okhttp3.Call;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.logging.HttpLoggingInterceptor;
import tv.danmaku.ijk.media.common.drm.DrmInitInfo;
import tv.danmaku.ijk.media.common.drm.DrmInitInfoParser;
import tv.danmaku.ijk.media.drm.DefaultDrmSessionManager.Mode;
import tv.danmaku.ijk.media.drm.DrmSession.DrmSessionException;
import tv.danmaku.ijk.media.drm.upstream.HttpDataSource;
import tv.danmaku.ijk.media.drm.upstream.OkHttpDataSource;
import tv.danmaku.ijk.media.drm.util.Assertions;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/** Helper class to download, renew and release offline licenses. */
@RequiresApi(18)
public final class OfflineLicenseHelper {

  private static final DrmInitData EMPTY_DRM_INIT_DATA = new DrmInitData();
  private static final String EMPTY_DRM_SAMPLE_MIME_TYPE = null;

  private final ConditionVariable conditionVariable;
  private final DefaultDrmSessionManager drmSessionManager;
  private final HandlerThread handlerThread;
  private final DrmSessionEventListener.EventDispatcher eventDispatcher;
  private DrmSessionException lastException;

  public static OkHttpDataSource.Factory createOkHttpDataSourceFactory() {
    OkHttpClient client = new OkHttpClient.Builder()
            .addInterceptor(new HttpLoggingInterceptor().setLevel(HttpLoggingInterceptor.Level.HEADERS))
            .build();
    return new OkHttpDataSource.Factory(new Call.Factory() {
      @NonNull
      @Override
      public Call newCall(@NonNull Request request) {
        return client.newCall(request);
      }
    });
  }

  /**
   * Instantiates a new instance which uses Widevine CDM. Call {@link #release()} when the instance
   * is no longer required.
   *
   * @param defaultLicenseUrl The default license URL. Used for key requests that do not specify
   *     their own license URL.
   * @param httpDataSourceFactory A factory from which to obtain {@link HttpDataSource} instances.
   * @param eventDispatcher A {@link DrmSessionEventListener.EventDispatcher} used to distribute
   *     DRM-related events.
   * @return A new instance which uses Widevine CDM.
   */
  public static OfflineLicenseHelper newWidevineInstance(
      String defaultLicenseUrl,
      HttpDataSource.Factory httpDataSourceFactory,
      DrmSessionEventListener.EventDispatcher eventDispatcher) {
    return newWidevineInstance(
        defaultLicenseUrl,
        /* forceDefaultLicenseUrl= */ false,
        httpDataSourceFactory,
        eventDispatcher);
  }

  /**
   * Instantiates a new instance which uses Widevine CDM. Call {@link #release()} when the instance
   * is no longer required.
   *
   * @param defaultLicenseUrl The default license URL. Used for key requests that do not specify
   *     their own license URL.
   * @param forceDefaultLicenseUrl Whether to use {@code defaultLicenseUrl} for key requests that
   *     include their own license URL.
   * @param httpDataSourceFactory A factory from which to obtain {@link HttpDataSource} instances.
   * @param eventDispatcher A {@link DrmSessionEventListener.EventDispatcher} used to distribute
   *     DRM-related events.
   * @return A new instance which uses Widevine CDM.
   */
  public static OfflineLicenseHelper newWidevineInstance(
      String defaultLicenseUrl,
      boolean forceDefaultLicenseUrl,
      HttpDataSource.Factory httpDataSourceFactory,
      DrmSessionEventListener.EventDispatcher eventDispatcher) {
    return newWidevineInstance(
        defaultLicenseUrl,
        forceDefaultLicenseUrl,
        httpDataSourceFactory,
        /* optionalKeyRequestParameters= */ null,
        eventDispatcher);
  }

  /**
   * Instantiates a new instance which uses Widevine CDM. Call {@link #release()} when the instance
   * is no longer required.
   *
   * @param defaultLicenseUrl The default license URL. Used for key requests that do not specify
   *     their own license URL.
   * @param forceDefaultLicenseUrl Whether to use {@code defaultLicenseUrl} for key requests that
   *     include their own license URL.
   * @param optionalKeyRequestParameters An optional map of parameters to pass as the last argument
   *     to {@link MediaDrm#getKeyRequest}. May be null.
   * @param eventDispatcher A {@link DrmSessionEventListener.EventDispatcher} used to distribute
   *     DRM-related events.
   * @return A new instance which uses Widevine CDM.
   * @see DefaultDrmSessionManager.Builder
   */
  public static OfflineLicenseHelper newWidevineInstance(
      String defaultLicenseUrl,
      boolean forceDefaultLicenseUrl,
      HttpDataSource.Factory httpDataSourceFactory,
      @Nullable Map<String, String> optionalKeyRequestParameters,
      DrmSessionEventListener.EventDispatcher eventDispatcher) {
    return new OfflineLicenseHelper(
        new DefaultDrmSessionManager.Builder()
            .setKeyRequestParameters(optionalKeyRequestParameters)
            .build(
                new HttpMediaDrmCallback(
                    defaultLicenseUrl, forceDefaultLicenseUrl, httpDataSourceFactory)),
        eventDispatcher);
  }

  public static OfflineLicenseHelper newWidevineInstance(
      MediaDrmCallback callback,
      @Nullable Map<String, String> optionalKeyRequestParameters,
      DrmSessionEventListener.EventDispatcher eventDispatcher) {
    return new OfflineLicenseHelper(
        new DefaultDrmSessionManager.Builder()
            .setKeyRequestParameters(optionalKeyRequestParameters)
            .build(callback),
        eventDispatcher);
  }

  /**
   * @deprecated Use {@link #OfflineLicenseHelper(DefaultDrmSessionManager,
   *     DrmSessionEventListener.EventDispatcher)} instead.
   */
  @Deprecated
  public OfflineLicenseHelper(
      UUID uuid,
      ExoMediaDrm.Provider mediaDrmProvider,
      MediaDrmCallback callback,
      @Nullable Map<String, String> optionalKeyRequestParameters,
      DrmSessionEventListener.EventDispatcher eventDispatcher) {
    this(
            new DefaultDrmSessionManager.Builder()
                .setUuidAndExoMediaDrmProvider(uuid, mediaDrmProvider)
                .setKeyRequestParameters(optionalKeyRequestParameters)
                .build(callback),
        eventDispatcher);
  }

  /**
   * Constructs an instance. Call {@link #release()} when the instance is no longer required.
   *
   * @param defaultDrmSessionManager The {@link DefaultDrmSessionManager} used to download licenses.
   * @param eventDispatcher A {@link DrmSessionEventListener.EventDispatcher} used to distribute
   *     DRM-related events.
   */
  public OfflineLicenseHelper(
      DefaultDrmSessionManager defaultDrmSessionManager,
      DrmSessionEventListener.EventDispatcher eventDispatcher) {
    Log.i("OfflineLicenseHelper", "new OfflineLicenseHelper " + this);
    this.drmSessionManager = defaultDrmSessionManager;
    this.eventDispatcher = eventDispatcher;
    handlerThread = new HandlerThread("ExoPlayer:OfflineLicenseHelper");
    handlerThread.start();
    conditionVariable = new ConditionVariable();
    DrmSessionEventListener eventListener =
        new DrmSessionEventListener() {
          @Override
          public void onDrmKeysLoaded() {
            conditionVariable.open();
            Log.i("OfflineLicenseHelper", "onDrmKeysLoaded");
          }

          @Override
          public void onDrmSessionManagerError(Exception e) {
            lastException = new DrmSessionException(e);
            conditionVariable.open();
            Log.i("OfflineLicenseHelper", "onDrmSessionManagerError : " + e);
          }

          @Override
          public void onDrmKeysRestored() {
            conditionVariable.open();
            Log.i("OfflineLicenseHelper", "onDrmKeysRestored");
          }

          @Override
          public void onDrmKeysRemoved() {
            conditionVariable.open();
            Log.i("OfflineLicenseHelper", "onDrmKeysRemoved");
          }
        };
    eventDispatcher.addEventListener(new Handler(handlerThread.getLooper()), eventListener);
  }

  public synchronized byte[] downloadLicense(String stringObj) throws DrmSessionException {
    List<DrmInitInfo> drmInitInfoList = DrmInitInfoParser.parse(stringObj);
    DrmInitInfo drmInitInfo = drmInitInfoList.get(0);
    List<DrmInitData.SchemeData> schemeDataList = new ArrayList<>();
    schemeDataList.add(new DrmInitData.SchemeData(drmInitInfo.uuid, drmInitInfo.sampleMimeType, drmInitInfo.psshData));
    DrmInitData drmInitData = new DrmInitData(drmInitInfo.schemeType, schemeDataList);
    return downloadLicense(drmInitData);
  }

  /**
   * Downloads an offline license.
   *
   * @param drmInitData The {@link DrmInitData} of the content whose license is to be downloaded. Must contain
   *     a non-null {@link DrmInitData}.
   * @return The key set id for the downloaded license.
   * @throws DrmSessionException Thrown when a DRM session error occurs.
   */
  public synchronized byte[] downloadLicense(DrmInitData drmInitData) throws DrmSessionException {
    Assertions.checkArgument(drmInitData != null);
    return blockingKeyRequest(DefaultDrmSessionManager.MODE_DOWNLOAD, null, drmInitData, EMPTY_DRM_SAMPLE_MIME_TYPE);
  }

  /**
   * Renews an offline license.
   *
   * @param offlineLicenseKeySetId The key set id of the license to be renewed.
   * @return The renewed offline license key set id.
   * @throws DrmSessionException Thrown when a DRM session error occurs.
   */
  public synchronized byte[] renewLicense(byte[] offlineLicenseKeySetId)
      throws DrmSessionException {
    Assertions.checkNotNull(offlineLicenseKeySetId);
    return blockingKeyRequest(
        DefaultDrmSessionManager.MODE_DOWNLOAD,
        offlineLicenseKeySetId,
        EMPTY_DRM_INIT_DATA,
        EMPTY_DRM_SAMPLE_MIME_TYPE);
  }

  /**
   * Releases an offline license.
   *
   * @param offlineLicenseKeySetId The key set id of the license to be released.
   * @throws DrmSessionException Thrown when a DRM session error occurs.
   */
  public synchronized void releaseLicense(byte[] offlineLicenseKeySetId)
      throws DrmSessionException {
    Assertions.checkNotNull(offlineLicenseKeySetId);
    blockingKeyRequest(
        DefaultDrmSessionManager.MODE_RELEASE,
        offlineLicenseKeySetId,
        EMPTY_DRM_INIT_DATA,
        EMPTY_DRM_SAMPLE_MIME_TYPE);
  }

  /**
   * Returns the remaining license and playback durations in seconds, for an offline license.
   *
   * @param offlineLicenseKeySetId The key set id of the license.
   * @return The remaining license and playback durations, in seconds.
   * @throws DrmSessionException Thrown when a DRM session error occurs.
   */
  public synchronized Pair<Long, Long> getLicenseDurationRemainingSec(byte[] offlineLicenseKeySetId)
      throws DrmSessionException {
    Assertions.checkNotNull(offlineLicenseKeySetId);
    drmSessionManager.prepare();
    DrmSession drmSession =
        openBlockingKeyRequest(
            DefaultDrmSessionManager.MODE_QUERY,
            offlineLicenseKeySetId,
            EMPTY_DRM_INIT_DATA,
            EMPTY_DRM_SAMPLE_MIME_TYPE);
    DrmSessionException error = drmSession.getError();
    Pair<Long, Long> licenseDurationRemainingSec =
        WidevineUtil.getLicenseDurationRemainingSec(drmSession);
    drmSession.release(eventDispatcher);
    drmSessionManager.release();
    // fix crash issue that state changed to STATE_ERROR after onDrmSessionManagerError callback,
    // so drmSession.getError() maybe return null
    if (error == null && (lastException != null)) {
      error = lastException;
    }
    lastException = null;
    if (error != null) {
      if (error.getCause() instanceof KeysExpiredException) {
        return Pair.create(0L, 0L);
      }
      throw error;
    }
    return Assertions.checkNotNull(licenseDurationRemainingSec);
  }

  /**
   * Releases the helper. Should be called when the helper is no longer required.
   */
  public void release() {
    handlerThread.quit();
    Log.i("OfflineLicenseHelper", "release OfflineLicenseHelper " + this);
  }

  private byte[] blockingKeyRequest(
      @Mode int licenseMode, @Nullable byte[] offlineLicenseKeySetId, DrmInitData drmInitData, String sampleMimeType)
      throws DrmSessionException {
    drmSessionManager.prepare();
    DrmSession drmSession = openBlockingKeyRequest(licenseMode, offlineLicenseKeySetId, drmInitData, sampleMimeType);
    DrmSessionException error = drmSession.getError();
    byte[] keySetId = drmSession.getOfflineLicenseKeySetId();
    drmSession.release(eventDispatcher);
    drmSessionManager.release();
    // fix crash issue that state changed to STATE_ERROR after onDrmSessionManagerError callback,
    // so drmSession.getError() maybe return null
    if (error == null && (lastException != null)) {
      error = lastException;
    }
    lastException = null;
    if (error != null) {
      throw error;
    }
    Log.i("OfflineLicenseHelper", "blockingKeyRequest result mode=" + licenseMode + ", drmInitData=" + drmInitData + ", setId=" + Arrays.toString(keySetId));
    return Assertions.checkNotNull(keySetId);
  }

  private DrmSession openBlockingKeyRequest(
      @Mode int licenseMode, @Nullable byte[] offlineLicenseKeySetId, DrmInitData drmInitData, String sampleMimeType) {
    Log.i("OfflineLicenseHelper", "openBlockingKeyRequest mode=" + licenseMode + ", drmInitData=" + drmInitData + ", setId=" + Arrays.toString(offlineLicenseKeySetId));
    Assertions.checkNotNull(drmInitData);
    drmSessionManager.setMode(licenseMode, offlineLicenseKeySetId);
    conditionVariable.close();
    DrmSession drmSession =
        drmSessionManager.acquireSession(handlerThread.getLooper(), eventDispatcher, drmInitData, sampleMimeType);
    // Block current thread until key loading is finished
    conditionVariable.block();
    Log.i("OfflineLicenseHelper", "openBlockingKeyRequest mode=" + licenseMode + " finished");
    return Assertions.checkNotNull(drmSession);
  }

}
