package tv.danmaku.ijk.media.drm.wrapper;

import android.media.MediaCrypto;
import android.media.MediaCryptoException;
import android.os.Build;
import android.os.Looper;
import android.util.Log;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import tv.danmaku.ijk.media.common.drm.DrmConstant;
import tv.danmaku.ijk.media.drm.DefaultDrmSessionManager;
import tv.danmaku.ijk.media.drm.DrmInitData;
import tv.danmaku.ijk.media.drm.DrmSession;
import tv.danmaku.ijk.media.drm.DrmSessionEventListener;
import tv.danmaku.ijk.media.drm.DrmSessionManager;
import tv.danmaku.ijk.media.drm.ExoMediaCrypto;
import tv.danmaku.ijk.media.drm.FrameworkMediaCrypto;

import static tv.danmaku.ijk.media.common.drm.DrmConstant.DrmSessionState.STATE_WAITING;
import static tv.danmaku.ijk.media.common.drm.DrmConstant.DrmSessionState.STATE_LOADING;
import static tv.danmaku.ijk.media.common.drm.DrmConstant.DrmSessionState.STATE_LOADED;
import static tv.danmaku.ijk.media.drm.util.Assertions.checkNotNull;

@RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
class DrmSessionHolder {
    private static class MyDrmInitData {
        private final DrmInitData drmInitData;
        private volatile boolean isActivated;

        private MyDrmInitData(DrmInitData drmInitData) {
            this.drmInitData = drmInitData;
        }

        private boolean isSameWith(DrmInitData other) {
            if (drmInitData == other) {
                return true;
            }
            return drmInitData != null && drmInitData.equals(other);
        }

        private boolean isSameWith(MyDrmInitData other) {
            if (other == null) {
                return false;
            }
            return this == other || isSameWith(other.drmInitData);
        }

        @NonNull
        @Override
        public String toString() {
            return "MyDrmInitData@" + Integer.toHexString(hashCode()) + "{" +
                    "drmInitData=" + drmInitData +
                    ", isActivated=" + isActivated +
                    '}';
        }
    }

    private static class MyDrmSession extends DrmSessionEventListener.EventDispatcher {
        private final DrmSessionHolder out;
        private final MyDrmInitData myDrmInitData;
        private final boolean isFirstSession;
        private DrmSession drmSession;
        private volatile boolean isLoaded = false;
        private Exception error;

        private MyDrmSession(DrmSessionHolder out, MyDrmInitData myDrmInitData) {
            this(out, myDrmInitData, false);
        }

        public MyDrmSession(DrmSessionHolder out, MyDrmInitData myDrmInitData, boolean isFirstSession) {
            this.out = out;
            this.myDrmInitData = myDrmInitData;
            this.isFirstSession = isFirstSession;
        }

        @Override
        public void drmSessionAcquired(boolean loaded) {
            if (loaded) {
                drmKeysLoaded();
            }
        }

        @Override
        public void drmKeysRestored() {
            drmKeysLoaded();
        }

        @Override
        public void drmKeysLoaded() {
            isLoaded = true;
            if (isFirstSession) {
                myDrmInitData.isActivated = true;
            }
            Log.i(out.TAG, "drmKeysLoaded " + myDrmInitData);
        }

        @Override
        public void drmSessionManagerError(Exception error) {
            this.error = error;
        }

        private void acquire(DrmSessionManager drmSessionManager,
                             Looper playbackLooper,
                             String sampleMimeType) {
            drmSession = checkNotNull(drmSessionManager)
                    .acquireSession(playbackLooper, this, myDrmInitData.drmInitData, sampleMimeType);
            if (myDrmInitData.drmInitData == null) {
                drmKeysLoaded();
            }
        }

        private void release() {
            drmSession.release(null);
        }

        public Exception getError() {
            return error;
        }

        public boolean isActivated() {
            return myDrmInitData.isActivated;
        }
    }

    private final String TAG;

    private final DefaultDrmSessionManager drmSessionManager;
    private final Looper playbackLooper;
    private final String sampleMimeType;
    private final boolean multiSession;
    private final Queue<MyDrmInitData> drmInitDataQueue = new ConcurrentLinkedQueue<>();
    private boolean isEnabled = false;
    private MyDrmSession currentDrmSession;
    private MyDrmSession nextDrmSession;
    private MediaCrypto currentMediaCrypto;
    private volatile MyDrmInitData seekDrmInitData;

    DrmSessionHolder(DefaultDrmSessionManager drmSessionManager, Looper looper, String sampleMimeType, boolean multiSession) {
        this.drmSessionManager = drmSessionManager;
        this.playbackLooper = looper;
        this.sampleMimeType = sampleMimeType;
        this.multiSession = multiSession;
        if (sampleMimeType.contains("video")) {
            TAG = "DrmSessionHolder-Video";
        } else if (sampleMimeType.contains("audio")) {
            TAG = "DrmSessionHolder-Audio";
        } else {
            TAG = "DrmSessionHolder";
        }
    }

    boolean isEnabled() {
        return isEnabled;
    }

    DrmConstant.DrmSessionState feedDrmInitData(DrmInitData drmInitData) {
        if (isFinished()) {
            return STATE_LOADED;
        }
        if (currentDrmSession != null && currentDrmSession.myDrmInitData.isSameWith(drmInitData)) {
            return currentDrmSession.isActivated() ? STATE_LOADED : STATE_LOADING;
        }
        if (nextDrmSession != null && nextDrmSession.myDrmInitData.isSameWith(drmInitData)) {
            return STATE_LOADING;
        }
        for (MyDrmInitData item : drmInitDataQueue) {
            if (item.isSameWith(drmInitData)) {
                return STATE_WAITING;
            }
        }
        MyDrmInitData myDrmInitData = new MyDrmInitData(drmInitData);
        drmInitDataQueue.add(myDrmInitData);
        isEnabled = true;
        Log.i(TAG, "feedDrmInitData new drmInitData " + myDrmInitData + " " + drmInitDataQueue.size());
        return STATE_WAITING;
    }

    DrmConstant.DrmSessionState seekDrmInitData(DrmInitData drmInitData) {
        if (isFinished()) {
            return STATE_LOADED;
        }
        if (currentDrmSession != null && currentDrmSession.myDrmInitData.isSameWith(drmInitData)) {
            if (seekDrmInitData != currentDrmSession.myDrmInitData) {
                seekDrmInitData = currentDrmSession.myDrmInitData;
                Log.i(TAG, "seekDrmInitData to currentDrmSession " + seekDrmInitData);
            }
            return currentDrmSession.isActivated() ? STATE_LOADED : STATE_LOADING;
        }
        if (nextDrmSession != null && nextDrmSession.myDrmInitData.isSameWith(drmInitData)) {
            if (seekDrmInitData != nextDrmSession.myDrmInitData) {
                seekDrmInitData = nextDrmSession.myDrmInitData;
                Log.i(TAG, "seekDrmInitData to nextDrmSession " + seekDrmInitData);
            }
            return STATE_LOADING;
        }
        boolean found = false;
        for (MyDrmInitData item : drmInitDataQueue) {
            if (item.isSameWith(drmInitData)) {
                found = true;
                break;
            }
        }
        if (!found) {
            drmInitDataQueue.clear();
            seekDrmInitData = new MyDrmInitData(drmInitData);
            drmInitDataQueue.add(seekDrmInitData);
            Log.i(TAG, "seekDrmInitData clear all drmInitData " + seekDrmInitData + " " + drmInitDataQueue.size());
            return STATE_WAITING;
        }
        while (!drmInitDataQueue.isEmpty()) {
            MyDrmInitData item = drmInitDataQueue.peek();
            assert item != null;
            if (item.isSameWith(drmInitData)) {
                seekDrmInitData = item;
                Log.i(TAG, "seekDrmInitData found target drmInitData " + seekDrmInitData + " " + drmInitDataQueue.size());
                break;
            } else {
                drmInitDataQueue.remove();
            }
        }
        return STATE_WAITING;
    }

    MediaCrypto getCurrentMediaCrypto() {
        return currentMediaCrypto;
    }

    DrmConstant.DrmSessionState getSeekDrmInitDataState() {
        if (isFinished()) {
            return STATE_LOADED;
        }
        if (seekDrmInitData == null) {
            return currentDrmSession != null && currentDrmSession.isActivated() ? STATE_LOADED : STATE_LOADING;
        }
        DrmConstant.DrmSessionState state = seekDrmInitData.isActivated ? STATE_LOADED : STATE_LOADING;

        if (state == STATE_LOADING && currentMediaCrypto != null && nextDrmSession != null && nextDrmSession.isLoaded) {
            boolean isMatched = nextDrmSession.myDrmInitData.isSameWith(seekDrmInitData);
            if (!isMatched) {
                throw new RuntimeException("the decode thread is blocked and will never exist successfully");
            }
        }
        return state;
    }

    void doSomeWork() throws Exception {
        checkError();
        if (canAcquireSession()) {
            acquireSession();
        }
        if (canCreateMediaCrypto()) {
            createMediaCrypto();
        }
        if (canAdvanceDrmSession()) {
            advanceDrmSession();
        }
    }

    boolean isFinished() {
        if (multiSession) {
            return false;
        }
        return currentDrmSession != null && currentDrmSession.isLoaded && currentDrmSession.myDrmInitData.drmInitData != null;
    }

    private void checkError() throws Exception {
        Exception exception = null;
        if (currentDrmSession != null && currentDrmSession.error != null) {
            exception = currentDrmSession.error;
        }
        if (nextDrmSession != null && nextDrmSession.error != null) {
            exception = nextDrmSession.error;
        }
        if (exception != null) {
            throw exception;
        }
    }

    private boolean canAcquireSession() {
        return !drmInitDataQueue.isEmpty();
    }

    private void acquireSession() {
        if (currentMediaCrypto == null && currentDrmSession == null) {
            MyDrmInitData myDrmInitData = drmInitDataQueue.remove();
            currentDrmSession = new MyDrmSession(this, myDrmInitData, true);
            Log.i(TAG, "acquireSession currentDrmSession " + currentDrmSession.myDrmInitData + " " + drmInitDataQueue.size());
            currentDrmSession.acquire(drmSessionManager, playbackLooper, sampleMimeType);
            return;
        }
        if (currentDrmSession != null && currentDrmSession.isLoaded && nextDrmSession == null) {
            MyDrmInitData myDrmInitData = drmInitDataQueue.remove();
            nextDrmSession = new MyDrmSession(this, myDrmInitData);
            Log.i(TAG, "acquireSession nextDrmSession " + nextDrmSession.myDrmInitData + " " + drmInitDataQueue.size());
            nextDrmSession.acquire(drmSessionManager, playbackLooper, sampleMimeType);
        }
    }

    private boolean canCreateMediaCrypto() {
        return currentMediaCrypto == null && currentDrmSession != null;
    }

    private void createMediaCrypto() throws MediaCryptoException {
        if (currentMediaCrypto != null) {
            return;
        }
        if (currentDrmSession == null) {
            throw new IllegalStateException("please create currentDrmSession first");
        }
        DrmSession session = currentDrmSession.drmSession;
        ExoMediaCrypto cryptoConfig;
        if ((cryptoConfig = session.getMediaCrypto()) instanceof FrameworkMediaCrypto) {
            FrameworkMediaCrypto frameworkMediaCrypto = (FrameworkMediaCrypto) cryptoConfig;
            currentMediaCrypto = new MediaCrypto(frameworkMediaCrypto.uuid, frameworkMediaCrypto.sessionId);
            Log.i(TAG, "createMediaCrypto currentMediaCrypto " + currentMediaCrypto);
        } else {
            throw new IllegalStateException("createMediaCrypto:it is not a FrameworkMediaCrypto instance " + cryptoConfig);
        }
    }

    private boolean canAdvanceDrmSession() {
        return currentMediaCrypto != null && nextDrmSession != null && nextDrmSession.isLoaded &&
                (nextDrmSession.myDrmInitData.isSameWith(seekDrmInitData) || !multiSession);
    }

    private void advanceDrmSession() throws MediaCryptoException {
        if (currentMediaCrypto == null) {
            throw new IllegalStateException("please create currentMediaCrypto first");
        }
        if (nextDrmSession == null || !nextDrmSession.isLoaded) {
            throw new IllegalStateException("please load nextDrmSession first");
        }
        if (!nextDrmSession.myDrmInitData.isSameWith(seekDrmInitData) && multiSession) {
            throw new IllegalStateException("the nextDrmSession is not match seekDrmInitData");
        }
        if (currentDrmSession.drmSession == nextDrmSession.drmSession) {
            return;
        }
        DrmSession session = nextDrmSession.drmSession;
        ExoMediaCrypto exoMediaCrypto = checkNotNull(session).getMediaCrypto();
        FrameworkMediaCrypto frameworkMediaCrypto;
        if (exoMediaCrypto instanceof FrameworkMediaCrypto) {
            frameworkMediaCrypto = (FrameworkMediaCrypto) checkNotNull(exoMediaCrypto);
        } else {
            throw new IllegalStateException("advanceDrmSession:it is not a FrameworkMediaCrypto instance");
        }
        byte[] sessionId = frameworkMediaCrypto.sessionId;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            currentMediaCrypto.setMediaDrmSession(sessionId);
            currentDrmSession.release();
            currentDrmSession = nextDrmSession;
            nextDrmSession = null;
            currentDrmSession.myDrmInitData.isActivated = true;
            Log.i(TAG, "advanceDrmSession currentDrmSession " + currentDrmSession.myDrmInitData);
        }
    }
}
