package tv.danmaku.ijk.media.drm.wrapper;

import android.media.MediaCrypto;
import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.util.Log;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import tv.danmaku.ijk.media.common.drm.DrmConstant;
import tv.danmaku.ijk.media.common.drm.DrmInitInfo;
import tv.danmaku.ijk.media.common.drm.DrmManager;
import tv.danmaku.ijk.media.common.drm.OnDrmErrorListener;
import tv.danmaku.ijk.media.drm.DefaultDrmSessionManager;
import tv.danmaku.ijk.media.drm.DrmInitData;
import tv.danmaku.ijk.media.drm.FrameworkMediaDrm;
import tv.danmaku.ijk.media.drm.HttpMediaDrmCallback;
import tv.danmaku.ijk.media.drm.MediaDrmCallbackException;
import tv.danmaku.ijk.media.drm.upstream.DefaultHttpDataSource;
import tv.danmaku.ijk.media.drm.upstream.HttpDataSource;
import tv.danmaku.ijk.media.drm.util.C;

import static tv.danmaku.ijk.media.common.drm.DrmConstant.ACQUIRE_SESSION_FLAG_INIT;
import static tv.danmaku.ijk.media.common.drm.DrmConstant.ACQUIRE_SESSION_FLAG_SEEK;
import static tv.danmaku.ijk.media.common.drm.DrmConstant.AUDIO_INDEX;
import static tv.danmaku.ijk.media.common.drm.DrmConstant.VIDEO_INDEX;
import static tv.danmaku.ijk.media.drm.util.Assertions.checkNotNull;

@RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
public class DefaultDrmManager implements DrmManager {
    private static final String TAG = "DefaultDrmManager";

    private static final int MSG_DO_SOME_WORK = 101;

    private static final int MEDIA_DRM_ERROR_PREFIX = 10000;
    private static final int HTTP_DRM_ERROR_PREFIX = 20000;
    private static final int UNKNOWN_DRM_ERROR_PREFIX = 90000;

    private final String drmLicenseUrl;
    private final Map<String, String> httpRequestHeaders;
    private final UUID uuid;
    private final boolean multiSession;
    private OnDrmErrorListener errorListener;

    private HandlerThread handlerThread = null;
    private Handler workHandler = null;
    private DefaultDrmSessionManager drmSessionManager = null;

    private final DrmSessionHolder[] drmSessionHolders = new DrmSessionHolder[2];

    private Exception doSomeWorkError = null;
    private boolean doSomeWorkLoopQuit = false;

    public DefaultDrmManager(String drmLicenceUrl, Map<String, String> httpRequestHeaders, UUID uuid, boolean multiSession) {
        this.drmLicenseUrl = drmLicenceUrl;
        this.httpRequestHeaders = new HashMap<>();
        if (httpRequestHeaders != null) {
            this.httpRequestHeaders.putAll(httpRequestHeaders);
        }
        this.uuid = uuid;
        this.multiSession = multiSession;
    }

    @Override
    public synchronized void prepare() {
        if (handlerThread == null) {
            handlerThread = new HandlerThread(DefaultDrmManager.class.getSimpleName());
            handlerThread.start();
            workHandler = new Handler(handlerThread.getLooper()) {
                @Override
                public void handleMessage(@NonNull Message msg) {
                    if (msg.what == MSG_DO_SOME_WORK) {
                        if (doSomeWorkError != null) {
                            doSomeWorkError.printStackTrace();
                            return;
                        }
                        try {
                            doSomeWork();
                        } catch (Exception e) {
                            Log.e(TAG, "doSomeWork error : " + e);
                            doSomeWorkError = e;
                            doSomeWorkError.printStackTrace();
                            doSomeWorkLoopQuit = true;
                            workHandler.removeMessages(MSG_DO_SOME_WORK);
                            notifyError(e);
                        }
                    }
                }
            };
            boolean needPlaceholderSession = false;
            if (drmSessionManager == null) {
                drmSessionManager = createManager(drmLicenseUrl, httpRequestHeaders, uuid, multiSession);
                drmSessionHolders[AUDIO_INDEX] = new DrmSessionHolder(drmSessionManager, workHandler.getLooper(), "audio/mp4", multiSession);
                drmSessionHolders[VIDEO_INDEX] = new DrmSessionHolder(drmSessionManager, workHandler.getLooper(), "video/mp4", multiSession);
                // {@link MediaCrypto#setMediaDrmSession(byte[])} min api level is 24
                needPlaceholderSession = Build.VERSION.SDK_INT > Build.VERSION_CODES.M;
            }
            workHandler.post(new Runnable() {
                @Override
                public void run() {
                    drmSessionManager.prepare();
                    Log.i(TAG, "prepare drm session manager");
                }
            });
            if (needPlaceholderSession) {
                drmSessionHolders[AUDIO_INDEX].feedDrmInitData(null);
                drmSessionHolders[VIDEO_INDEX].feedDrmInitData(null);
                workHandler.sendEmptyMessage(MSG_DO_SOME_WORK);
            }
        }
    }

    @Override
    public synchronized void release() {
        if (drmSessionManager != null) {
            checkNotNull(workHandler).post(new Runnable() {
                @Override
                public void run() {
                    drmSessionManager.release();
                    drmSessionManager = null;
                    doSomeWorkLoopQuit = true;
                    handlerThread.quitSafely();
                    Log.i(TAG, "release drm session manager");
                }
            });
        }
    }

    @Override
    public synchronized DrmConstant.DrmSessionState acquireSession(final DrmInitInfo drmInitInfo, final int flag) {
        @NonNull final String schemeType = drmInitInfo.schemeType;
        final UUID uuid = drmInitInfo.uuid;
        @Nullable final String sampleMimeType = drmInitInfo.sampleMimeType;
        final byte[] psshData = drmInitInfo.psshData;

        final boolean isSeek = (flag & ACQUIRE_SESSION_FLAG_SEEK) != 0;
        final boolean isInit = (flag & ACQUIRE_SESSION_FLAG_INIT) != 0;

        DrmSessionHolder targetHolder = drmSessionHolders[drmInitInfo.index];

        final DrmInitData drmInitData;
        if (uuid != null && psshData != null) {
            List<DrmInitData.SchemeData> schemeDataList = new ArrayList<>();
            schemeDataList.add(new DrmInitData.SchemeData(uuid, sampleMimeType, psshData));
            drmInitData = new DrmInitData(schemeType, schemeDataList);
        } else {
            drmInitData = null;
            Log.w(TAG, "acquireSession a null drmInitData");
        }

        DrmConstant.DrmSessionState result = isSeek ? targetHolder.seekDrmInitData(drmInitData) : targetHolder.feedDrmInitData(drmInitData);

        if (isInit || isSeek) {
            workHandler.removeMessages(MSG_DO_SOME_WORK);
            workHandler.sendEmptyMessage(MSG_DO_SOME_WORK);
        }
        return result;
    }

    @Override
    public MediaCrypto getMediaCrypto(final int type) {
        if (type != AUDIO_INDEX && type != VIDEO_INDEX) {
            return null;
        }
        final DrmSessionHolder targetHolder = drmSessionHolders[type];

        MediaCrypto[] result = new MediaCrypto[1];
        result[0] = targetHolder.getCurrentMediaCrypto();

        if (result[0] == null) {
            Log.w(TAG, "waiting for creating MediaCrypto");
            final Object lock = new Object();
            checkNotNull(workHandler).post(new Runnable() {
                @Override
                public void run() {
                    try {
                        result[0] = targetHolder.getCurrentMediaCrypto();
                    } finally {
                        synchronized (lock) {
                            lock.notifyAll();
                        }
                    }
                }
            });
            synchronized (lock) {
                try {
                    lock.wait();
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            Log.w(TAG, "MediaCrypto is created");
        }
        return result[0];
    }

    @Override
    public DrmConstant.DrmSessionState getDrmSessionState(int type, int flag) {
        if (type != AUDIO_INDEX && type != VIDEO_INDEX) {
            return DrmConstant.DrmSessionState.STATE_UNKNOWN;
        }

        final DrmSessionHolder targetHolder = drmSessionHolders[type];
        return targetHolder.getSeekDrmInitDataState();
    }

    @Override
    public void setOnDrmErrorListener(OnDrmErrorListener listener) {
        errorListener = listener;
    }

    private void doSomeWork() throws Exception {
        final DrmSessionHolder audioHolder = drmSessionHolders[AUDIO_INDEX];
        final DrmSessionHolder videoHolder = drmSessionHolders[VIDEO_INDEX];
        boolean audioFirstSessionLoaded = true;
        boolean videoFirstSessionLoaded = true;
        if (audioHolder.isEnabled()) {
            audioHolder.doSomeWork();
            audioFirstSessionLoaded = audioHolder.isFinished();
        }
        if (videoHolder.isEnabled()) {
            videoHolder.doSomeWork();
            videoFirstSessionLoaded = videoHolder.isFinished();
        }
        if (!multiSession && audioFirstSessionLoaded && videoFirstSessionLoaded) {
            doSomeWorkLoopQuit = true;
            Log.i(TAG, "quit doSomeWork looper when first session loaded for none-multi session");
        }
        if (!doSomeWorkLoopQuit) {
            workHandler.removeMessages(MSG_DO_SOME_WORK);
            workHandler.sendEmptyMessageDelayed(MSG_DO_SOME_WORK, 10);
        }
    }

    private void notifyError(Exception e) {
        if (errorListener == null) {
            return;
        }
        if (e instanceof MediaDrmCallbackException) {
            errorListener.onDrmError(MEDIA_DRM_ERROR_PREFIX, e);
        } else if (e instanceof HttpDataSource.InvalidResponseCodeException) {
            errorListener.onDrmError(HTTP_DRM_ERROR_PREFIX + ((HttpDataSource.InvalidResponseCodeException) e).responseCode, e);
        } else {
            errorListener.onDrmError(UNKNOWN_DRM_ERROR_PREFIX, e);
        }
    }

    private static DefaultDrmSessionManager createManager(
            String licenseUrl, Map<String, String> httpRequestHeaders, UUID uuid, boolean multiSession) {
        HttpDataSource.Factory dataSourceFactory =
                new DefaultHttpDataSource.Factory().setUserAgent("ijk-drm-http");
        HttpMediaDrmCallback httpDrmCallback =
                new HttpMediaDrmCallback(licenseUrl, dataSourceFactory);
        for (Map.Entry<String, String> entry : httpRequestHeaders.entrySet()) {
            httpDrmCallback.setKeyRequestProperty(entry.getKey(), entry.getValue());
        }
        DefaultDrmSessionManager.Builder builder = new DefaultDrmSessionManager.Builder();
        if (multiSession) {
            builder.setSessionKeepaliveMs(C.TIME_UNSET);
        }
        DefaultDrmSessionManager drmSessionManager = builder
                .setUuidAndExoMediaDrmProvider(uuid, FrameworkMediaDrm.DEFAULT_PROVIDER)
                .setMultiSession(multiSession)
                .setUseDrmSessionsForClearContent(C.TRACK_TYPE_AUDIO, C.TRACK_TYPE_VIDEO)
                .build(httpDrmCallback);
        drmSessionManager.setMode(DefaultDrmSessionManager.MODE_PLAYBACK, null);
        return drmSessionManager;
    }
}
