/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package tv.danmaku.ijk.media.drm.util;

import org.checkerframework.checker.nullness.qual.EnsuresNonNull;
import org.checkerframework.dataflow.qual.Pure;

import androidx.annotation.Nullable;

/**
 * Provides methods for asserting the truth of expressions and properties.
 */
public final class Assertions {

  private static final boolean ASSERTIONS_ENABLED = true;

  private Assertions() {}

  /**
   * Throws {@link IllegalArgumentException} if {@code expression} evaluates to false.
   *
   * @param expression The expression to evaluate.
   * @throws IllegalArgumentException If {@code expression} is false.
   */
  @Pure
  public static void checkArgument(boolean expression) {
    if (ASSERTIONS_ENABLED && !expression) {
      throw new IllegalArgumentException();
    }
  }

  /**
   * Throws {@link IllegalArgumentException} if {@code expression} evaluates to false.
   *
   * @param expression The expression to evaluate.
   * @param errorMessage The exception message if an exception is thrown. The message is converted
   *     to a {@link String} using {@link String#valueOf(Object)}.
   * @throws IllegalArgumentException If {@code expression} is false.
   */
  @Pure
  public static void checkArgument(boolean expression, Object errorMessage) {
    if (ASSERTIONS_ENABLED && !expression) {
      throw new IllegalArgumentException(String.valueOf(errorMessage));
    }
  }

  /**
   * Throws {@link IllegalStateException} if {@code expression} evaluates to false.
   *
   * @param expression The expression to evaluate.
   * @throws IllegalStateException If {@code expression} is false.
   */
  @Pure
  public static void checkState(boolean expression) {
    if (ASSERTIONS_ENABLED && !expression) {
      throw new IllegalStateException();
    }
  }

  /**
   * Throws {@link IllegalStateException} if {@code expression} evaluates to false.
   *
   * @param expression The expression to evaluate.
   * @param errorMessage The exception message if an exception is thrown. The message is converted
   *     to a {@link String} using {@link String#valueOf(Object)}.
   * @throws IllegalStateException If {@code expression} is false.
   */
  @Pure
  public static void checkState(boolean expression, Object errorMessage) {
    if (ASSERTIONS_ENABLED && !expression) {
      throw new IllegalStateException(String.valueOf(errorMessage));
    }
  }

  /**
   * Throws {@link IllegalStateException} if {@code reference} is null.
   *
   * @param <T> The type of the reference.
   * @param reference The reference.
   * @param errorMessage The exception message to use if the check fails. The message is converted
   *     to a string using {@link String#valueOf(Object)}.
   * @return The non-null reference that was validated.
   * @throws IllegalStateException If {@code reference} is null.
   */
  @SuppressWarnings({"contracts.postcondition.not.satisfied", "return.type.incompatible"})
  @EnsuresNonNull({"#1"})
  @Pure
  public static <T> T checkStateNotNull(@Nullable T reference, Object errorMessage) {
    if (ASSERTIONS_ENABLED && reference == null) {
      throw new IllegalStateException(String.valueOf(errorMessage));
    }
    return reference;
  }

  /**
   * Throws {@link NullPointerException} if {@code reference} is null.
   *
   * @param <T> The type of the reference.
   * @param reference The reference.
   * @return The non-null reference that was validated.
   * @throws NullPointerException If {@code reference} is null.
   */
  @SuppressWarnings({"contracts.postcondition.not.satisfied", "return.type.incompatible"})
  @EnsuresNonNull({"#1"})
  @Pure
  public static <T> T checkNotNull(@Nullable T reference) {
    if (ASSERTIONS_ENABLED && reference == null) {
      throw new NullPointerException();
    }
    return reference;
  }

}
