/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package tv.danmaku.ijk.media.drm;

import android.os.Handler;

import tv.danmaku.ijk.media.drm.util.Assertions;

import java.util.concurrent.CopyOnWriteArrayList;

import static tv.danmaku.ijk.media.drm.util.Util.postOrRun;

/** Listener of {@link DrmSessionManager} events. */
public interface DrmSessionEventListener {

  /**
   * Called each time a drm session is acquired.
   */
  default void onDrmSessionAcquired(boolean loaded) {}

  /**
   * Called each time keys are loaded.
   */
  default void onDrmKeysLoaded() {}

  /**
   * Called when a drm error occurs.
   *
   * <p>This method being called does not indicate that playback has failed, or that it will fail.
   * The player may be able to recover from the error and continue. Hence applications should
   * <em>not</em> implement this method to display a user visible error or initiate an application
   * level retry (Player.EventListener#onPlayerError is the appropriate place to implement
   * such behavior). This method is called to provide the application with an opportunity to log the
   * error if it wishes to do so.
   *
   * @param error The corresponding exception.
   */
  default void onDrmSessionManagerError(Exception error) {}

  /**
   * Called each time offline keys are restored.
   */
  default void onDrmKeysRestored() {}

  /**
   * Called each time offline keys are removed.
   */
  default void onDrmKeysRemoved() {}

  /**
   * Called each time a drm session is released.
   */
  default void onDrmSessionReleased() {}

  /** Dispatches events to {@link DrmSessionEventListener DrmSessionEventListeners}. */
  class EventDispatcher {

    private final CopyOnWriteArrayList<EventDispatcher.ListenerAndHandler> listenerAndHandlers;

    /** Creates an event dispatcher. */
    public EventDispatcher() {
      this(
          /* listenerAndHandlers= */ new CopyOnWriteArrayList<>());
    }

    private EventDispatcher(
        CopyOnWriteArrayList<EventDispatcher.ListenerAndHandler> listenerAndHandlers) {
      this.listenerAndHandlers = listenerAndHandlers;
    }

    /**
     * Adds a listener to the event dispatcher.
     *
     * @param handler A handler on the which listener events will be posted.
     * @param eventListener The listener to be added.
     */
    public void addEventListener(Handler handler, DrmSessionEventListener eventListener) {
      Assertions.checkNotNull(handler);
      Assertions.checkNotNull(eventListener);
      listenerAndHandlers.add(new ListenerAndHandler(handler, eventListener));
    }

    /**
     * Removes a listener from the event dispatcher.
     *
     * @param eventListener The listener to be removed.
     */
    public void removeEventListener(DrmSessionEventListener eventListener) {
      for (ListenerAndHandler listenerAndHandler : listenerAndHandlers) {
        if (listenerAndHandler.listener == eventListener) {
          listenerAndHandlers.remove(listenerAndHandler);
        }
      }
    }

    /** Dispatches {@link #onDrmSessionAcquired(boolean)}. */
    public void drmSessionAcquired(boolean loaded) {
      for (ListenerAndHandler listenerAndHandler : listenerAndHandlers) {
        DrmSessionEventListener listener = listenerAndHandler.listener;
        postOrRun(
            listenerAndHandler.handler,
                () -> listener.onDrmSessionAcquired(loaded));
      }
    }

    /** Dispatches {@link #onDrmKeysLoaded()}. */
    public void drmKeysLoaded() {
      for (ListenerAndHandler listenerAndHandler : listenerAndHandlers) {
        DrmSessionEventListener listener = listenerAndHandler.listener;
        postOrRun(
                listenerAndHandler.handler, listener::onDrmKeysLoaded);
      }
    }

    /** Dispatches {@link #onDrmSessionManagerError(Exception)}. */
    public void drmSessionManagerError(Exception error) {
      for (ListenerAndHandler listenerAndHandler : listenerAndHandlers) {
        DrmSessionEventListener listener = listenerAndHandler.listener;
        postOrRun(
            listenerAndHandler.handler,
            () -> listener.onDrmSessionManagerError(error));
      }
    }

    /** Dispatches {@link #onDrmKeysRestored()}. */
    public void drmKeysRestored() {
      for (ListenerAndHandler listenerAndHandler : listenerAndHandlers) {
        DrmSessionEventListener listener = listenerAndHandler.listener;
        postOrRun(
            listenerAndHandler.handler,
                listener::onDrmKeysRestored);
      }
    }

    /** Dispatches {@link #onDrmKeysRemoved()}. */
    public void drmKeysRemoved() {
      for (ListenerAndHandler listenerAndHandler : listenerAndHandlers) {
        DrmSessionEventListener listener = listenerAndHandler.listener;
        postOrRun(
            listenerAndHandler.handler,
                listener::onDrmKeysRemoved);
      }
    }

    /** Dispatches {@link #onDrmSessionReleased()}. */
    public void drmSessionReleased() {
      for (ListenerAndHandler listenerAndHandler : listenerAndHandlers) {
        DrmSessionEventListener listener = listenerAndHandler.listener;
        postOrRun(
            listenerAndHandler.handler,
                listener::onDrmSessionReleased);
      }
    }

    private static final class ListenerAndHandler {

      public Handler handler;
      public DrmSessionEventListener listener;

      public ListenerAndHandler(Handler handler, DrmSessionEventListener listener) {
        this.handler = handler;
        this.listener = listener;
      }
    }
  }
}
